﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Xml;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeDesc
{
    public class FsAccessFlagPreset
    {
        public ulong Value { get; set; }
        public string Name { get; set; }
        public FsAccessFlagPreset(ulong value, string name)
        {
            Value = value;
            Name = name;
        }
    }

    public class FsAccessFlagPresetDefinition
    {
        public static List<FsAccessFlagPreset> Set = new List<FsAccessFlagPreset>
            {
                new FsAccessFlagPreset((ulong) 1 <<  0, "ApplicationInfo"),
                new FsAccessFlagPreset((ulong) 1 <<  1, "BootModeControl"),
                new FsAccessFlagPreset((ulong) 1 <<  2, "Calibration"),
                new FsAccessFlagPreset((ulong) 1 <<  3, "SystemSaveData"),
                new FsAccessFlagPreset((ulong) 1 <<  4, "GameCard"),
                new FsAccessFlagPreset((ulong) 1 <<  5, "SaveDataBackUp"),
                new FsAccessFlagPreset((ulong) 1 <<  6, "SaveDataManagement"),
                new FsAccessFlagPreset((ulong) 1 <<  7, "BisAllRaw"),
                new FsAccessFlagPreset((ulong) 1 <<  8, "GameCardRaw"),
                new FsAccessFlagPreset((ulong) 1 <<  9, "GameCardPrivate"),
                new FsAccessFlagPreset((ulong) 1 << 10, "SetTime"),
                new FsAccessFlagPreset((ulong) 1 << 11, "ContentManager"),
                new FsAccessFlagPreset((ulong) 1 << 12, "ImageManager"),
                new FsAccessFlagPreset((ulong) 1 << 13, "CreateSaveData"),
                new FsAccessFlagPreset((ulong) 1 << 14, "SystemSaveDataManagement"),
                new FsAccessFlagPreset((ulong) 1 << 15, "BisFileSystem"),
                new FsAccessFlagPreset((ulong) 1 << 16, "SystemUpdate"),
                new FsAccessFlagPreset((ulong) 1 << 17, "SaveDataMeta"),
                new FsAccessFlagPreset((ulong) 1 << 18, "DeviceSaveData"),
                new FsAccessFlagPreset((ulong) 1 << 19, "SettingsControl"),
                new FsAccessFlagPreset((ulong) 1 << 20, "SystemData"),
                new FsAccessFlagPreset((ulong) 1 << 21, "SdCard"),
                new FsAccessFlagPreset((ulong) 1 << 22, "Host"),
                new FsAccessFlagPreset((ulong) 1 << 23, "FillBis"),
                new FsAccessFlagPreset((ulong) 1 << 24, "CorruptSaveData"),
                new FsAccessFlagPreset((ulong) 1 << 25, "SaveDataForDebug"),
                new FsAccessFlagPreset((ulong) 1 << 26, "FormatSdCard"),
                new FsAccessFlagPreset((ulong) 1 << 27, "GetRightsId"),
                new FsAccessFlagPreset((ulong) 1 << 28, "RegisterExternalKey"),
                new FsAccessFlagPreset((ulong) 1 << 29, "RegisterUpdatePartition"),
                new FsAccessFlagPreset((ulong) 1 << 30, "SaveDataTransfer"),
                new FsAccessFlagPreset((ulong) 1 << 31, "DeviceDetection"),
                new FsAccessFlagPreset((ulong) 1 << 32, "AccessFailureResolution"),
                new FsAccessFlagPreset((ulong) 1 << 33, "SaveDataTransferVersion2"),
                new FsAccessFlagPreset((ulong) 1 << 62, "Debug"),
                new FsAccessFlagPreset((ulong) 1 << 63, "FullPermission"),
            };
    }

    /// <summary>
    /// Fs アクセス制御データの共通部分を管理するクラス
    /// </summary>
    public class FsAccessControl
    {
        public bool IsImported { get; protected set; }

        protected ulong flagPresetsBit;

        public FsAccessControl()
        {
            flagPresetsBit = 0;
        }
    }

    public class Utility
    {
        public static int RoundUp(int target, uint align)
        {
            return (int)((target + (align - 1)) & ~(align - 1));
        }
    }

    /// <summary>
    /// Fs アクセス制御ディスクリプタを管理するクラス
    /// </summary>
    public class FacDescriptor : FsAccessControl
    {
        internal byte version;
        internal ulong contentOwnerIdMin;
        internal ulong contentOwnerIdMax;
        internal ulong saveDataOwnerIdMin;
        internal ulong saveDataOwnerIdMax;
        internal List<ulong> contentOwnerIds;
        internal List<ulong> saveDataOwnerIds;

        private int DescriptorSize = 44;
        private bool isImported;

        public FacDescriptor()
        {
            version = 1;
            contentOwnerIdMin = 0;
            contentOwnerIdMax = 0;
            contentOwnerIds = new List<ulong>();
            saveDataOwnerIdMin = 0;
            saveDataOwnerIdMax = 0;
            saveDataOwnerIds = new List<ulong>();
            isImported = false;
        }

        public void ImportDescFile(DescModel.InputDescModel model)
        {
            if (model.FsAccessControlDescriptor == null)
            {
                return;
            }

            ulong bit = 0;
            foreach (var flagPreset in model.FsAccessControlDescriptor.FlagPresets)
            {
                bit |= FsAccessFlagPresetDefinition.Set.Where(x => x.Name == flagPreset).Select(x => x.Value).FirstOrDefault();
            }

            flagPresetsBit = bit;
            contentOwnerIdMin = model.FsAccessControlDescriptor.ContentOwnerIdMinValue;
            contentOwnerIdMax = model.FsAccessControlDescriptor.ContentOwnerIdMaxValue;
            foreach (var ownerId in model.FsAccessControlDescriptor.ContentOwnerIds)
            {
                contentOwnerIds.Add(DescModel.ConvertUtils.ConvertHexString(ownerId, "FsAccessControlDescriptor/ContentOwnerIds"));
            }
            saveDataOwnerIdMin = model.FsAccessControlDescriptor.SaveDataOwnerIdMinValue;
            saveDataOwnerIdMax = model.FsAccessControlDescriptor.SaveDataOwnerIdMaxValue;
            foreach (var ownerId in model.FsAccessControlDescriptor.SaveDataOwnerIds)
            {
                saveDataOwnerIds.Add(DescModel.ConvertUtils.ConvertHexString(ownerId, "FsAccessControlDescriptor/SaveDataOwnerIds"));
            }

            DescriptorSize += 8 * contentOwnerIds.Count + 8 * saveDataOwnerIds.Count;
            isImported = true;
        }

        public void ExportDescFile(ref DescModel.OutputDescModel model)
        {
            if (!isImported)
            {
                return;
            }

            model.FsAccessControlDescriptor = new DescModel.FaDescriptorModel();
            model.FsAccessControlDescriptor.FlagPresets = new List<string>();
            foreach (var name in FsAccessFlagPresetDefinition.Set.Where(x => (x.Value & flagPresetsBit) != 0).Select(x => x.Name))
            {
                model.FsAccessControlDescriptor.FlagPresets.Add(name);
            }
            model.FsAccessControlDescriptor.ContentOwnerIdMin = contentOwnerIdMin == 0 ? null : DescModel.ConvertUtils.ConvertToHexString(contentOwnerIdMin);
            model.FsAccessControlDescriptor.ContentOwnerIdMax = contentOwnerIdMax == 0 ? null : DescModel.ConvertUtils.ConvertToHexString(contentOwnerIdMax);
            model.FsAccessControlDescriptor.ContentOwnerIds = new List<string>();
            foreach (var ownerId in contentOwnerIds)
            {
                model.FsAccessControlDescriptor.ContentOwnerIds.Add(DescModel.ConvertUtils.ConvertToHexString(ownerId));
            }
            model.FsAccessControlDescriptor.SaveDataOwnerIdMin = saveDataOwnerIdMin == 0 ? null : DescModel.ConvertUtils.ConvertToHexString(saveDataOwnerIdMin);
            model.FsAccessControlDescriptor.SaveDataOwnerIdMax = saveDataOwnerIdMin == 0 ? null : DescModel.ConvertUtils.ConvertToHexString(saveDataOwnerIdMax);
            model.FsAccessControlDescriptor.SaveDataOwnerIds = new List<string>();
            foreach (var ownerId in saveDataOwnerIds)
            {
                model.FsAccessControlDescriptor.SaveDataOwnerIds.Add(DescModel.ConvertUtils.ConvertToHexString(ownerId));
            }
        }

        public byte[] ExportBinary()
        {
            byte[] descriptor = new byte[DescriptorSize];
            Buffer.BlockCopy(BitConverter.GetBytes(version), 0, descriptor, 0, 1);
            if (contentOwnerIds.Count > 255)
            {
                throw new ArgumentException(string.Format("Too many SaveDataOwnerIds are specified. ({0}/255)", contentOwnerIds.Count));
            }
            if (saveDataOwnerIds.Count > 255)
            {
                throw new ArgumentException(string.Format("Too many SaveDataOwnerIds are specified. ({0}/255)", saveDataOwnerIds.Count));
            }
            Buffer.BlockCopy(BitConverter.GetBytes(contentOwnerIds.Count), 0, descriptor, 1, 1);
            Buffer.BlockCopy(BitConverter.GetBytes(saveDataOwnerIds.Count), 0, descriptor, 2, 1);
            Buffer.BlockCopy(BitConverter.GetBytes(flagPresetsBit), 0, descriptor, 4, 8);
            Buffer.BlockCopy(BitConverter.GetBytes(contentOwnerIdMin), 0, descriptor, 12, 8);
            Buffer.BlockCopy(BitConverter.GetBytes(contentOwnerIdMax), 0, descriptor, 20, 8);
            Buffer.BlockCopy(BitConverter.GetBytes(saveDataOwnerIdMin), 0, descriptor, 28, 8);
            Buffer.BlockCopy(BitConverter.GetBytes(saveDataOwnerIdMax), 0, descriptor, 36, 8);
            for (int i = 0; i < contentOwnerIds.Count; i++)
            {
                Buffer.BlockCopy(BitConverter.GetBytes(contentOwnerIds[i]), 0, descriptor, 44 + 8 * i, 8);
            }
            for (int i = 0; i < saveDataOwnerIds.Count; i++)
            {
                Buffer.BlockCopy(BitConverter.GetBytes(saveDataOwnerIds[i]), 0, descriptor, 44 + 8 * (contentOwnerIds.Count + i), 8);
            }
            return descriptor;
        }
    }

    /// <summary>
    /// Fs アクセス制御データを管理するクラス
    /// </summary>
    public class FacData : FsAccessControl
    {
        internal byte version;
        internal List<ulong> contentOwnerIds;
        internal List<ulong> saveDataOwnerIds;
        internal List<byte> saveDataOwnerIdAccesibilities;

        private int DataSize;
        private int ContentOwnerInfoSize;
        private int SaveDataOwnerInfoSize;

        private DescModel.FaDataModel tmpModel;

        public FacData()
        {
            version = 1;
            DataSize = 28;
            contentOwnerIds = new List<ulong>();
            saveDataOwnerIds = new List<ulong>();
            saveDataOwnerIdAccesibilities = new List<byte>();
            IsImported = false;
        }

        public void ImportFacData(DescModel.FaDataModel model)
        {
            if (IsImported)
            {
                return;
            }

            ulong bit = 0;
            foreach (var flagPreset in model.FlagPresets)
            {
                bit |= FsAccessFlagPresetDefinition.Set.Where(x => x.Name == flagPreset).Select(x => x.Value).FirstOrDefault();
            }
            flagPresetsBit = bit;

            foreach (var id in model.ContentOwnerIds)
            {
                contentOwnerIds.Add(Convert.ToUInt64(id, 16));
            }

            foreach (var id in model.SaveDataOwnerIds)
            {
                saveDataOwnerIds.Add(Convert.ToUInt64(id.Id, 16));
                saveDataOwnerIdAccesibilities.Add(id.AccessibilityValue);
            }

            ContentOwnerInfoSize = contentOwnerIds.Count != 0 ? 4 + contentOwnerIds.Count * 8 : 0;
            SaveDataOwnerInfoSize = saveDataOwnerIds.Count != 0 ? 4 + Utility.RoundUp(saveDataOwnerIds.Count, 4) + saveDataOwnerIds.Count * 8 : 0;

            DataSize += ContentOwnerInfoSize + SaveDataOwnerInfoSize;
            IsImported = true;
        }

        public byte[] ExportBinary()
        {
            byte[] data = new byte[DataSize];
            Buffer.BlockCopy(BitConverter.GetBytes(version), 0, data, 0, 1);
            Buffer.BlockCopy(BitConverter.GetBytes(flagPresetsBit), 0, data, 4, 8);

            int offset = 28;
            Buffer.BlockCopy(BitConverter.GetBytes(offset), 0, data, 12, 4);
            Buffer.BlockCopy(BitConverter.GetBytes(ContentOwnerInfoSize), 0, data, 16, 4);
            Buffer.BlockCopy(BitConverter.GetBytes(offset + ContentOwnerInfoSize), 0, data, 20, 4);
            Buffer.BlockCopy(BitConverter.GetBytes(SaveDataOwnerInfoSize), 0, data, 24, 4);

            if (ContentOwnerInfoSize != 0)
            {
                Buffer.BlockCopy(BitConverter.GetBytes(contentOwnerIds.Count), 0, data, offset, 4);
                offset += 4;
                foreach (var id in contentOwnerIds)
                {
                    Buffer.BlockCopy(BitConverter.GetBytes(id), 0, data, offset, 8);
                    offset += 8;
                }
            }

            if (SaveDataOwnerInfoSize != 0)
            {
                Buffer.BlockCopy(BitConverter.GetBytes(saveDataOwnerIds.Count), 0, data, offset, 4);
                offset += 4;
                foreach (var accessibility in saveDataOwnerIdAccesibilities)
                {
                    Buffer.BlockCopy(BitConverter.GetBytes(accessibility), 0, data, offset, 1);
                    offset += 1;
                }
                offset = Utility.RoundUp(offset, 4);
                foreach (var id in saveDataOwnerIds)
                {
                    Buffer.BlockCopy(BitConverter.GetBytes(id), 0, data, offset, 8);
                    offset += 8;
                }
            }

            return data;
        }

        // desc source からのデータ引き継ぎ
        public void ImportDescFile(DescModel.InputDescModel model)
        {
            if (model.Default == null ||
                model.Default.FsAccessControlData == null)
            {
                return;
            }

            // 一旦 flagPreset に変換し、並び替えなおして出力
            if (model.Default.FsAccessControlData.FlagPresets != null)
            {
                ulong bit = 0;
                foreach (var flagPreset in model.Default.FsAccessControlData.FlagPresets)
                {
                    bit |= FsAccessFlagPresetDefinition.Set.Where(x => x.Name == flagPreset).Select(x => x.Value).FirstOrDefault();
                }
                flagPresetsBit = bit;
            }

            tmpModel =  model.Default.FsAccessControlData;
        }

        public void ExportDescFile(ref DescModel.OutputDescModel model)
        {
            if (tmpModel != null)
            {
                model.Default.FsAccessControlData = tmpModel;
                model.Default.FsAccessControlData.FlagPresets = new List<string>();
                foreach (var name in FsAccessFlagPresetDefinition.Set.Where(x => (x.Value & flagPresetsBit) != 0).Select(x => x.Name))
                {
                    model.Default.FsAccessControlData.FlagPresets.Add(name);
                }
            }
            else
            {
                model.Default.FsAccessControlData = new DescModel.FaDataModel();
                model.Default.FsAccessControlData.FlagPresets = new List<string>();
            }
        }
    }
}
