﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Numerics;
using System.Text;
using System.Threading.Tasks;

namespace MakeAddrRegionHeader
{
    internal enum AddrWidth
    {
        Bit32,
        Bit64
    }

    internal enum Sign
    {
        Positive,
        Negative
    }

    // 有効な値は、[0, アドレス空間の終端] です。
    // アドレス空間の終端は、32 ビットのとき 2^32、64 ビットのとき 2^64 です。
    internal struct AddrValue : IComparable<AddrValue>, IEquatable<AddrValue>
    {
        public AddrWidth AddrWidth { get; private set; }
        public BigInteger Value { get; private set; }

        public AddrValue(AddrWidth addrWidth, ulong value)
            : this(addrWidth, new BigInteger(value))
        {
        }

        public AddrValue(AddrWidth addrWidth, BigInteger value)
            : this()
        {
            ExceptionUtil.ThrowIfUndefinedEnumArgument(typeof(AddrWidth), addrWidth, "addrWidth");
            if (!IsValidValue(addrWidth, value))
            {
                throw new AddrOverflowException();
            }

            this.AddrWidth = addrWidth;
            this.Value = value;
        }

        private static bool IsValidValue(AddrWidth addrWidth, BigInteger value)
        {
            return 0 <= value && value <= AddrUtil.AddrValueEnd(addrWidth);
        }

        public static AddrValue operator +(AddrValue lhs, AddrValue rhs)
        {
            ThrowIfWidthInconsistent(lhs, rhs);

            switch (lhs.AddrWidth)
            {
                case AddrWidth.Bit32:
                    return new AddrValue(AddrWidth.Bit32, lhs.Value + rhs.Value);
                case AddrWidth.Bit64:
                    return new AddrValue(AddrWidth.Bit64, lhs.Value + rhs.Value);
                default:
                    throw new InvalidOperationException();
            }
        }

        public static AddrValue operator -(AddrValue lhs, AddrValue rhs)
        {
            ThrowIfWidthInconsistent(lhs, rhs);

            switch (lhs.AddrWidth)
            {
                case AddrWidth.Bit32:
                    return new AddrValue(AddrWidth.Bit32, lhs.Value - rhs.Value);
                case AddrWidth.Bit64:
                    return new AddrValue(AddrWidth.Bit64, lhs.Value - rhs.Value);
                default:
                    throw new InvalidOperationException();
            }
        }

        public static AddrValue operator +(AddrValue lhs, Offset rhs)
        {
            ThrowIfWidthInconsistent(lhs, rhs.Value);

            switch (rhs.Sign)
            {
                case Sign.Positive:
                    return lhs + rhs.Value;
                case Sign.Negative:
                    return lhs - rhs.Value;
                default:
                    throw new InvalidOperationException();
            }
        }

        public static bool operator <=(AddrValue lhs, AddrValue rhs)
        {
            ThrowIfWidthInconsistent(lhs, rhs);

            return lhs.Value <= rhs.Value;
        }

        public static bool operator >=(AddrValue lhs, AddrValue rhs)
        {
            ThrowIfWidthInconsistent(lhs, rhs);

            return lhs.Value <= rhs.Value;
        }

        private static void ThrowIfWidthInconsistent(AddrValue lhs, AddrValue rhs)
        {
            if (!(lhs.AddrWidth == rhs.AddrWidth))
            {
                throw new ArgumentException("ビット幅の異なる AddrValue を計算しようとしました。");
            }
        }

        #region IComparable<AddrValue> メンバー

        public int CompareTo(AddrValue other)
        {
            return Value.CompareTo(other.Value);
        }

        #endregion

        #region IEquatable<AddrValue> メンバー

        public bool Equals(AddrValue other)
        {
            return Value.Equals(other.Value);
        }

        #endregion

        public override bool Equals(object obj)
        {
            if (obj == null || GetType() != obj.GetType())
            {
                return false;
            }

            return Equals((AddrValue)obj);
        }

        public override int GetHashCode()
        {
            return Value.GetHashCode();
        }

        public override string ToString()
        {
            switch (AddrWidth)
            {
                case AddrWidth.Bit32:
                    return ToZeroPaddedHexString(Value, 8);
                case AddrWidth.Bit64:
                    return ToZeroPaddedHexString(Value, 16);
                default:
                    throw new InvalidOperationException();
            }
        }

        // BigInteger は、16 進数に文字列化されるとき、最上位ビットを符号ビットとみなして文字列化されるようである。
        // つまり、正の数の場合最上位ビットが 0 になるように追加の '0' が挿入されることがある。
        // 例: (new BigInteger(0x8)).ToString("X") == "08"
        // 独自に 0 詰めを行う。
        private static string ToZeroPaddedHexString(BigInteger value, int width)
        {
            return value.ToString("x", CultureInfo.InvariantCulture).TrimStart('0').PadLeft(width, '0');
        }
    }

    internal class Offset
    {
        public Sign Sign { get; private set; }
        public AddrValue Value { get; private set; }

        public Offset(Sign sign, AddrValue value)
        {
            ExceptionUtil.ThrowIfUndefinedEnumArgument(typeof(Sign), sign, "sign");

            this.Sign = sign;
            this.Value = value;
        }
    }
}
