﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Htcs;

namespace HtcDaemon.SerialConnection
{
    internal sealed class SerialConnectionMux : Mux
    {
        private bool disposed;
        private HtcsPeerName targetPeerName;
        private string targetEndPointString;
        private Stream lowStream;
        private IChannel controlChannel;

        private CancellationTokenSource cancellationTokenSource = new CancellationTokenSource();
        private AutoResetEvent sendQueuedEvent = new AutoResetEvent(false);
        private ConcurrentQueue<ISendRequest> sendQueue = new ConcurrentQueue<ISendRequest>();
        private Thread sendThread;
        private Thread receiveThread;

        public override HtcsPeerName TargetPeerName { get { return targetPeerName; } }
        public override string TargetEndPointString { get { return targetEndPointString; } }
        public override IChannel ControlChannel { get { return controlChannel; } }

        public SerialConnectionMux(Stream lowStream, HtcsPeerName targetPeerName, string targetEndPointString)
        {
            this.targetPeerName = targetPeerName;
            this.lowStream = lowStream;
            this.targetEndPointString = targetEndPointString;
            this.controlChannel = CreateDataChannelCore(HtcConstants.ControlChannelId, 0);
            this.sendThread = new Thread(SendThreadFunction) { Name = "SerialConnectionMux.SendThread" };
            this.receiveThread = new Thread(ReceiveThreadFunction) { Name = "SerialConnectionMux.ReceiveThread" };
        }

        protected override void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    this.cancellationTokenSource.Cancel();
                    this.sendQueuedEvent.Set();
                    this.lowStream.Close();
                    this.controlChannel.Cancel();
                    this.receiveThread.Join();
                    this.sendThread.Join();
                    this.sendQueuedEvent.Dispose();
                    this.cancellationTokenSource.Dispose();
                }
                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }

            // 基底クラスのDisposeを呼び出す
            base.Dispose(disposing);
        }

        public override void Start()
        {
            this.receiveThread.Start();
            this.sendThread.Start();
        }

        public override void Stop()
        {
            Dispose();
        }

        protected override IChannel CreateDataChannelCore(int channelId, uint remoteBufferLength)
        {
            return new SerialConnectionChannel(this, channelId, remoteBufferLength);
        }

        public void SendMessage(Message message)
        {
            sendQueue.Enqueue(new SendRequestMessage(message));
            sendQueuedEvent.Set();
        }

        public void EnqueueSendRequest(SerialConnectionChannel channel)
        {
            System.Diagnostics.Debug.Assert(channel.ChannelId != HtcConstants.ControlChannelId, "コントロールチャンネルが必要です。");

            sendQueue.Enqueue(new SendRequestChannel(channel));
            sendQueuedEvent.Set();
        }

        private void SendThreadFunction()
        {
            var token = cancellationTokenSource.Token;
            while (!token.IsCancellationRequested)
            {
                sendQueuedEvent.WaitOne();
                ISendRequest request;
                while (sendQueue.TryDequeue(out request))
                {
                    try
                    {
                        request.Execute(this);
                    }
                    catch (OperationCanceledException)
                    {
                        // 後始末は ReceiveThread で行う
                        return;
                    }
                    catch (IOException)
                    {
                        // 後始末は ReceiveThread で行う
                        return;
                    }
                }
            }
        }

        private void ReceiveThreadFunction()
        {
            var token = cancellationTokenSource.Token;
            try
            {
                while (!token.IsCancellationRequested)
                {
                    BinaryReader br = new BinaryReader(lowStream, Encoding.ASCII, true);
                    var message = new Message(br);

                    var channel = (SerialConnectionChannel)GetChannel(message.ChannelId);
                    if (channel != null)
                    {
                        lock (channel)
                        {
                            if (channel.ChannelId != HtcConstants.ControlChannelId)
                            {
                                channel.SetAckedSeq(message.AckedSeq);
                                EnqueueSendRequest(channel);
                            }
                            channel.OnReceived(message);
                        }
                    }
                }
            }
            catch (Exception e)
            {
                if (e is EndOfStreamException ||
                    e is SocketException ||
                    e is IOException ||
                    e is OperationCanceledException)
                {
                    Console.WriteLine("Disconnected Target {0}.", TargetPeerName);
                }
                else
                {
                    Console.WriteLine(e);
                }
            }

            CancelAllChannels();

            // ブリッジ停止
            // チャンネル削除
            RaiseDisconnected();
        }

        private void OnDisconnected(object sender, TargetEventArgs e)
        {
            RaiseDisconnected();
        }

        private interface ISendRequest
        {
            void Execute(SerialConnectionMux mux);
        }

        private class SendRequestMessage : ISendRequest
        {
            private Message message;

            public SendRequestMessage(Message message)
            {
                this.message = message;
            }

            public void Execute(SerialConnectionMux mux)
            {
                byte[] data = message.Data;
                mux.lowStream.Write(data, 0, data.Length);
            }
        }

        private class SendRequestChannel : ISendRequest
        {
            private SerialConnectionChannel channel;

            public SendRequestChannel(SerialConnectionChannel channel)
            {
                this.channel = channel;
            }

            public void Execute(SerialConnectionMux mux)
            {
                lock (channel)
                {
                    uint sendableLength = channel.GetSendableLength();
                    int queuedLength = channel.GetSendQueueQueuedLength();
                    uint lengthToSend = Math.Min(Math.Min(sendableLength, (uint)queuedLength), HtcConstants.MessageBodyLengthMax);
                    if (lengthToSend == 0)
                    {
                        return;
                    }

                    channel.AddSentSeq(lengthToSend);

                    var messageWriter = new MessageWriter(channel.ChannelId, (int)lengthToSend);
                    channel.DequeueFromSendQueue(messageWriter.GetBuffer(), Message.BodyPosition, (int)lengthToSend);
                    var message = messageWriter.MakeMessage();
                    var data = message.Data;
                    mux.lowStream.Write(data, 0, data.Length);
                }
            }
        }
    }
}
