﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.ConstrainedExecution;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;
using Microsoft.Win32.SafeHandles;

namespace HtcDaemon.Native
{
    internal static partial class NativeMethods
    {
        internal enum USBD_PIPE_TYPE : int
        {
            UsbdPipeTypeControl = 0,
            UsbdPipeTypeIsochronous = 1,
            UsbdPipeTypeBulk = 2,
            UsbdPipeTypeInterrupt = 3
        }

        [StructLayout(LayoutKind.Sequential, Pack = 4, Size = 12)]
        internal struct WINUSB_PIPE_INFORMATION
        {
            public USBD_PIPE_TYPE PipeType;
            public byte PipeId;
            public ushort MaximumPacketSize;
            public byte Interval;
        }

        [StructLayout(LayoutKind.Sequential, Pack = 4)]
        internal struct USB_INTERFACE_DESCRIPTOR
        {
            public byte bLength;
            public byte bDescriptorType;
            public byte bInterfaceNumber;
            public byte bAlternateSetting;
            public byte bNumEndpoints;
            public byte bInterfaceClass;
            public byte bInterfaceSubClass;
            public byte bInterfaceProtocol;
            public byte iInterface;
        }

        private const byte USB_ENDPOINT_DIRECTION_MASK = 0x80;

        public const uint SHORT_PACKET_TERMINATE    = 0x01;
        public const uint AUTO_CLEAR_STALL          = 0x02;
        public const uint PIPE_TRANSFER_TIMEOUT     = 0x03;
        public const uint IGNORE_SHORT_PACKETS      = 0x04;
        public const uint ALLOW_PARTIAL_READS       = 0x05;
        public const uint AUTO_FLUSH                = 0x06;
        public const uint RAW_IO                    = 0x07;
        public const uint MAXIMUM_TRANSFER_SIZE     = 0x08;
        public const uint RESET_PIPE_ON_RESUME      = 0x09;

        public static bool WinUsb_ReadPipe(
            WinUsbHandle InterfaceHandle,
            byte PipeID,
            byte[] Buffer,
            int Offset,
            int Count,
            out int LengthTransferred)
        {
            if (Buffer.Length < Offset + Count)
            {
                throw new ArgumentException("Offset and Count describe invalid range in Buffer");
            }

            using (var gchandle = new ScopedGCHandle(Buffer, GCHandleType.Pinned))
            {
                return WinUsb_ReadPipe(
                    InterfaceHandle,
                    PipeID,
                    Marshal.UnsafeAddrOfPinnedArrayElement(Buffer, (int)Offset),
                    Count,
                    out LengthTransferred,
                    IntPtr.Zero);
            }
        }

        public static bool WinUsb_WritePipe(
            WinUsbHandle InterfaceHandle,
            byte PipeID,
            byte[] Buffer,
            int Offset,
            int Count,
            out int LengthTransferred)
        {
            if (Buffer.Length < Offset + Count)
            {
                throw new ArgumentException("Offset and Count describe invalid range in Buffer");
            }

            using (var gchandle = new ScopedGCHandle(Buffer, GCHandleType.Pinned))
            {
                return WinUsb_WritePipe(
                    InterfaceHandle,
                    PipeID,
                    Marshal.UnsafeAddrOfPinnedArrayElement(Buffer, (int)Offset),
                    Count,
                    out LengthTransferred,
                    IntPtr.Zero);
            }
        }

        public static bool USB_ENDPOINT_DIRECTION_OUT(byte addr)
        {
            return (addr & USB_ENDPOINT_DIRECTION_MASK) == 0;
        }

        public static bool USB_ENDPOINT_DIRECTION_IN(byte addr)
        {
            return (addr & USB_ENDPOINT_DIRECTION_MASK) != 0;
        }

        [DllImport("winusb.dll", SetLastError = true, CharSet = CharSet.Auto)]
        [return: MarshalAs(UnmanagedType.Bool)]
        public static extern bool WinUsb_Initialize(
            SafeFileHandle DeviceHandle,
            out WinUsbHandle InterfaceHandle);

        [DllImport("winusb.dll", SetLastError = true, CharSet = CharSet.Auto)]
        [return: MarshalAs(UnmanagedType.Bool)]
        public static extern bool WinUsb_Free(
            IntPtr InterfaceHandle);

        [DllImport("winusb.dll", SetLastError = true, CharSet = CharSet.Auto)]
        [return: MarshalAs(UnmanagedType.Bool)]
        public static extern bool WinUsb_QueryPipe(
            WinUsbHandle InterfaceHandle,
            byte AlternateInterfaceNumber,
            byte PipeIndex,
            out WINUSB_PIPE_INFORMATION PipeInformation);

        [DllImport("winusb.dll", SetLastError = true)]
        [return: MarshalAs(UnmanagedType.Bool)]
        public static extern bool WinUsb_QueryInterfaceSettings(
            WinUsbHandle InterfaceHandle,
            byte AlternateInterfaceNumber,
            out USB_INTERFACE_DESCRIPTOR UsbAltInterfaceDescriptor);

        [DllImport("winusb.dll", SetLastError = true)]
        [return: MarshalAs(UnmanagedType.Bool)]
        public static extern bool WinUsb_AbortPipe(
            WinUsbHandle InterfaceHandle,
            byte PipeID);

        [DllImport("winusb.dll", SetLastError = true)]
        [return: MarshalAs(UnmanagedType.Bool)]
        private static extern bool WinUsb_ReadPipe(
            WinUsbHandle InterfaceHandle,
            byte PipeID,
            IntPtr Buffer,
            int BufferLength,
            out int LengthTransferred,
            IntPtr Overlapped);

        [DllImport("winusb.dll", SetLastError = true)]
        [return: MarshalAs(UnmanagedType.Bool)]
        private static extern bool WinUsb_WritePipe(
            WinUsbHandle InterfaceHandle,
            byte PipeID,
            IntPtr Buffer,
            int BufferLength,
            out int LengthTransferred,
            IntPtr Overlapped);

        [DllImport("winusb.dll", SetLastError = true)]
        [return: MarshalAs(UnmanagedType.Bool)]
        public static extern bool WinUsb_GetPipePolicy(
            WinUsbHandle InterfaceHandle,
            byte PipeID,
            uint PolicyType,
            ref int ValueLength,
            out uint Value);

        [DllImport("winusb.dll", SetLastError = true)]
        [return: MarshalAs(UnmanagedType.Bool)]
        public static extern bool WinUsb_SetPipePolicy(
            WinUsbHandle InterfaceHandle,
            byte PipeID,
            uint PolicyType,
            int ValueLength,
            ref uint Value);
    }
}
