﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Htcs;

namespace HtcDaemon
{
    // 1つのターゲットとの接続。
    internal class HostTargetConnection : IDisposable
    {
        private Mux mux;
        //private ushort protocolVersion;
        private bool disposed;
        private ControlCommunicator controlCommunicator;
        private List<PortMapItem> portMapItems = new List<PortMapItem>();

        public event EventHandler<TargetEventArgs> Disconnected;
        public event EventHandler<TargetPortRegisteredEventArgs> TargetPortRegistered;
        public event EventHandler<TargetPortUnregisteredEventArgs> TargetPortUnregistered;

        public HtcsPeerName TargetPeerName { get { return mux.TargetPeerName; } }
        // デバッグ用: ターゲットの接続のターゲット側端点を示す文字列
        public string TargetEndPointString { get { return mux.TargetEndPointString; } }

        public HostTargetConnection(Mux mux, ControlCommunicator controlCommunicator)
        {
            this.mux = mux;
            this.mux.Disconnected += OnDisconnected;
            this.controlCommunicator = controlCommunicator;
            this.controlCommunicator.TargetPortRegistered += OnTargetPortRegistered;
            this.controlCommunicator.TargetPortUnregistered += OnTargetPortUnregistered;

            Console.WriteLine("HostTargetConnection created {0} {1}", TargetPeerName, TargetEndPointString);
        }

        ~HostTargetConnection()
        {
            this.Dispose(false);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    controlCommunicator.Stop();
                    mux.Stop();
                }
                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        public void Start()
        {
            mux.Start();
            controlCommunicator.Start();
        }

        public void Stop()
        {
            Dispose();
        }

        public void NotifyHostPortRegistered(PortMapItem portMapItem)
        {
            controlCommunicator.NotifyHostPortRegistered(portMapItem);
        }

        public void NotifyHostPortUnregistered(HtcsPortDescriptor htcsPortDescriptor)
        {
            controlCommunicator.NotifyHostPortUnregistered(htcsPortDescriptor);
        }

        private void OnDisconnected(object sender, TargetEventArgs e)
        {
            Console.WriteLine("HostTargetConnection disconnected {0}", TargetPeerName);

            foreach (var item in portMapItems)
            {
                RaiseTargetPortUnregistered(item.HtcsPortDescriptor);
            }
            portMapItems.Clear();

            RaiseDisconnected(e.TargetPeerName);
        }

        private void OnTargetPortRegistered(object sender, TargetPortRegisteredEventArgs e)
        {
            lock (portMapItems)
            {
                portMapItems.Add(e.PortMapItem);
            }
            RaiseTargetPortRegistered(e.PortMapItem);
        }

        private void OnTargetPortUnregistered(object sender, TargetPortUnregisteredEventArgs e)
        {
            lock (portMapItems)
            {
                portMapItems.RemoveAt(portMapItems.FindIndex(d => d.HtcsPortDescriptor == e.HtcsPortDescriptor));
            }
            RaiseTargetPortUnregistered(e.HtcsPortDescriptor);
        }

        private void RaiseDisconnected(HtcsPeerName targetPeerName)
        {
            if (Disconnected != null)
            {
                Disconnected(this, new TargetEventArgs(targetPeerName));
            }
        }

        private void RaiseTargetPortRegistered(PortMapItem portMapItem)
        {
            if (TargetPortRegistered != null)
            {
                TargetPortRegistered(this, new TargetPortRegisteredEventArgs(portMapItem));
            }
        }

        private void RaiseTargetPortUnregistered(HtcsPortDescriptor htcsPortDescriptor)
        {
            if (TargetPortUnregistered != null)
            {
                TargetPortUnregistered(this, new TargetPortUnregisteredEventArgs(htcsPortDescriptor));
            }
        }
    }
}
