﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net.Sockets;
using System.Text;
using System.Threading.Tasks;
using Nintendo.Htcs;

namespace HtcDaemon.DirectSocket
{
    internal class DirectSocketChannel : IChannel
    {
        private bool disposed;
        private NetworkStream underlyingStream;

        public event EventHandler<ChannelEventArgs> Disconnected;

        public DirectSocketChannel(Socket socket, HtcsPeerName targetPeerName)
        {
            this.TargetPeerName = targetPeerName;
            this.underlyingStream = new NetworkStream(socket, true);
        }

        public int ChannelId { get { throw new NotSupportedException(); } }
        public HtcsPeerName TargetPeerName { get; set; }

        ~DirectSocketChannel()
        {
            this.Dispose(false);
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    this.underlyingStream.Close();
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }

        public Message ReceiveMessage()
        {
            try
            {
                BinaryReader br = new BinaryReader(underlyingStream, Encoding.ASCII, true);
                return new Message(br);
            }
            catch (Exception e)
            {
                if (!IsSocketDisconnectedException(e))
                {
                    Console.WriteLine(e);
                    throw;
                }

                // 本当は下位の通信スレッドから Raise されるはずなので、別スレッドで行う
                Task.Run(
                    () => RaiseDisconnected());

                throw new ChannelDisconnectedException(e.Message, e);
            }
        }

        public void SendMessage(Message message)
        {
            try
            {
                byte[] data = message.Data;
                underlyingStream.Write(data, 0, data.Length);
            }
            catch (Exception e)
            {
                if (!IsSocketDisconnectedException(e))
                {
                    Console.WriteLine(e);
                    throw;
                }

                // 本当は下位の通信スレッドから Raise されるはずなので、別スレッドで行う
                Task.Run(
                    () => RaiseDisconnected());

                throw new ChannelDisconnectedException(e.Message, e);
            }
        }

        public void SendData(byte[] data, int offset, int count)
        {
            // DirectSocket ではコントロールチャンネルのみが存在し、SendMessage だけを使用する
            throw new NotSupportedException();
        }

        private bool IsSocketDisconnectedException(Exception e)
        {
            return
                e is OperationCanceledException ||
                e is SocketException ||
                e is IOException;
        }

        public void Cancel()
        {
            this.underlyingStream.Close();
        }

        private void RaiseDisconnected()
        {
            if (Disconnected != null)
            {
                Disconnected(this, new ChannelEventArgs(this));
            }
        }
    }
}
