﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;

namespace HtcDaemon
{
    // byte[] で表されるブロックを追加すると、それらブロックが連続したバイト列であるかのように（別スレッドから）読み出せます。
    // Enqueue と Dequeue を並列に実行することができます。Enqueue 同士、Dequeue 同士は並列に実行できません。
    public class ByteBlockQueue
    {
        private ConcurrentQueue<Block> queuedBlocks = new ConcurrentQueue<Block>();
        private Block partialBlock = null;          // 前回の Dequeue で一部だけ Dequeue されたブロック
        private int partialBlockBytesConsumed = 0;  // partialBlock から読み出し済みのバイト数
        private int queuedBytes = 0;

        public ByteBlockQueue()
        {
        }

        public int Length
        {
            get { return this.queuedBytes; }
        }

        // ストリームに指定したバイト列を追加します。
        // data を所有します。
        public void Enqueue(byte[] data, int offset, int count)
        {
            if (offset + count > data.Length)
            {
                throw new IndexOutOfRangeException();
            }

            this.queuedBlocks.Enqueue(new Block(data, offset, count));
            Interlocked.Add(ref this.queuedBytes, count);
        }

        // 追加されたバイト列から、指定のサイズを取得する。
        public int Dequeue(byte[] buffer, int offset, int count)
        {
            int removedBytes = this.DequeueCore(buffer, offset, count);
            Interlocked.Add(ref this.queuedBytes, -removedBytes);
            return removedBytes;
        }

        // queuedBytes の更新以外を行う
        private int DequeueCore(byte[] buffer, int offset, int count)
        {
            int totalBytesRemoved = 0;

            // partialBlock から読む
            if (this.partialBlock != null)
            {
                int partialMessageRemainingBytes = this.partialBlock.Count - this.partialBlockBytesConsumed;
                if (partialMessageRemainingBytes > count)
                {
                    Buffer.BlockCopy(
                        this.partialBlock.Data,
                        this.partialBlock.Offset + this.partialBlockBytesConsumed,
                        buffer,
                        offset,
                        count);

                    this.partialBlockBytesConsumed += count;
                    return count;
                }

                Buffer.BlockCopy(
                    this.partialBlock.Data,
                    this.partialBlock.Offset + this.partialBlockBytesConsumed,
                    buffer,
                    offset,
                    partialMessageRemainingBytes);

                this.partialBlock = null;
                this.partialBlockBytesConsumed += partialMessageRemainingBytes;
                totalBytesRemoved += partialMessageRemainingBytes;
            }

            while (totalBytesRemoved < count)
            {
                Block block;

                // もう受信されたデータがなければ抜ける
                if (!this.queuedBlocks.TryDequeue(out block))
                {
                    return totalBytesRemoved;
                }

                if (block.Count > count - totalBytesRemoved)
                {
                    Buffer.BlockCopy(
                        block.Data,
                        block.Offset,
                        buffer,
                        offset + totalBytesRemoved,
                        count - totalBytesRemoved);

                    // 残りは次回の Dequeue で読み出す
                    this.partialBlock = block;
                    this.partialBlockBytesConsumed = count - totalBytesRemoved;

                    return count;
                }
                else
                {
                    Buffer.BlockCopy(
                        block.Data,
                        block.Offset,
                        buffer,
                        offset + totalBytesRemoved,
                        block.Count);

                    totalBytesRemoved += block.Count;
                }
            }

            return count;
        }

        private class Block
        {
            public Block(byte[] data, int offset, int count)
            {
                this.Data = data;
                this.Offset = offset;
                this.Count = count;
            }

            public byte[] Data { get; private set; }

            public int Offset { get; private set; }

            public int Count { get; private set; }
        }
    }
}
