﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Htcs;

namespace HtcDaemon.Builtin.HostFs
{
    internal class HostFsCommunicator : BuiltInServiceCommunicatorBase
    {
        private enum HostFsCommand
        {
            HostFsCommand_None = 0,
            HostFsCommand_Result,
            HostFsCommand_FileExistsA,
            HostFsCommand_FileExistsW,
            HostFsCommand_OpenFileA,
            HostFsCommand_OpenFileW,
            HostFsCommand_CloseFile,
            HostFsCommand_ReadFile,
            HostFsCommand_WriteFile,
            HostFsCommand_GetFileSize,
            HostFsCommand_SetFileSize,
            HostFsCommand_FlushFile,

            HostFsCommand_DirectoryExistsA,
            HostFsCommand_DirectoryExistsW,
            HostFsCommand_OpenDirectoryA,
            HostFsCommand_OpenDirectoryW,
            HostFsCommand_CloseDirectory,
            HostFsCommand_ReadDirectoryA,
            HostFsCommand_ReadDirectoryW,

            HostFsCommand_CreateDirectoryA,
            HostFsCommand_CreateDirectoryW,
            HostFsCommand_DeleteFileA,
            HostFsCommand_DeleteFileW,
            HostFsCommand_DeleteDirectoryA,
            HostFsCommand_DeleteDirectoryW,
            HostFsCommand_MoveFileA,
            HostFsCommand_MoveFileW,
            HostFsCommand_MoveDirectoryA,
            HostFsCommand_MoveDirectoryW,
            HostFsCommand_Count
        }

        private enum HostFsResultCode
        {
            HostFsResultCode_Success = 0,
            HostFsResultCode_NotFound,
            HostFsResultCode_AccessDenied,
            HostFsResultCode_InvalidHandle,
            HostFsResultCode_InvalidArgument,
            HostFsResultCode_Failure,
            HostFsResultCode_DirectoryNotEmpty,
            HostFsResultCode_PathAlreadyExists,
            HostFsResultCode_Count,
        }

        private enum HostFsEntryType
        {
            HostFsEntryType_File = 0,
            HostFsEntryType_Directory = 1,
            HostFsEntryType_Count
        }

        // TORIAEZU: 最大の読み出しサイズ
        private const int MaxReadSize = 0x7fffffff;
        private const int WriterBufferSize = 4096;
        // TORIAEZU: パスは、カレントディレクトリ相対
        private const string BasePath = @".\";

        private HostFileSystem fileSystem;
        private BinaryReader reader;
        private BinaryWriter writer;
        private HandleTable<HostFile> fileHandleManager;
        private HandleTable<HostDirectory> directoryHandleManager;
        private CancellationTokenSource cancellationTokenSource = new CancellationTokenSource();
        private bool disposed;

        public HostFsCommunicator(Socket socket_)
            : base(socket_)
        {
            this.fileSystem = new HostFileSystem(BasePath);
            this.reader = new BinaryReader(Stream, Encoding.Unicode, true);
            this.writer = new BinaryWriter(new BufferedStream(Stream, WriterBufferSize), Encoding.Unicode, true);
            this.fileHandleManager = new HandleTable<HostFile>();
            this.directoryHandleManager = new HandleTable<HostDirectory>();
        }

        protected override void Run()
        {
            var token = this.cancellationTokenSource.Token;

            try
            {
                while (!token.IsCancellationRequested)
                {
                    var requestHeader = ReadHeader();
                    Console.WriteLine("HostFileCommand {0}", requestHeader.CommandType);
                    switch (requestHeader.CommandType)
                    {
                        case HostFsCommand.HostFsCommand_None:
                        case HostFsCommand.HostFsCommand_Result:
                            HandleNone(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_OpenFileA:
                            HandleOpenFileA(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_OpenFileW:
                            HandleOpenFileW(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_CloseFile:
                            HandleCloseFile(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_ReadFile:
                            HandleReadFile(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_WriteFile:
                            HandleWriteFile(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_GetFileSize:
                            HandleGetFileSize(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_SetFileSize:
                            HandleSetFileSize(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_FlushFile:
                            HandleFlushFile(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_OpenDirectoryA:
                            HandleOpenDirectoryA(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_OpenDirectoryW:
                            HandleOpenDirectoryW(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_CloseDirectory:
                            HandleCloseDirectory(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_ReadDirectoryA:
                            HandleReadDirectoryA(requestHeader);
                            break;

                        case HostFsCommand.HostFsCommand_FileExistsA:
                            HandleFileExistsA(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_FileExistsW:
                            HandleFileExistsW(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_DirectoryExistsA:
                            HandleDirectoryExistsA(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_DirectoryExistsW:
                            HandleDirectoryExistsW(requestHeader);
                            break;

                        case HostFsCommand.HostFsCommand_CreateDirectoryA:
                            HandleCreateDirectoryA(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_CreateDirectoryW:
                            HandleCreateDirectoryW(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_DeleteFileA:
                            HandleDeleteFileA(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_DeleteFileW:
                            HandleDeleteFileW(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_DeleteDirectoryA:
                            HandleDeleteDirectoryA(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_DeleteDirectoryW:
                            HandleDeleteDirectoryW(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_MoveFileA:
                            HandleMoveFileA(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_MoveFileW:
                            HandleMoveFileW(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_MoveDirectoryA:
                            HandleMoveDirectoryA(requestHeader);
                            break;
                        case HostFsCommand.HostFsCommand_MoveDirectoryW:
                            HandleMoveDirectoryW(requestHeader);
                            break;
                    }
                }
            }
            catch (EndOfStreamException)
            {
            }
            catch (IOException e)
            {
                // ソケットがリモート側で閉じられた
                if (!(e.InnerException is SocketException))
                {
                    throw;
                }
            }
            finally
            {
                this.Cleanup();
            }
        }

        protected override void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    this.cancellationTokenSource.Cancel();
                    this.Cleanup();
                    this.writer.Dispose();
                    this.reader.Dispose();
                    this.cancellationTokenSource.Dispose();
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }

            // 基底クラスのDisposeを呼び出す
            base.Dispose(disposing);
        }
        #region 各リクエストのハンドラ
        //Header:
        //  u32  bodyLength;
        //  u16  type;
        //  u16  result;

        //mb_string:
        //  u16     length;
        //  char    string[length]

        //wide_string:
        //  u16     length;
        //  wchar_t string[length]

        private void HandleNone(Header requestHeader)
        {
            Debug.Assert(false, "不正なメッセージを受信しました");
            this.reader.ReadBytes(requestHeader.Length);
        }

        // request:
        //   mb_string path;
        //
        // response:
        //   u32       exists;
        private void HandleFileExistsA(Header requestHeader)
        {
            string path = ReadMultibyteString();
            HandleFileExistsCore(path);
        }

        // request:
        //   wide_string    path;
        //
        // response:
        //   u8        exists;
        private void HandleFileExistsW(Header requestHeader)
        {
            string path = ReadWideString();
            HandleFileExistsCore(path);
        }

        private void HandleFileExistsCore(string path)
        {
            HostFsResultCode result = HostFsResultCode.HostFsResultCode_Success;
            bool exists = fileSystem.FileExists(path);

            Header responseHeader = new Header(1, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Write(exists);
            this.writer.Flush();

            Console.WriteLine("HostFileExists(\"{0}\") = {1} ({2})", path, exists, result);
        }

        // request:
        //   mb_string path;
        //
        // response:
        //   u32       exists;
        private void HandleDirectoryExistsA(Header requestHeader)
        {
            string path = ReadMultibyteString();
            HandleDirectoryExistsCore(path);
        }

        // request:
        //   wide_string    path;
        //
        // response:
        //   u8        exists;
        private void HandleDirectoryExistsW(Header requestHeader)
        {
            string path = ReadWideString();
            HandleDirectoryExistsCore(path);
        }

        private void HandleDirectoryExistsCore(string path)
        {
            HostFsResultCode result = HostFsResultCode.HostFsResultCode_Success;
            bool exists = fileSystem.DirectoryExists(path);

            Header responseHeader = new Header(1, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Write(exists);
            this.writer.Flush();

            Console.WriteLine("HostFileExists(\"{0}\") = {1} ({2})", path, exists, result);
        }

        // request:
        //   u32       openMode;
        //   mb_string path;
        //
        //
        // response:
        //   u32 handle;
        private void HandleOpenFileA(Header requestHeader)
        {
            uint openMode = this.reader.ReadUInt32();
            string path = ReadMultibyteString();
            HandleOpenFileCore(path, openMode);
        }

        // request:
        //   u32         openMode;
        //   wide_string path;
        //
        // response:
        //   u32 handle;
        private void HandleOpenFileW(Header requestHeader)
        {
            uint openMode = this.reader.ReadUInt32();
            string path = ReadWideString();
            HandleOpenFileCore(path, openMode);
        }

        private void HandleOpenFileCore(string path, uint openMode)
        {
            HostFsResultCode result = HostFsResultCode.HostFsResultCode_Success;
            uint handle = HandleTable<HostFile>.InvalidHandleValue;
            try
            {
                HostFile file = new HostFile(path, (HostFsOpenMode)openMode);
                handle = this.fileHandleManager.Allocate(file);
            }
            catch (ArgumentException)
            {
                result = HostFsResultCode.HostFsResultCode_InvalidArgument;
            }
            catch (NotSupportedException)
            {
                // パスの形式が無効
                result = HostFsResultCode.HostFsResultCode_InvalidArgument;
            }
            catch (IOException e)
            {
                result = ResultCodeFromIOException(e);
            }
            catch (UnauthorizedAccessException)
            {
                result = HostFsResultCode.HostFsResultCode_AccessDenied;
            }

            Header responseHeader = new Header(4, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Write(handle);
            this.writer.Flush();

            Console.WriteLine("OpenHostFile(\"{0}\", {1}) = {2} ({3})", path, openMode, handle, result);
        }

        // request:
        //   u32    handle;
        //
        // response:
        //   (none)
        private void HandleCloseFile(Header requestHeader)
        {
            uint handle = this.reader.ReadUInt32();

            HostFsResultCode result = HostFsResultCode.HostFsResultCode_Success;
            try
            {
                HostFile file = fileHandleManager.GetValue(handle);
                fileHandleManager.Free(handle);
                file.Close();
            }
            catch (KeyNotFoundException)
            {
                result = HostFsResultCode.HostFsResultCode_InvalidHandle;
            }

            Header responseHeader = new Header(0, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Flush();

            Console.WriteLine("CloseHostFile({0}) ({1})", handle, result);
        }

        // request:
        //   u32        handle;
        //   s32        size;
        //   s64        offset;
        //
        // response:
        //   u32        sizeRead;
        //   u8         data[sizeRead]; // TODO: 大きい場合は分割する
        private void HandleReadFile(Header requestHeader)
        {
            uint handle = this.reader.ReadUInt32();
            int  size   = this.reader.ReadInt32();
            long offset = this.reader.ReadInt64();

            byte[] buffer = new byte[size];
            int sizeRead = 0;

            HostFsResultCode result = HostFsResultCode.HostFsResultCode_Success;
            try
            {
                HostFile file = fileHandleManager.GetValue(handle);
                sizeRead = file.Read(offset, buffer, 0, size);
            }
            catch (KeyNotFoundException)
            {
                result = HostFsResultCode.HostFsResultCode_InvalidHandle;
            }
            catch (ArgumentOutOfRangeException)
            {
                result = HostFsResultCode.HostFsResultCode_InvalidArgument;
            }
            catch (IOException e)
            {
                result = ResultCodeFromIOException(e);
            }

            Header responseHeader = new Header(sizeRead + 4, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Write(sizeRead);
            this.writer.Write(buffer, 0, sizeRead);
            this.writer.Flush();

            Console.WriteLine("ReadHostFile({0}, {1}, {2}) = {3} ({4})", handle, offset, size, sizeRead, result);
        }

        // request:
        //   u32        handle;
        //   u32        size;
        //   s64        offset;
        //   u8         flush;
        //   u8         data[size];
        //
        // response:
        //   u32        sizeWritten;
        private void HandleWriteFile(Header requestHeader)
        {
            uint handle = this.reader.ReadUInt32();
            int size = this.reader.ReadInt32();
            long offset = this.reader.ReadInt64();
            bool flush = this.reader.ReadBoolean();
            byte[] data = this.reader.ReadBytes(size);

            HostFsResultCode result = HostFsResultCode.HostFsResultCode_Success;
            try
            {
                HostFile file = fileHandleManager.GetValue(handle);
                file.Write(offset, data, 0, size, flush);
            }
            catch (KeyNotFoundException)
            {
                result = HostFsResultCode.HostFsResultCode_InvalidHandle;
            }
            catch (NotSupportedException)
            {
                // 書き込めない
                result = HostFsResultCode.HostFsResultCode_AccessDenied;
            }
            catch (ArgumentOutOfRangeException)
            {
                result = HostFsResultCode.HostFsResultCode_InvalidArgument;
            }
            catch (IOException e)
            {
                result = ResultCodeFromIOException(e);
            }

            Header responseHeader = new Header(4, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Write(size);
            this.writer.Flush();

            Console.WriteLine("WriteHostFile({0}, {1}, data, {2}, {3}) = {4} ({5})", handle, offset, size, flush, size, result);
        }

        // request:
        //   u32    handle;
        //
        // response:
        //   s64    size;
        private void HandleGetFileSize(Header requestHeader)
        {
            uint handle = this.reader.ReadUInt32();
            long fileSize = 0;

            HostFsResultCode result = HostFsResultCode.HostFsResultCode_Success;
            try
            {
                HostFile file = fileHandleManager.GetValue(handle);
                fileSize = file.GetSize();
            }
            catch (KeyNotFoundException)
            {
                result = HostFsResultCode.HostFsResultCode_InvalidHandle;
            }

            Header responseHeader = new Header(8, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Write(fileSize);
            this.writer.Flush();

            Console.WriteLine("GetHostFileSize({0}) = {1} ({2})", handle, fileSize, result);
        }

        // request:
        //   u32    handle;
        //   s32    padding;
        //   s64    size;
        //
        // response:
        //   (none)
        private void HandleSetFileSize(Header requestHeader)
        {
            uint handle = this.reader.ReadUInt32();
            this.reader.ReadInt32();
            long size = this.reader.ReadInt64();

            HostFsResultCode result = HostFsResultCode.HostFsResultCode_Success;
            try
            {
                HostFile file = fileHandleManager.GetValue(handle);
                file.SetSize(size);
            }
            catch (KeyNotFoundException)
            {
                result = HostFsResultCode.HostFsResultCode_InvalidHandle;
            }
            catch (NotSupportedException)
            {
                // 書き込めない
                result = HostFsResultCode.HostFsResultCode_AccessDenied;
            }
            catch (IOException e)
            {
                // FileStream.SetLength のリファレンスには、スローされる IOException の詳細が記載されていない
                result = ResultCodeFromIOException(e);
            }

            Header responseHeader = new Header(0, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Flush();

            Console.WriteLine("SetFileSize({0}, {1}) ({2})", handle, size, result);
        }

        // request:
        //   u32    handle;
        //
        // response:
        //   (none)
        private void HandleFlushFile(Header requestHeader)
        {
            uint handle = this.reader.ReadUInt32();

            HostFsResultCode result = HostFsResultCode.HostFsResultCode_Success;
            try
            {
                HostFile file = fileHandleManager.GetValue(handle);
                file.Flush();
            }
            catch (KeyNotFoundException)
            {
                result = HostFsResultCode.HostFsResultCode_InvalidHandle;
            }
            catch (IOException e)
            {
                // FileStream.Flush のリファレンスには、スローされる IOException の詳細が記載されていない
                result = ResultCodeFromIOException(e);
            }

            Header responseHeader = new Header(0, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Flush();

            Console.WriteLine("FlushFile({0}) ({1})", handle, result);
        }

        // request:
        //   mb_string path;
        //
        //
        // response:
        //   u32 handle;
        private void HandleOpenDirectoryA(Header requestHeader)
        {
            string path = ReadMultibyteString();
            HandleOpenDirectoryCore(path);
        }

        // request:
        //   wide_string path;
        //
        //
        // response:
        //   u32 handle;
        private void HandleOpenDirectoryW(Header requestHeader)
        {
            string path = ReadWideString();
            HandleOpenDirectoryCore(path);
        }

        private void HandleOpenDirectoryCore(string path)
        {
            HostFsResultCode result = HostFsResultCode.HostFsResultCode_Success;
            uint handle = HandleTable<HostDirectory>.InvalidHandleValue;
            try
            {
                HostDirectory directory = new HostDirectory(path);
                handle = this.directoryHandleManager.Allocate(directory);
            }
            catch (ArgumentException)
            {
                result = HostFsResultCode.HostFsResultCode_InvalidArgument;
            }
            catch (NotSupportedException)
            {
                // パスの形式が無効
                result = HostFsResultCode.HostFsResultCode_InvalidArgument;
            }
            catch (IOException e)
            {
                result = ResultCodeFromIOException(e);
            }
            catch (UnauthorizedAccessException)
            {
                result = HostFsResultCode.HostFsResultCode_AccessDenied;
            }

            Header responseHeader = new Header(4, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Write(handle);
            this.writer.Flush();

            Console.WriteLine("OpenHostDirectory(\"{0}\") = {1} ({2})", path, handle, result);
        }

        private void HandleCloseDirectory(Header requestHeader)
        {
            uint handle = this.reader.ReadUInt32();

            HostFsResultCode result = HostFsResultCode.HostFsResultCode_Success;
            try
            {
                HostDirectory directory = directoryHandleManager.GetValue(handle);
                directoryHandleManager.Free(handle);
                directory.Close();
            }
            catch (KeyNotFoundException)
            {
                result = HostFsResultCode.HostFsResultCode_InvalidHandle;
            }

            Header responseHeader = new Header(0, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Flush();

            Console.WriteLine("CloseHostFile({0}) ({1})", handle, result);
        }

        // request:
        //   u32        handle;
        //   s32        size;
        //   s64        offset;
        //
        // response:
        //   u32        sizeRead;
        //   u8         data[sizeRead]; // TODO: 大きい場合は分割する
        private void HandleReadDirectoryA(Header requestHeader)
        {
            uint handle = this.reader.ReadUInt32();
            int count = this.reader.ReadInt32();
            HostFileSystemEntry[] entries = new HostFileSystemEntry[0];

            HostFsResultCode result = HostFsResultCode.HostFsResultCode_Success;
            try
            {
                HostDirectory directory = directoryHandleManager.GetValue(handle);
                entries = directory.Read(count);
            }
            catch (KeyNotFoundException)
            {
                result = HostFsResultCode.HostFsResultCode_InvalidHandle;
            }
            catch (ArgumentOutOfRangeException)
            {
                result = HostFsResultCode.HostFsResultCode_InvalidArgument;
            }
            catch (IOException e)
            {
                result = ResultCodeFromIOException(e);
            }

            Header responseHeader = new Header(entries.Length * HostFileSystemEntry.TransferSize + 2, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Write(entries.Length);
            foreach (var entry in entries)
            {
                entry.Write(this.writer);
            }
            this.writer.Flush();

            Console.WriteLine("ReadHostDirectory({0}, {1}) = {2} ({3})", handle, count, entries.Length, result);
        }

        // request:
        //   mb_string path;
        //
        //
        // response:
        //   (none);
        private void HandleCreateDirectoryA(Header requestHeader)
        {
            string path = ReadMultibyteString();
            HandleCreateDirectoryCore(path);
        }

        // request:
        //   wide_string path;
        //
        //
        // response:
        //   (none);
        private void HandleCreateDirectoryW(Header requestHeader)
        {
            string path = ReadWideString();
            HandleCreateDirectoryCore(path);
        }

        private void HandleCreateDirectoryCore(string path)
        {
            HostFsResultCode result = HostFsResultCode.HostFsResultCode_Success;
            try
            {
                fileSystem.CreateDirectory(path);
            }
            catch (ArgumentException)
            {
                result = HostFsResultCode.HostFsResultCode_InvalidArgument;
            }
            catch (NotSupportedException)
            {
                // パスの形式が無効
                result = HostFsResultCode.HostFsResultCode_InvalidArgument;
            }
            catch (IOException e)
            {
                result = ResultCodeFromIOException(e);
            }
            catch (UnauthorizedAccessException)
            {
                result = HostFsResultCode.HostFsResultCode_AccessDenied;
            }

            Header responseHeader = new Header(0, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Flush();

            Console.WriteLine("CreateDirectory(\"{0}\") ({1})", path, result);
        }

        // request:
        //   mb_string path;
        //
        //
        // response:
        //   (none);
        private void HandleDeleteFileA(Header requestHeader)
        {
            string path = ReadMultibyteString();
            HandleDeleteFileCore(path);
        }

        // request:
        //   wide_string path;
        //
        //
        // response:
        //   (none);
        private void HandleDeleteFileW(Header requestHeader)
        {
            string path = ReadWideString();
            HandleDeleteFileCore(path);
        }

        private void HandleDeleteFileCore(string path)
        {
            HostFsResultCode result = HostFsResultCode.HostFsResultCode_Success;
            try
            {
                fileSystem.DeleteFile(path);
            }
            catch (ArgumentException)
            {
                result = HostFsResultCode.HostFsResultCode_InvalidArgument;
            }
            catch (NotSupportedException)
            {
                // パスの形式が無効
                result = HostFsResultCode.HostFsResultCode_InvalidArgument;
            }
            catch (IOException e)
            {
                result = ResultCodeFromIOException(e);
            }
            catch (UnauthorizedAccessException)
            {
                result = HostFsResultCode.HostFsResultCode_AccessDenied;
            }

            Header responseHeader = new Header(0, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Flush();

            Console.WriteLine("DeleteFile(\"{0}\") ({1})", path, result);
        }

        // request:
        //   mb_string path;
        //
        //
        // response:
        //   (none);
        private void HandleDeleteDirectoryA(Header requestHeader)
        {
            string path = ReadMultibyteString();
            HandleDeleteDirectoryCore(path);
        }

        // request:
        //   wide_string path;
        //
        //
        // response:
        //   (none);
        private void HandleDeleteDirectoryW(Header requestHeader)
        {
            string path = ReadWideString();
            HandleDeleteDirectoryCore(path);
        }

        private void HandleDeleteDirectoryCore(string path)
        {
            HostFsResultCode result = HostFsResultCode.HostFsResultCode_Success;
            try
            {
                fileSystem.DeleteDirectory(path);
            }
            catch (DirectoryNotEmptyException)
            {
                result = HostFsResultCode.HostFsResultCode_DirectoryNotEmpty;
            }
            catch (ArgumentException)
            {
                result = HostFsResultCode.HostFsResultCode_InvalidArgument;
            }
            catch (NotSupportedException)
            {
                // パスの形式が無効
                result = HostFsResultCode.HostFsResultCode_InvalidArgument;
            }
            catch (IOException e)
            {
                result = ResultCodeFromIOException(e);
            }
            catch (UnauthorizedAccessException)
            {
                result = HostFsResultCode.HostFsResultCode_AccessDenied;
            }

            Header responseHeader = new Header(0, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Flush();

            Console.WriteLine("DeleteDirectory(\"{0}\") ({1})", path, result);
        }
        // request:
        //   u16  currentPathLength;
        //   u16  newPathLength;
        //   char currentPath[currentPathLength];
        //   char newPath[newPathLength];
        //
        //
        // response:
        //   (none);
        private void HandleMoveFileA(Header requestHeader)
        {
            int currentPathLength = this.reader.ReadUInt16();
            int newPathLength = this.reader.ReadUInt16();
            string currentPath = ReadMultibyteStringWithLength(currentPathLength);
            string newPath = ReadMultibyteStringWithLength(newPathLength);
            HandleMoveFileCore(currentPath, newPath);
        }

        // request:
        //   u16      currentPathLength;
        //   u16      newPathLength;
        //   char16_t currentPath[currentPathLength];
        //   char16_t newPath[newPathLength];
        //
        //
        // response:
        //   (none);
        private void HandleMoveFileW(Header requestHeader)
        {
            int currentPathLength = this.reader.ReadUInt16();
            int newPathLength = this.reader.ReadUInt16();
            string currentPath = ReadWideStringWithLength(currentPathLength);
            string newPath = ReadWideStringWithLength(newPathLength);
            HandleMoveFileCore(currentPath, newPath);
        }

        private void HandleMoveFileCore(string currentPath, string newPath)
        {
            HostFsResultCode result = DoErrorHandledMoveAction(() =>
                {
                    fileSystem.MoveFile(currentPath, newPath);
                });

            Header responseHeader = new Header(0, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Flush();

            Console.WriteLine("MoveFile(\"{0}\", \"{1}\") ({2})", currentPath, newPath, result);
        }

        // request:
        //   mb_string currentPath;
        //   mb_string newPath;
        //
        //
        // response:
        //   (none);
        private void HandleMoveDirectoryA(Header requestHeader)
        {
            int currentPathLength = this.reader.ReadUInt16();
            int newPathLength = this.reader.ReadUInt16();
            string currentPath = ReadMultibyteStringWithLength(currentPathLength);
            string newPath = ReadMultibyteStringWithLength(newPathLength);
            HandleMoveDirectoryCore(currentPath, newPath);
        }

        // request:
        //   wide_string currentPath;
        //   wide_string newPath;
        //
        //
        // response:
        //   (none);
        private void HandleMoveDirectoryW(Header requestHeader)
        {
            int currentPathLength = this.reader.ReadUInt16();
            int newPathLength = this.reader.ReadUInt16();
            string currentPath = ReadWideStringWithLength(currentPathLength);
            string newPath = ReadWideStringWithLength(newPathLength);
            HandleMoveDirectoryCore(currentPath, newPath);
        }

        private void HandleMoveDirectoryCore(string currentPath, string newPath)
        {
            HostFsResultCode result = DoErrorHandledMoveAction(() =>
                {
                    fileSystem.MoveDirectory(currentPath, newPath);
                });

            Header responseHeader = new Header(0, HostFsCommand.HostFsCommand_Result, result);
            this.WriteHeader(responseHeader);
            this.writer.Flush();

            Console.WriteLine("MoveDirectory(\"{0}\", \"{1}\") ({2})", currentPath, newPath, result);
        }

        private HostFsResultCode DoErrorHandledMoveAction(Action action)
        {
            try
            {
                action();
            }
            catch (ArgumentException)
            {
                return HostFsResultCode.HostFsResultCode_InvalidArgument;
            }
            catch (NotSupportedException)
            {
                // パスの形式が無効
                return HostFsResultCode.HostFsResultCode_InvalidArgument;
            }
            catch (PathAlreadyExistsException)
            {
                return HostFsResultCode.HostFsResultCode_PathAlreadyExists;
            }
            catch (IOException e)
            {
                return ResultCodeFromIOException(e);
            }
            catch (UnauthorizedAccessException)
            {
                return HostFsResultCode.HostFsResultCode_AccessDenied;
            }

            return HostFsResultCode.HostFsResultCode_Success;
        }

        #endregion

        private Header ReadHeader()
        {
            int length = this.reader.ReadInt32();
            int commandType = this.reader.ReadInt16();
            int resultCode = this.reader.ReadInt16();

            if (commandType < 0 || (int)HostFsCommand.HostFsCommand_Count <= commandType)
            {
                commandType = (int)HostFsCommand.HostFsCommand_None;
            }

            return new Header(length, (HostFsCommand)commandType, (HostFsResultCode)resultCode);
        }

        private void WriteHeader(Header header)
        {
            this.writer.Write(header.Length);
            this.writer.Write((ushort)header.CommandType);
            this.writer.Write((ushort)header.ResultCode);
        }

        private string ReadMultibyteString()
        {
            int length = this.reader.ReadUInt16();
            return ReadMultibyteStringWithLength(length);
        }

        private string ReadMultibyteStringWithLength(int length)
        {
            byte[] bytes = this.reader.ReadBytes(length);
            return Encoding.Default.GetString(bytes);
        }

        private string ReadWideString()
        {
            int length = this.reader.ReadUInt16();
            return ReadWideStringWithLength(length);
        }

        private string ReadWideStringWithLength(int length)
        {
            byte[] bytes = this.reader.ReadBytes(length * 2);
            return Encoding.Unicode.GetString(bytes);
        }

        private HostFsResultCode ResultCodeFromIOException(IOException e)
        {
            if (e is FileNotFoundException || e is DirectoryNotFoundException)
            {
                return HostFsResultCode.HostFsResultCode_NotFound;
            }
            else if (e is PathTooLongException)
            {
                return HostFsResultCode.HostFsResultCode_InvalidArgument;
            }
            else
            {
                return HostFsResultCode.HostFsResultCode_Failure;
            }
        }

        private void Cleanup()
        {
            this.fileHandleManager.Dispose();
            this.directoryHandleManager.Dispose();
        }

        private class HandleTable<T> : IDisposable
            where T : IDisposable
        {
            public const uint InvalidHandleValue = 0xffffffff;
            private uint currentValue = 1;
            private Dictionary<uint, T> files = new Dictionary<uint, T>();
            private bool disposed;

            public HandleTable()
            {
            }

            ~HandleTable()
            {
                this.Dispose(false);
            }

            public void Dispose()
            {
                this.Dispose(true);
                GC.SuppressFinalize(this);
            }

            public T GetValue(uint handle)
            {
                return files[handle];
            }

            // TORIAEZU: handle が uint の最大になると困る
            public uint Allocate(T file)
            {
                uint handle = currentValue;
                currentValue++;

                files[handle] = file;
                return handle;
            }

            public void Free(uint handle)
            {
                files.Remove(handle);
            }

            protected virtual void Dispose(bool disposing)
            {
                if (!this.disposed)
                {
                    if (disposing)
                    {
                        // ここでマネージドリソースをDisposeする
                        foreach (var file in this.files.Values)
                        {
                            file.Dispose();
                        }
                    }

                    // ここでアンマネージドリソースを解放する
                    this.disposed = true;
                }
            }
        }

        private class Header
        {
            public Header(int length, HostFsCommand commandType, HostFsResultCode resultCode)
            {
                this.Length = length;
                this.CommandType = commandType;
                this.ResultCode = resultCode;
            }

            public int Length { get; private set; }
            public HostFsCommand CommandType { get; private set; }
            public HostFsResultCode ResultCode { get; private set; }
        }
    }
}
