﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace HtcDaemon.Bridging
{
    internal class Bridge : IDisposable
    {
        private bool disposed;
        private CancellationTokenSource cancellationTokenSource = new CancellationTokenSource();
        private Thread sendThread;
        private Thread receiveThread;
        private Socket socket;
        private NetworkStream stream;
        private IChannel channel;

        public event EventHandler<ChannelEventArgs> TargetDisconnected;
        public event EventHandler<ChannelEventArgs> HostAppDisconnected;

        // channel は所有しない、socket は所有する
        public Bridge(IChannel channel, Socket socket)
        {
            this.channel = channel;
            this.socket = socket;
            this.stream = new NetworkStream(socket, false);
            this.sendThread = new Thread(SendThreadFunction) { Name = "Bridge.SendThread" };
            this.receiveThread = new Thread(ReceiveThreadFunction) { Name = "Bridge.ReceiveThread" };
        }

        #region IDisposableの実装

        ~Bridge()
        {
            Dispose(false);
        }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    this.cancellationTokenSource.Cancel();
                    this.channel.Cancel();
                    try
                    {
                        this.socket.Shutdown(SocketShutdown.Both);
                    }
                    catch (SocketException)
                    {
                        // RST を受信していた場合、Shutdown から SocketException がスローされる
                    }
                    this.socket.Close();
                    this.stream.Close();
                    this.receiveThread.Join();
                    this.sendThread.Join();
                    this.cancellationTokenSource.Dispose();
                }
                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }

        #endregion

        public void Start()
        {
            this.sendThread.Start();
            this.receiveThread.Start();
        }

        // ターゲットチャンネル→ホスト側アプリ
        private void ReceiveThreadFunction(object obj)
        {
            var token = cancellationTokenSource.Token;

            try
            {
                while (!token.IsCancellationRequested)
                {
                    var message = channel.ReceiveMessage();
                    stream.Write(message.Data, Message.HeaderLength, message.BodyLength);
                }
            }
            catch (ObjectDisposedException)
            {
            }
            catch (SocketException)
            {
            }
            catch (IOException)
            {
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
                RaiseTargetDisconnected(channel);
            }
        }

        // ホスト側アプリ→ターゲットチャンネル
        private void SendThreadFunction()
        {
            var token = cancellationTokenSource.Token;

            try
            {
                while (!token.IsCancellationRequested)
                {
                    byte[] buffer = new byte[HtcConstants.MessageBodyLengthMax];
                    int readSize = stream.Read(buffer, 0, buffer.Length);
                    if (readSize == 0)
                    {
                        // graceful disconnection
                        break;
                    }

                    Array.Resize(ref buffer, readSize);
                    channel.SendData(buffer, 0, readSize);
                }
            }
            catch (SocketException)
            {
            }
            catch (IOException)
            {
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
            }
            RaiseHostAppDisconnected(channel);
        }

        protected void RaiseTargetDisconnected(IChannel channel)
        {
            if (TargetDisconnected != null)
            {
                TargetDisconnected(this, new ChannelEventArgs(channel));
            }
        }

        protected void RaiseHostAppDisconnected(IChannel channel)
        {
            if (HostAppDisconnected != null)
            {
                HostAppDisconnected(this, new ChannelEventArgs(channel));
            }
        }
    }
}
