﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.Serialization.Json;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace NactBuildLogParser
{
    public class LineWithLevel
    {
        public int Level { get; set; }
        public string Line { get; set; }
    }

    public class LogParser
    {
        private const int IndentSize = 2;

        public static string ParseToJsonText(string filepath)
        {
            return ParseToJsonTextFromText(File.ReadAllText(filepath));
        }

        public static string ParseToJsonTextFromText(string text)
        {
            var targets = GetTargetRange(text);
            var lineWithLevels = ConvertToLineWithLevel(targets, IndentSize);
            var jsonDictionary = ConvertToJsonDictionary(lineWithLevels);
            var jsonText = ConvertToJsonText(jsonDictionary);

            return jsonText;
        }

        /// <summary>
        /// パース対象の範囲を取得します。
        /// </summary>
        /// <param name="lines"></param>
        /// <returns></returns>
        public static List<string> GetTargetRange(string text)
        {
            var lines = text.Split(new[] {'\r', '\n' }, StringSplitOptions.RemoveEmptyEntries).ToList();

            // 開始の行を検索する。
            var startLineIndex = lines.FindIndex(x => x.IndexOf("ビルド完了") >= 0);

            // 終わりの行を検索する。
            var endLineIndex = lines.FindIndex(startLineIndex, x => x.IndexOf("Process exited with code 0") >= 0);

            // 対象範囲を取得
            var targets = lines.GetRange(startLineIndex, endLineIndex - startLineIndex);

            // 行頭の余計な部分を除外
            var startLine = lines[startLineIndex];
            var indentLevelRoot = startLine.IndexOf("ビルド完了");
            return targets.Select(x => x.Substring(indentLevelRoot)).ToList();
        }

        /// <summary>
        /// インデントレベル付きのデータに変換します。
        /// </summary>
        /// <param name="lines"></param>
        /// <param name="indentSize"></param>
        /// <returns></returns>
        public static List<LineWithLevel> ConvertToLineWithLevel(List<string> lines, int indentSize)
        {
            return lines.Select(x =>
            {
                var level = x.TakeWhile(y => y == ' ').Count() / indentSize;

                return new LineWithLevel { Line = x.Trim(), Level = level };
            }).ToList();
        }

        /// <summary>
        /// インデントレベル付きデータから、Json に変換するための型に変換します。
        /// </summary>
        /// <param name="lines"></param>
        /// <returns></returns>
        public static List<Dictionary<string, object>> ConvertToJsonDictionary(List<LineWithLevel> lines)
        {
            var json = new List<Dictionary<string, object>>();
            var keys = new Dictionary<int, string>();

            foreach (var line in lines)
            {
                var path = string.Join("/", keys.Take(line.Level).Select(x => x.Value));
                var dictionary = ConvertToKeyDictionary(path, line.Line);
                json.Add(dictionary);

                keys[line.Level] = (string)dictionary["name"];
            }

            return json;
        }

        /// <summary>
        /// 対象の型を Json のテキスト情報に変換します。
        /// </summary>
        /// <param name="data"></param>
        /// <returns></returns>
        public static string ConvertToJsonText(List<Dictionary<string, object>> data)
        {
            var settings = new DataContractJsonSerializerSettings();
            settings.UseSimpleDictionaryFormat = true;

            var serializer = new DataContractJsonSerializer(typeof(List<Dictionary<string, object>>), settings);
            var stream = new MemoryStream();
            serializer.WriteObject(stream, data);

            return Encoding.UTF8.GetString(stream.ToArray());
        }

        /// <summary>
        /// 対象文字列を、ディクショナリに変換します。
        /// </summary>
        /// <param name="line"></param>
        /// <returns></returns>
        public static Dictionary<string, object> ConvertToKeyDictionary(string path, string line)
        {
            var mD = Regex.Match(line, @"^(.+?)\s*\-\s*([\-\d]{2}:[\-\d]{2}:[\-\d]{2}\.\d*)\s*(\d+)\s*回\s*(\d+\.\d+)$");
            if (mD.Success)
            {
                // "cl.exe                   - --:10:29.319   285 回  02.208" パターン
                //
                // Group[1]: cl.exe
                // Group[2]: --:10:29.319
                // Group[3]: 285
                // Group[4]: 02.208

                var name = mD.Groups[1].Value;
                var msec = ParseNactTime(mD.Groups[2].Value).TotalMilliseconds;
                var num = int.Parse(mD.Groups[3].Value);
                var average_msec = ParseNactTime2(mD.Groups[4].Value).TotalMilliseconds;

                return new Dictionary<string, object>
                    {
                        {"name", name },
                        {"msec", msec },
                        {"num", num },
                        {"average_msec", average_msec },
                        {"path", path }
                    };
            }

            var mC = Regex.Match(line, @"^(\w+):\s*([\-\d]{2}:[\-\d]{2}:[\-\d]{2}\.\d*)\s*(\d+) 回$");
            if (mC.Success)
            {
                // "ExecuteProgram:        --:11:44.153   388 回" パターン
                //
                // Group[1]: ExecuteProgram
                // Group[2]: --:11:44.153
                // Group[3]: 388
                var name = mC.Groups[1].Value;
                var msec = ParseNactTime(mC.Groups[2].Value).TotalMilliseconds;
                var num = int.Parse(mC.Groups[3].Value);

                return new Dictionary<string, object>
                    {
                        {"name", name },
                        {"msec", msec },
                        {"num", num },
                        {"path", path }
                    };
            }

            var mB = Regex.Match(line, @"^(\w+):\s*([\-\d]{2}:[\-\d]{2}:[\-\d]{2}\.\d*)$");
            if (mB.Success)
            {
                // "処理時間:              --:-2:10.050" パターン
                //
                // Group[1]: 処理時間
                // Group[2]: --:-2:10.050
                var name = mB.Groups[1].Value;
                var msec = ParseNactTime(mB.Groups[2].Value).TotalMilliseconds;

                return new Dictionary<string, object>
                    {
                        {"name", name },
                        {"msec", msec },
                        {"path", path }
                    };
            }

            var mA = Regex.Match(line, @"^(\w*)\s*=\s*(\d*)$");
            if (mA.Success)
            {
                // "実行対象ルール数 = 401" パターン
                //
                // Group[1]: 実行対象ルール数
                // Group[2]: 401
                var name = mA.Groups[1].Value;
                var num = int.Parse(mA.Groups[2].Value);

                return new Dictionary<string, object>
                    {
                        {"name", name },
                        {"num", num },
                        {"path", path }
                    };
            }

            var mE = Regex.Match(line, @"^(\w*)$");
            if (mE.Success)
            {
                // "ビルド完了" パターン
                var name = mE.Groups[1].Value;

                return new Dictionary<string, object>
                    {
                        {"name", name },
                        {"path", path }
                    };
            }

            throw new ArgumentOutOfRangeException(string.Format("不明なフォーマットです。: {0}", line));
        }

        /// <summary>
        /// "--:-2:10.050" のような Nact の時間情報をパースします。
        /// </summary>
        /// <param name="time"></param>
        /// <returns></returns>
        public static TimeSpan ParseNactTime(string time)
        {
            var match = Regex.Match(time, @"([\-\d]{2}):([\-\d]{2}):([\-\d]{2})\.(\d*)");

            var hours = ParseNactNumber(match.Groups[1].Value);
            var minutes = ParseNactNumber(match.Groups[2].Value);
            var seconds = ParseNactNumber(match.Groups[3].Value);
            var milliseconds = ParseNactNumber(match.Groups[4].Value);

            return new TimeSpan(0, hours, minutes, seconds, milliseconds);
        }

        /// <summary>
        /// "02.034" のような Nact の時間情報をパースします。
        /// </summary>
        /// <param name="time"></param>
        /// <returns></returns>
        public static TimeSpan ParseNactTime2(string time)
        {
            var match = Regex.Match(time, @"(\d*).(\d*)");

            var seconds = ParseNactNumber(match.Groups[1].Value);
            var milliseconds = ParseNactNumber(match.Groups[2].Value);

            return new TimeSpan(0, 0, 0, seconds, milliseconds);
        }

        /// <summary>
        /// "--", "-2", "34" のような数値情報をパースします。
        /// </summary>
        /// <param name="number"></param>
        /// <returns></returns>
        public static int ParseNactNumber(string number)
        {
            var trimmedNumber = number.TrimStart('-');
            if (trimmedNumber == string.Empty)
            {
                return 0;
            }

            return int.Parse(trimmedNumber);
        }
    }
}
