﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;
using ContentsUploader.Assistants;

namespace ContentsUploader.Commands
{
    using static Constants;

    public class UploadCommand : CommandBase
    {
        [CommandLineOption('s', "input", Description = "The file/derctory path of the nsp that you want to upload.", IsRequired = true)]
        public override string NspPathOption { get; set; }

        [CommandLineOption('t', "token", DefaultValue = "", Description = "CLI Token")]
        public override string TokenOption { get; set; }

        [CommandLineOption('u', "user", DefaultValue = "", Description = "Login ID of CLI Token")]
        public override string UserOption { get; set; }

        [CommandLineOption('p', "password", DefaultValue = "", Description = "Password of CLI Token")]
        public override string PasswordOption { get; set; }

        [CommandLineOption('m', "mastering-enabled", DefaultValue = false, Description = "Mastering enabled option. If you set this option, mastering process is done on the server. Default Value was false")]
        public bool IsMasteringEnabled { get; set; }

        [CommandLineOption("allow-non-mastered", DefaultValue = false, IsHidden = true, Description = "Allow not-mastered rom upload option. If you set this option, upload not-mastered rom doesn't cause error. Default Value was false")]
        public bool IsAllowNonMastered { get; set; }

        [CommandLineOption("approve-accept-enabled", DefaultValue = false, Description = "Approve accept enabled option. If you set this option, you can upload version-up patch. But you want to upload non version-up rom for same id, you have to retry revoke command.")]
        public bool IsApproveAcceptEnabled { get; set; }

        [CommandLineOption("skip-revoke-enabled", DefaultValue = false, Description = "skip revoke")]
        public bool IsSkipRevokeEnabled { get; set; }

        [CommandLineOption("skip-delivery-enabled", DefaultValue = false, Description = "skip delivery")]
        public bool IsSkipDeliveryEnabled { get; set; }

        [CommandLineOption("server-checking-enabled", DefaultValue = false, Description = "be enabled all server checking")]
        public bool IsServerCheckingEnabled { get; set; }

        [CommandLineOption("timestamp-order-enabled", DefaultValue = false, IsHidden = true, Description = "upload by timestamp order")]
        public bool IsTimestampOrderEnabled { get; set; }

        [CommandLineOption('j', "execute-parallel", DefaultValue = 1, Description = "Execute parallel upload")]
        public int ParallelNum { get; set; }

        [CommandLineOption("polling-timeout", DefaultValue = 0, Description = "polling timeout minutes for status transition")]
        public int PollingTimeout { get; set; }

        [CommandLineOption("aoc-archive-number", DefaultValue = 0, Description = "Archive number of AddOnContent")]
        public int AocArchiveNo { get; set; }

        [CommandLineOption("use-index-for-aoc-archive-number", DefaultValue = false, Description = "Use index for Archive number")]
        public bool IsUseIndexForAocArchiveNo { get; set; }

        public override void Run()
        {
            Run("Upload", Mode.IsTokenRequired);
        }

        protected override bool ValidateOptions()
        {
            var valid = true;
            if (ParallelNum < 1 || ParallelNum > 8)
            {
                Log.WriteLine($"Error: Please input parallel num within the range of 1-8.");
                valid = false;
            }
            if (PollingTimeout <= 0)
            {
                Log.WriteLine($"Note: To set timeout of rom status transition, please input minutes by --poling-timeout.");
            }
            return valid;
        }

        protected override bool RunByDirectory()
        {
            var path = string.Empty;
            if (!ToolUtility.ConvertToAbsoluteDirectoryPath(out path, NspPathOption))
            {
                Log.WriteLine($"Error: Directory not found. Please check the path \"{path}\".");
                return false;
            }

            var nsps = NspAccessor.ReadDirectory(path, IsVerbose);
            if (nsps.Count == 0)
            {
                Log.WriteLine($"Error: Nsp file not found. Please check the path \"{path}\".");
                return false;
            }
            if (IsSkipRevokeEnabled == false)
            {
                RevokeByDirectory(nsps);
            }

            if (ParallelNum == 1)
            {
                return SequentialUpload(nsps);
            }
            else
            {
                return ParallelUpload(nsps);
            }
        }

        protected override bool RunByFile()
        {
            var path = string.Empty;
            if (!ToolUtility.ConvertToAbsoluteNspFilePath(out path, NspPathOption))
            {
                Log.WriteLine($"Error: Nsp file not found. Please check the path \"{path}\".");
                return false;
            }

            var nsp = new NspAccessor(path, IsVerbose);
            if (IsSkipRevokeEnabled == false)
            {
                RevokeByFile(nsp);
            }

            var nsps = new List<NspAccessor> { nsp };
            return SequentialUpload(nsps);
        }

        private bool RevokeByDirectory(List<NspAccessor> nsps)
        {
            var revoked = true;
            foreach (var nsp in nsps)
            {
                revoked &= RevokeByFile(nsp);
            }
            return revoked;
        }

        private bool RevokeByFile(NspAccessor nsp)
        {
            var revoked = true;
            var d4c = new D4cHelper(Setting.Current);
            foreach (var content in nsp.NspInnerContent)
            {
                if (nsp.Type == ContentMetaType.Application)
                {
                    revoked &= d4c.RevokeApplicationId(nsp.ApplicationId);
                }
                else
                {
                    revoked &= d4c.RevokeContentMetaId(content.ContentMetaId, content.Version);
                }
            }
            return revoked;
        }

        // 実行結果
        private class Result : Tuple<NspAccessor, bool, string, string>
        {
            public NspAccessor Nsp { get { return Item1; } }
            public bool IsSuccess { get { return Item2; } }
            public bool IsFailure { get { return !IsSuccess; } }
            public string RomId { get { return Item3; } }
            public string Message { get { return Item4; } }

            protected Result(NspAccessor nsp, bool result, string romId, string message) : base(nsp, result, romId, message) { }
        }

        // 実行結果：成功
        private class ResultSuccess : Result
        {
            public ResultSuccess(NspAccessor nsp, string romId, string message) : base(nsp, true, romId, message) { }
        }

        // 実行結果：失敗
        private class ResultFailure : Result
        {
            public ResultFailure(NspAccessor nsp, string message) : base(nsp, false, "", message) { }
            public ResultFailure(NspAccessor nsp, string romId, string message) : base(nsp, false, romId, message) { }
        }

        // 実行結果一覧表示
        private bool PrintResultSummary(List<Result> results)
        {
            // 全アップロードの成否判定と、成否毎に nsp 分ける
            var uploaded = true;
            var success = new List<NspAccessor>();
            var failure = new Dictionary<string, List<NspAccessor>>();
            foreach (var result in results)
            {
                if (result.IsSuccess)
                {
                    success.Add(result.Nsp);
                }
                else
                {
                    uploaded = false;

                    var key = result.Message;
                    if (!failure.ContainsKey(key))
                    {
                        failure.Add(key, new List<NspAccessor>());
                    }
                    failure[key].Add(result.Nsp);
                }
            }

            // サマリ出力
            Log.WriteLine($"========================================");
            if (success.Count > 0)
            {
                // 成功一覧
                Log.WriteLine($"Summary of Result Success:");
                foreach (var nsp in NspAccessor.SortForRegister(success))
                {
                    Log.WriteLine($"  {nsp.Path}");
                }
            }
            if (failure.Count > 0)
            {
                // 失敗一覧
                var keys = new List<string>(failure.Keys);
                keys.Sort();
                foreach (var key in keys)
                {
                    Log.WriteLine($"Summary of Result {key}");
                    foreach (var nsp in NspAccessor.SortForRegister(failure[key]))
                    {
                        Log.WriteLine($"  {nsp.Path}");
                    }
                }
            }
            return uploaded;
        }

        // 直列型アップロード
        private bool SequentialUpload(List<NspAccessor> nsps)
        {
            var results = new List<Result>();
            foreach (var nsp in nsps)
            {
                var rops = new RopsExecutor(Setting.Current);
                var result = Upload(rops, nsp);

                Log.WriteLine($"Result: {nsp.Path}, {result.RomId}, {result.Message}");
                results.Add(result);
            }
            return PrintResultSummary(results);
        }

        // 並列型アップロード
        private bool ParallelUpload(List<NspAccessor> nsps)
        {
            var results = new List<Result>();
            foreach (var parallelList in NspAccessor.SortForParallelUpload(nsps))
            {
                ParallelOptions options = new ParallelOptions();
                options.MaxDegreeOfParallelism = ParallelNum;
                Parallel.ForEach(
                    parallelList,
                    options,
                    () => (Result)null,
                    (nsp, status, result) =>
                    {
                        Log.WriteLine($"{nsp.Path} <");

                        // 並列実行の時は最終結果以外のログ出力を OFF にする
                        var rops = new RopsExecutor(Setting.Current, false);
                        result = Upload(rops, nsp);

                        Log.WriteLine($"{nsp.Path} > Result: {result.RomId}, {result.Message}");
                        return result;
                    },
                    (result) =>
                    {
                        lock (results)
                        {
                            results.Add(result);
                        }
                    });
            }
            return PrintResultSummary(results);
        }

        private Result Upload(RopsExecutor rops, NspAccessor nsp)
        {
            rops.Version();

            // 承認の差し戻し
            var uploadable = true;
            foreach (var content in nsp.NspInnerContent)
            {
                rops.RejectAndPending(content.ContentMetaId);

                // ticket 有無を確認
                if (!content.HasTicket && !IsMasteringEnabled && !IsAllowNonMastered)
                {
                    uploadable = false;
                }
            }
            if (!uploadable)
            {
                // ticket なしは許可されていないので失敗
                return new ResultFailure(nsp, "Error: Ticket not found. Please input --mastering-enabled.");
            }

            // upload 実行
            var romId = string.Empty;
            if (!rops.Upload(out romId, nsp.ApplicationId, nsp.Path, IsServerCheckingEnabled))
            {
                return new ResultFailure(nsp, "Error: Upload");
            }

            // Rom のステータスの遷移待機（UPLOAD -> SCREENING -> SUBMITTBLE）
            var submittable = WaitUntillRomStatusChanged(rops, romId, "SUBMITTABLE", "FAILED", "SCREENING_REJECTED");
            if (!string.IsNullOrEmpty(submittable))
            {
                return new ResultFailure(nsp, romId, submittable);
            }

            // submit リトライ回数
            int maxCount = 30;
            rops.WriteLog($"Try to execute rops submit {maxCount} times per 1 minutes.");

            // Aoc アーカイブ番号
            int aocArchiveNo = 0;
            if (nsp.Type == ContentMetaType.AddOnContent)
            {
                aocArchiveNo = IsUseIndexForAocArchiveNo ? (AocArchiveNo + nsp.AocIndex) : AocArchiveNo;
            }

            // submit 実行
            int count = 0;
            for (count = 0; count < maxCount; count++)
            {
                var isRetryable = false;
                if (rops.Submit(out isRetryable, romId, aocArchiveNo, IsMasteringEnabled, IsServerCheckingEnabled))
                {
                    // submit に成功したらループを抜ける
                    break;
                }
                else if (!isRetryable)
                {
                    // 再試行不可なら submit は失敗
                    return new ResultFailure(nsp, romId, "Error: Submit");
                }
                else
                {
                    // 再試行可能なら 1 分後に submit をリトライ
                    System.Threading.Thread.Sleep(60000);
                }
            }

            // submit をリトライしても busy 状態が続く場合アップロードタスクを取りやめ
            if (count == maxCount)
            {
                return new ResultFailure(nsp, romId, $"Error: Fail to submit {maxCount} times");
            }

            // Rom のステータスの遷移待機（SUBMITTBLE -> PREENCRYPT -> ENCRYPT -> PUBLISH -> CHECKING）
            var checking = WaitUntillRomStatusChanged(rops, romId, "CHECKING", "FAILED");
            if (!string.IsNullOrEmpty(checking))
            {
                return new ResultFailure(nsp, romId, checking);
            }

            // Rom 承認
            var timeout = PollingTimeout > 0 ? PollingTimeout : ApproveRomTimeoutMinutes;
            var retry = ApproveRomRetryCountMax;
            var interval = ApproveRomIntervalMilliseconds;
            if (!ToolUtility.RetryUntilSuccess(() => rops.ApproveRom(romId, IsApproveAcceptEnabled, timeout), retry, interval))
            {
                return new ResultFailure(nsp, romId, "Error: Approve Rom");
            }

            // 配信設定、タイトル承認
            foreach (var content in nsp.NspInnerContent)
            {
                // skip delivery オプションが付いている場合、配信設定をしないでアップロードする
                if (!IsSkipDeliveryEnabled)
                {
                    if (!rops.Delivery(content.ContentMetaId, content.Version))
                    {
                        return new ResultFailure(nsp, romId, "Error: Delivery");
                    }
                }

                if (!rops.ApproveTitle(content.ContentMetaId))
                {
                    return new ResultFailure(nsp, romId, "Error: Approve Title");
                }
            }
            return new ResultSuccess(nsp, romId, "Contents Uploader Done.");
        }

        private string WaitUntillRomStatusChanged(RopsExecutor rops, string romId, string success, params string[] failure)
        {
            // success 状態になるまで待機、failure 状態なら即時失敗
            var start = System.DateTime.Now;
            var status = string.Empty;
            while (status != success)
            {
                if ((PollingTimeout > 0) && ((System.DateTime.Now - start).TotalMinutes > PollingTimeout))
                {
                    return $"Error: Timeout. Status is {status}";
                }
                if (!rops.GetRomStatus(out status, romId))
                {
                    return $"Error: Fail to be {success}. Request failed.";
                }
                if (failure.Contains(status))
                {
                    return $"Error: Fail to be {success}. Status is {status}.";
                }
                System.Threading.Thread.Sleep(10000);
            }

            // Rom 要求が完了するまで待機
            var failed = false;
            while (!rops.IsRomRequestDone(out failed, romId))
            {
                if ((PollingTimeout > 0) && ((System.DateTime.Now - start).TotalMinutes > PollingTimeout))
                {
                    return $"Error: Timeout. Request not completed.";
                }
                System.Threading.Thread.Sleep(10000);
            }
            if (failed)
            {
                return $"Error: Request failed.";
            }
            // 成功時は空文字を返す
            return "";
        }
    }
}
