﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Net;
using System.Text;

namespace ContentsUploader.Assistants
{
    // アカウント情報
    public class Account
    {
        public static readonly Account DebugUser = new Account("userallall", "R3_userallall");

        public string Id { get; }
        public string Password { get; }
        public bool IsValid { get; }

        public Account(string id, string password)
        {
            Id = id;
            Password = password;
            IsValid = !string.IsNullOrEmpty(Id) && !string.IsNullOrEmpty(Password);
        }

        public override string ToString()
        {
            return Id;
        }
    }

    public class WebAccessor
    {
        public enum Method
        {
            POST,
            DELETE,
            GET,
            PATCH,
            PUT,
        }

        private class ContentType
        {
            public static readonly ContentType Json = new ContentType("application/json");
            public static readonly ContentType MergePatchJson = new ContentType("application/merge-patch+json");

            public string Value { get; }

            private ContentType(string value)
            {
                Value = value;
            }

            public override string ToString()
            {
                return Value;
            }
        }

        private bool IsVerbose { get; }     // 詳細出力の有効化
        private bool IsRedirection { get; } // 標準出力リダイレクトの有効化
        private string Proxy { get; }       // プロキシ―

        public WebAccessor(Setting setting, bool redirection = true)
        {
            Proxy = setting.Proxy;
            IsVerbose = setting.IsVerbose;
            IsRedirection = redirection;
        }

        public WebAccessor(string proxy, bool verbose, bool redirection = true)
        {
            Proxy = proxy;
            IsVerbose = verbose;
            IsRedirection = redirection;
        }

        // data 送信なし、Basic 認証なし
        public bool HttpRequest(Method method, Uri uri)
        {
            var response = string.Empty;
            return HttpRequestCommon(out response, uri, method.ToString(), string.Empty, string.Empty, string.Empty, ContentType.Json);
        }
        public bool HttpRequest(out string response, Method method, Uri uri)
        {
            return HttpRequestCommon(out response, uri, method.ToString(), string.Empty, string.Empty, string.Empty, ContentType.Json);
        }

        // data 送信なし、Basic 認証なし
        public bool HttpRequest(Method method, Uri uri, string data)
        {
            var response = string.Empty;
            return HttpRequestCommon(out response, uri, method.ToString(), data, string.Empty, string.Empty, ContentType.Json);
        }
        public bool HttpRequest(out string response, Method method, Uri uri, string data)
        {
            return HttpRequestCommon(out response, uri, method.ToString(), data, string.Empty, string.Empty, ContentType.Json);
        }

        // data 送信なし、Basic 認証あり
        public bool HttpRequest(Method method, Uri uri, Account account)
        {
            var response = string.Empty;
            return HttpRequestCommon(out response, uri, method.ToString(), string.Empty, account.Id, account.Password, ContentType.Json);
        }
        public bool HttpRequest(out string response, Method method, Uri uri, Account account)
        {
            return HttpRequestCommon(out response, uri, method.ToString(), string.Empty, account.Id, account.Password, ContentType.Json);
        }

        // data 送信あり、Basic 認証あり
        public bool HttpRequest(Method method, Uri uri, string data, Account account)
        {
            var response = string.Empty;
            return HttpRequestCommon(out response, uri, method.ToString(), data, account.Id, account.Password, ContentType.Json);
        }
        public bool HttpRequest(out string response, Method method, Uri uri, string data, Account account)
        {
            return HttpRequestCommon(out response, uri, method.ToString(), data, account.Id, account.Password, ContentType.Json);
        }

        // curl -X PATCH --header "Content-Type: application/merge-patch+json"
        public bool HttpRequestMergePatchJson(Uri uri, string data, Account account)
        {
            var response = string.Empty;
            return HttpRequestCommon(out response, uri, Method.PATCH.ToString(), data, account.Id, account.Password, ContentType.MergePatchJson);
        }
        public bool HttpRequestMergePatchJson(out string response, Uri uri, string data, Account account)
        {
            return HttpRequestCommon(out response, uri, Method.PATCH.ToString(), data, account.Id, account.Password, ContentType.MergePatchJson);
        }

        // Basic 認証のみ
        public bool BasicAuthentication(Uri uri, Account account)
        {
            var response = string.Empty;
            return HttpRequestCommon(out response, uri, string.Empty, string.Empty, account.Id, account.Password, ContentType.Json);
        }
        public bool BasicAuthentication(out string response, Uri uri, Account account)
        {
            return HttpRequestCommon(out response, uri, string.Empty, string.Empty, account.Id, account.Password, ContentType.Json);
        }

        // リクエスト共通処理
        private bool HttpRequestCommon(out string response, Uri uri, string method, string data, string id, string password, ContentType type)
        {
            if (IsVerbose)
            {
                var sb = new StringBuilder();
                sb.Append($"HTTP Request: Method={method}, URI={uri}");
                if (!string.IsNullOrEmpty(id) && !string.IsNullOrEmpty(password))
                {
                    sb.Append($", User={id}:{password}");
                }
                if (!string.IsNullOrEmpty(Proxy))
                {
                    sb.Append($", Proxy={Proxy}");
                }
                WriteLog(sb.ToString());
                WriteData(data, type.ToString());
            }

            var request = WebRequest.CreateHttp(uri);
            request.Accept = ContentType.Json.ToString();
            request.ContentType = type.ToString();

            if (!string.IsNullOrEmpty(Proxy))
            {
                request.Proxy = new WebProxy(Proxy);
            }

            if (!string.IsNullOrEmpty(method))
            {
                request.Method = method;
            }

            if (!string.IsNullOrEmpty(data))
            {
                byte[] buffer = Encoding.UTF8.GetBytes(data);
                using (var stream = request.GetRequestStream())
                {
                    stream.Write(buffer, 0, buffer.Length);
                }
            }

            if (!string.IsNullOrEmpty(id) && !string.IsNullOrEmpty(password))
            {
                request.Headers["Authorization"] = "Basic " +
                    Convert.ToBase64String(
                        Encoding.GetEncoding("ISO-8859-1").GetBytes($"{id}:{password}"));
            }

            response = string.Empty;
            return SendRequest(out response, request);
        }

        // リクエスト送信～受信
        private bool SendRequest(out string contents, HttpWebRequest request)
        {
            try
            {
                using (var response = request.GetResponse() as HttpWebResponse)
                {
                    WriteLog($"HTTP Response: Method={request.Method}, URI={request.Address}, StatusCode={response.StatusCode}");
                    using (var reader = new StreamReader(response.GetResponseStream()))
                    {
                        contents = reader.ReadToEnd();
                        if (IsVerbose)
                        {
                            WriteData(contents, response.ContentType);
                        }
                    }
                }
                return true;
            }
            catch (WebException e)
            {
                // 返送されたエラーステータスを表示
                if (e.Status == WebExceptionStatus.ProtocolError)
                {
                    using (var response = e.Response as HttpWebResponse)
                    {
                        WriteLog($"HTTP Response: Method={request.Method}, URI={request.Address}, StatusCode={response.StatusCode}");
                        using (var reader = new StreamReader(response.GetResponseStream()))
                        {
                            contents = reader.ReadToEnd();
                            WriteData(contents, response.ContentType);
                        }
                    }
                }
                else
                {
                    contents = "";
                    WriteLog($"HTTP Request Exception: {e.Status}");
                }
                return false;
            }
        }

        private void WriteData(string data, string type)
        {
            if (type.Contains("json"))
            {
                data = ToolUtility.FormatIndentedJson(data);
            }
            if (!string.IsNullOrEmpty(data))
            {
                WriteLogAsIs(data);
            }
        }

        private void WriteLog(string log)
        {
            if (IsRedirection)
            {
                Log.WriteLine(log);
            }
        }

        private void WriteLogAsIs(string log)
        {
            if (IsRedirection)
            {
                Log.WriteLineAsIs(log);
            }
        }
    }
}
