﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;

namespace CommandUtility
{
    public class TemporaryFileHolder : IDisposable
    {
        /// <summary>
        /// ツール名から NintendoSDK のツールのガイドラインに沿った一時ディレクトリの名前を生成する。
        /// 一時ディレクトリの名前はツール名＋日時＋ランダム文字列から構成される。
        ///
        /// 生成する名前の例： Nintendo_ToolA_2015-06-08-15-03-07_jyafvb2z.xoh
        /// </summary>
        public static string MakeTemporaryName(string toolName, string name)
        {
            return string.Format("Nintendo_{0}_{1}_{2}_{3}", toolName, name, DateTime.Now.ToString("yyyy-MM-dd-HH-mm-ss"), Path.GetRandomFileName());
        }

        /// <summary>
        /// ツールの名前だけ指定してホルダーを作成します。
        /// 標準の一時ディレクトリのパスを使用します。
        /// </summary>
        public TemporaryFileHolder(string toolName, bool afterDelete = true)
        {
            TemporaryRoot = GetTemporaryRoot();
            ToolName = toolName;
            AfterDelete = afterDelete;
        }

        /// <summary>
        /// 今使用するべき一時ディレクトリのパスを取得します。
        /// </summary>
        /// <returns></returns>
        public static string GetTemporaryRoot()
        {
            if (GlobalTemporaryRoot != null)
            {
                return GlobalTemporaryRoot;
            }

            string TemporaryRoot = null;

            try
            {
                // 本来は別途設定できるようにするべきだが
                // 最初に MakeInitialImage で使用していたためここに特定のこのツールの名前を書く
                TemporaryRoot = Environment.GetEnvironmentVariable("MAKE_INITIAL_IMAGE_TEMP");
            }
            catch (Exception)
            {
                // If we throw a security exception, assume default behavior
                TemporaryRoot = null;
            }

            if (TemporaryRoot == null)
            {
                TemporaryRoot = Path.GetTempPath();
            }

            return TemporaryRoot;
        }

        /// <summary>
        /// 一時ファイルの配置場所を指定して作成します。
        /// </summary>
        public TemporaryFileHolder(string toolName, string temporaryRoot, bool afterDelete = true)
        {
            ToolName = toolName;
            TemporaryRoot = temporaryRoot == null ? GetTemporaryRoot() : temporaryRoot;
            AfterDelete = afterDelete;
        }

        /// <summary>
        /// 一時ファイルのパスを作成します。
        /// ファイルは作成しません。
        /// 作成したファイルはこのインスタンスにより管理され、Dispose 時に削除されます。
        /// </summary>
        public FileInfo CreateTemporaryFilePath(string name, string extension = "")
        {
            var tempFile = new FileInfo(Path.Combine(TemporaryRoot, MakeTemporaryName(ToolName, name) + extension));

            if (tempFile.Exists)
            {
                throw new System.IO.IOException(string.Format("Temporary file already exists: {0}", tempFile.FullName));
            }

            AddTemporaryFile(tempFile);
            return tempFile;
        }

        /// <summary>
        /// 一時ファイルを追加します。
        /// 別の場所で作成した一時ファイルを削除したい場合に使用します。
        /// </summary>
        public void AddTemporaryFile(FileInfo fileInfo)
        {
            files.Add(fileInfo);
        }

        /// <summary>
        /// 一時ディレクトリのパスを作成します。
        /// ディレクトリは作成しません。
        /// 作成したディレクトリはこのインスタンスにより管理され、Dispose 時に削除されます。
        /// </summary>
        public DirectoryInfo CreateTemporaryDirectoryPath(string name)
        {
            var tempDirectory = new DirectoryInfo(Path.Combine(TemporaryRoot, MakeTemporaryName(ToolName, name)));

            if (tempDirectory.Exists)
            {
                throw new System.IO.IOException(string.Format("Temporary directory already exists: {0}", tempDirectory.FullName));
            }

            AddTemporaryDirectory(tempDirectory);
            return tempDirectory;
        }

        /// <summary>
        /// 一時ディレクトリを作成します。
        /// 作成したディレクトリはこのインスタンスにより管理され、Dispose 時に削除されます。
        /// </summary>
        public DirectoryInfo CreateTemporaryDirectory(string name)
        {
            var temporaryDirectory = CreateTemporaryDirectoryPath(name);
            temporaryDirectory.Create();
            return temporaryDirectory;
        }

        /// <summary>
        /// 一時ディレクトリを追加します。
        /// 別の場所で作成した一時ディレクトリを削除したい場合に使用できます。
        /// </summary>
        public void AddTemporaryDirectory(DirectoryInfo directoryInfo)
        {
            directories.Add(directoryInfo);
        }

        /// <summary>
        /// 一時ファイルや一時ディレクトリを全て削除します。
        /// Dispose 時に呼ばれるものなので、通常は明示的に呼ぶ必要はありません。
        /// </summary>
        public void DeleteAll()
        {
            foreach (var file in files)
            {
                if (File.Exists(file.FullName))
                {
                    File.Delete(file.FullName);
                }
            }

            foreach (var directory in directories)
            {
                if (Directory.Exists(directory.FullName))
                {
                    directory.DeleteWithRetry(true);
                }
            }
        }

        public void Dispose()
        {
            if (AfterDelete)
            {
                DeleteAll();
            }
        }

        public static string GlobalTemporaryRoot { get; set; }

        public string ToolName { get; private set; }
        public string TemporaryRoot { get; private set; }
        public bool AfterDelete { get; private set; }
        private List<FileInfo> files = new List<FileInfo>();
        private List<DirectoryInfo> directories = new List<DirectoryInfo>();
    }
}
