﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Diagnostics;
using System.IO;
using System.Text.RegularExpressions;

namespace CommandUtility
{
    public class SdkTool
    {
        public static string MakeArgumentString(string[] arguments)
        {
            return string.Join(" ", from argument in arguments select "\"" + Regex.Replace(argument, @"(\\+)$", @"$1$1") + "\"");
        }

        public static void Execute(string toolName, params string[] arguments)
        {
            Execute(SdkPath.FindToolPath(toolName), arguments);
        }

        public static void Execute(FileInfo toolPath, string[] arguments, string workingDirectory = null, StringBuilder output = null)
        {
            Execute(
                toolPath,
                arguments,
                workingDirectory,
                message =>
                {
                    if(output != null)
                    {
                        output.AppendLine(message);
                    }
                    else
                    {
                        Console.WriteLine("[Tool] {0}", message);
                    }
                },
                errorMessage =>
                {
                    Console.Error.WriteLine(errorMessage);
                });
        }

        public static void Execute(
            FileInfo toolPath,
            string[] arguments,
            string workingDirectory,
            Action<string> standardOutput,
            Action<string> standardError)
        {
            string argString = MakeArgumentString(arguments);

            Log.WriteLine("Execute: {0} {1}", toolPath.FullName, argString);

            using (var process = Process.Start(new ProcessStartInfo()
            {
                FileName = toolPath.FullName,
                Arguments = argString,
                CreateNoWindow = true,
                UseShellExecute = false,
                RedirectStandardError = true,
                RedirectStandardOutput = true,
                WorkingDirectory = workingDirectory
            }))
            {
                if (standardOutput != null)
                {
                    process.OutputDataReceived += MakeMessageHandler(message => {
                        standardOutput(message);
                    });
                }
                else
                {
                    process.OutputDataReceived += MakeMessageHandler(message => {
                        Console.WriteLine("[Tool] {0}", message);
                    });
                }

                if (standardError != null)
                {
                    process.ErrorDataReceived += MakeMessageHandler(message => {
                        standardError(message);
                    });
                }
                else
                {
                    process.ErrorDataReceived += MakeMessageHandler(message => {
                        Console.Error.WriteLine(message);
                    });
                }

                process.BeginOutputReadLine();
                process.BeginErrorReadLine();
                process.WaitForExit();

                if (process.ExitCode != 0)
                {
                    throw new InvalidDataException($"Failed to execute: ExitCode={process.ExitCode} path={toolPath.FullName}. Arguments={string.Join(" ", arguments)}");
                }
            }
        }

        private static DataReceivedEventHandler MakeMessageHandler(Action<string> messageHandler)
        {
            return (o, e) =>
            {
                if (e.Data != string.Empty && e.Data != null)
                {
                    messageHandler(e.Data);
                }
            };
        }
    }
}
