﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.IO;
using YamlDotNet.Serialization;

namespace CommandUtility
{
    public class ConvertUtility
    {
        public static Dictionary<string, string> MakeVariablesDictionary(string[] arguments)
        {
            var dictionary = new Dictionary<string, string>();
            if (arguments == null)
            {
                return new Dictionary<string, string>();
            }
            else
            {
                foreach (var text in arguments)
                {
                    var splitted = text.Split(new char[] { '=' }, 2);

                    if (splitted.Length != 2)
                    {
                        throw new ArgumentException("Invalid variable definition format: " + text);
                    }

                    dictionary[splitted[0]] = splitted[1];
                }

                return dictionary;
            }
        }

        public static string ReplaceVariables(string text, Dictionary<string, string> dictionary)
        {
            var patterns = from tuple in dictionary
                           select new Tuple<string, Regex>(tuple.Value, new Regex("(?<!\\\\)\\$\\{" + tuple.Key + "\\}"));
            return patterns.Aggregate<Tuple<string, Regex>, string>(
                text,
                (replaced, variable) =>
                {
                    return variable.Item2.Replace(replaced, variable.Item1);
                });
        }

        public static string ReplaceSdkToolVariables(string text)
        {
            return Regex.Replace(text, "(?<!\\\\)\\$\\{SdkTool:([^\\}]+)\\}", match => {
                return SdkPath.FindToolPath(match.Groups[1].Value).FullName.Replace("\\", "/");
            });
        }

        public static T LoadYaml<T>(string text)
        {
            var deserializer = new Deserializer();

            using(var textReader = new StringReader(text))
            {
                return deserializer.Deserialize<T>(textReader);
            }
        }

        public static T LoadYamlConfig<T>(FileInfo yamlFile, Dictionary<string, string> variables)
        {
            var configText = FileUtility.LoadText(yamlFile);

            var replaced1 = ReplaceSdkToolVariables(configText);
            var replaced2 = ReplaceVariables(replaced1, variables);

            return LoadYaml<T>(replaced2);
        }

        public static string DumpYaml(object data)
        {
            using (var stringWriter = new StringWriter())
            {
                var serializer = new YamlDotNet.Serialization.Serializer(YamlDotNet.Serialization.SerializationOptions.EmitDefaults);
                serializer.Serialize(stringWriter, data);
                stringWriter.Flush();

                return stringWriter.ToString();
            }
        }

        public delegate int ConvertFunction(byte[] outputBuffer, byte[] inputBuffer, int inputBufferSize);

        public static void ConvertStream(Stream output, Stream input, int bufferSize, ConvertFunction convertFunction)
        {
            var buffer = new byte[bufferSize];
            var convertedBuffer = new byte[bufferSize];
            int readSize = input.Read(buffer, 0, bufferSize);
            while (readSize != 0)
            {
                int convertedSize = convertFunction(convertedBuffer, buffer, readSize);
                output.Write(convertedBuffer, 0, convertedSize);
                readSize = input.Read(buffer, 0, bufferSize);
            }
        }
    }
}
