﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace ElfLibrary
{
    using System;
    using System.IO;

    /// <summary>
    /// プリミティブ データ型を特定のアーキテクチャタイプおよびバイトオーダタイプのバイナリ値として読み取ります。
    /// </summary>
    public sealed class ElfBinaryReader
    {
        /// <summary>
        /// ElfBinaryReader クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="reader">バイナリリーダーです。</param>
        /// <param name="byteOrder">バイトオーダタイプです。</param>
        public ElfBinaryReader(BinaryReader reader, ElfByteOrderType byteOrder = ElfByteOrderType.LittleEndian)
        {
            this.BinaryReader = reader;
            this.ByteOrder = byteOrder;
        }

        /// <summary>
        /// バイトオーダタイプを取得します。
        /// </summary>
        public ElfByteOrderType ByteOrder { get; private set; }

        private BinaryReader BinaryReader { get; set; }

        /// <summary>
        /// ファイルストリームから 1 バイトを読み取り、ストリームの位置を 1 バイト進めます。
        /// </summary>
        /// <returns>読み取ったバイトです。</returns>
        public byte ReadByte()
        {
            return this.BinaryReader.ReadByte();
        }

        /// <summary>
        /// ファイルストリームから指定されたバイト数を読み取り、ストリームの位置をそのバイト数だけ進めます。
        /// </summary>
        /// <param name="count">読み取るバイト数です。</param>
        /// <returns>読み取ったバイトです。</returns>
        public byte[] ReadBytes(int count)
        {
            return this.BinaryReader.ReadBytes(count);
        }

        /// <summary>
        /// ファイルストリームから 2 バイト符号なし整数を読み取り、ストリームの位置を 2 バイト進めます。
        /// </summary>
        /// <returns>読み取った 2 バイト符号なし整数です。</returns>
        public ushort ReadUInt16()
        {
            var value = this.BinaryReader.ReadUInt16();

            if (this.ByteOrder == ElfByteOrderType.LittleEndian)
            {
                return value;
            }
            else
            {
                return (ushort)NumericConvert.SwapEndian(value);
            }
        }

        /// <summary>
        /// ファイルストリームから 4 バイト符号なし整数を読み取り、ストリームの位置を 4 バイト進めます。
        /// </summary>
        /// <returns>読み取った 4 バイト符号なし整数です。</returns>
        public uint ReadUInt32()
        {
            var value = this.BinaryReader.ReadUInt32();

            if (this.ByteOrder == ElfByteOrderType.LittleEndian)
            {
                return value;
            }
            else
            {
                return (uint)NumericConvert.SwapEndian(value);
            }
        }

        /// <summary>
        /// ファイルストリームから 8 バイト符号なし整数を読み取り、ストリームの位置を 8 バイト進めます。
        /// </summary>
        /// <returns>読み取った 8 バイト符号なし整数です。</returns>
        public ulong ReadUInt64()
        {
            var value = this.BinaryReader.ReadUInt64();

            if (this.ByteOrder == ElfByteOrderType.LittleEndian)
            {
                return value;
            }
            else
            {
                return (ulong)NumericConvert.SwapEndian(value);
            }
        }
    }
}
