﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace Bin2Obj
{
    using System;
    using Nintendo.Foundation.IO;
    using ElfLibrary;
    public class Bin2ObjParams
    {
        public const string DefaultSymbolBeging = "%b_begin";
        public const string DefaultSymbolEnd = "%b_end";
        public const string DefaultCompatibleBegin = "_binary_%t";
        public const string DefaultCompatibleEnd = "_binary_%t_end";
        public const string DefaultSectionRWData = ".data";
        public const string DefaultSectionROData = ".rodata";
        public const ushort DefaultAlign = 4;

        private string sectionName;
        private string setMachine;
        private bool setBigEndian = false;
        private bool setLittleEndian = false;
        private bool setLp64 = false;
        private bool setIlp32 = false;

        [CommandLineOption('b', "begin",
            DefaultValue = DefaultSymbolBeging,
            Description = "Set symbol name for top of binary. (*)",
            ValueName = "SYMBOL_BEGIN")]
        public string SymbolBegin { get; set; }

        [CommandLineOption('e', "end",
            DefaultValue = DefaultSymbolEnd,
            Description = "Set symbol name for bottom of binary. (*)",
            ValueName = "SYMBOL_END")]
        public string SymbolEnd { get; set; }

        [CommandLineOption('c', "compatible",
            DefaultValue = false,
            Description = "Use compatible symbols with BinToElf.exe\n"
                        + "Same as \"-b _binary_%%f -e _binary_%%f_end.\n")]
        public bool Compatible{ get; private set; }

        [CommandLineOption('a', "align",
            DefaultValue = DefaultAlign,
            Description = "Set binary data alignment in bytes.",
            ValueName = "ALIGNMENT")]
        public ushort Align { get; private set; }

        public bool Writable { get { return !ReadOnly; } }
        [CommandLineOption('r', "readonly",
            DefaultValue = false,
            Description = "Handle binary data as readonly.")]
        public bool ReadOnly { get; private set; }

        private string sectionRoDataValue = DefaultSectionROData;
        public string SectionRoData
        {
            get { return sectionRoDataValue; }
            set { sectionRoDataValue = value; }
        }
        private string sectionRwDataValue = DefaultSectionRWData;
        public string SectionRwData
        {
            get { return sectionRwDataValue; }
            set { sectionRwDataValue = value; }
        }
        [CommandLineOption('s', "section",
            Description = "Set section name.",
            ValueName = "SECTION")]
        public string SectionName
        {
            get { return sectionName; }
            set
            {
                sectionName = value;
                SectionRoData = value;
                SectionRwData = value;
            }
        }

        private ushort defaultMachineValue = (ushort)ElfMachineType.ARM;
        public ushort DefaultMachine
        {
            get
            {
                return defaultMachineValue;
            }
            set
            {
                defaultMachineValue = value;
            }
        }
        private ushort machineValue = 0;
        public ushort Machine
        {
            get { return machineValue; }
            set { machineValue = value; }
        }
        [CommandLineOption('m', "machine",
            Description = "Machine [arm|ppc|aarch64]. (default=arm)",
            ValueName = "arm|ppc|aarch64")]
        public string SetMachine
        {
            get { return setMachine; }
            set
            {
                if ("arm" == value)
                {
                    Machine = (ushort)ElfMachineType.ARM;
                    DefaultEndian = (byte)ElfByteOrderType.LittleEndian;
                }
                else if ("ppc" == value)
                {
                    Machine = (ushort)ElfMachineType.PowerPC;
                    DefaultEndian = (byte)ElfByteOrderType.BigEndian;
                }
                else if ("aarch64" == value)
                {
                    Machine = (byte)ElfMachineType.AArch64;
                    DefaultEndian = (byte)ElfByteOrderType.LittleEndian;
                }
                else
                {
                    Console.WriteLine(string.Format("{0} は対応しておりません。", value));
                    throw new ArgumentException();
                }

                this.setMachine = value;
            }
        }

        private byte defaultEndianValue = (byte)ElfByteOrderType.LittleEndian;
        public byte DefaultEndian
        {
            get { return defaultEndianValue; }
            set { defaultEndianValue = value; }
        }
        private byte endianValue  = (byte)ElfByteOrderType.None;
        public byte Endian
        {
            get
            {
                return endianValue;
            }
            set
            {
                endianValue = value;
            }
        }
        // Parserの規約上オプション名に大文字を付けれないので、１文字オプションをなくす
        [CommandLineOption("big-endian",
            Description = "Output in big    endian format.")]
        public bool SetBigEndian
        {
            get { return setBigEndian; }
            set
            {
                Endian = (byte)ElfByteOrderType.BigEndian;
                DefaultMachine = (byte)ElfMachineType.PowerPC;
                setBigEndian = true;
            }
        }

        [CommandLineOption("little-endian",
            Description = "Output in little endian format.")]
        public bool SetLittleEndian
        {
            get { return setLittleEndian; }
            set
            {
                Endian = (byte)ElfByteOrderType.LittleEndian;
                DefaultMachine = (byte)ElfMachineType.ARM;
                setLittleEndian = true;
            }
        }

        private byte architectureValue = (byte)ElfArchitectureType.None;
        public byte Architecture
        {
            get { return architectureValue; }
            set { architectureValue = value; }
        }
        [CommandLineOption("lp64",
            Description = "Output in lp64 format.")]
        public bool SetLp64
        {
            get { return setLp64; }
            set
            {
                Architecture = (byte)ElfArchitectureType.Elf64;
                setLp64 = true;
            }
        }

        [CommandLineOption("ilp32",
            Description = "Output in ilp32 format.")]
        public bool SetIlp32
        {
            get { return setIlp32; }
            set
            {
                Architecture = (byte)ElfArchitectureType.Elf32;
                setIlp32 = true;
            }
        }

        [CommandLineValue(0, ValueName = "BINARY_FILE", Description = "Binary filename.")]
        public string BinaryFileName { get; set; }

        [CommandLineValue(1, ValueName = "OBJECT_FILE", Description = "Object filename.")]
        public string ObjectFileName { get; set; }
    }
}
