﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/image/image_ExifExtractor.h>
#include <nn/image/image_JpegEncoder.h>
#include <nn/image/image_JpegDecoder.h>
#include <nn/nn_Abort.h>
#include <memory>
#include <string>
#include <nn/nn_Log.h>

#include "../Util/DeclareAlive.h"
#include "JpegStatus.h"

//#define GetStringBufferLength(s) ((s) ? (s)->Length + 1 : 0)
//#define CreateStringIfNotNull(s) ((s) ? gcnew String(s) : nullptr)

namespace Nintendo { namespace Authoring { namespace ImageLibrary {
    using namespace System;
    using namespace System::Drawing;
    using namespace System::Drawing::Imaging;
    using namespace System::Runtime::InteropServices;

    namespace {
        String^ CreateStringIfNotNull(const char* s) {
            return s ? gcnew String(s) : nullptr;
        }

        template<typename T>
        int32_t GetStringBufferLength(T s)
        {
            return s ? s->Length + 1 : 0;
        }
    }

    public enum class SamplingRatio
    {
        Ratio_420,
        Ratio_422,
        Ratio_444,
    };

    public ref struct Exif
    {
    public:
        Exif()
        {
            Orientation = 1;
        }
        property String^        Maker;
        property String^        Model;
        property Int32          Orientation;
        property String^        Software;
        property String^        DateTime;
        property array<Byte>^   MakerNote;
        property Size^          Dimension;
        property String^        UniqueId;
        property array<Byte>^   Thumbnail;
    };

    public ref class ExifExtractor
    {
    public:
        ExifExtractor()
        {
            auto workBufferSize = nn::image::ExifExtractor::GetWorkBufferSize();
            m_pWorkBuffer = new Byte[workBufferSize];
            m_pExifExtractor = new nn::image::ExifExtractor(m_pWorkBuffer, workBufferSize);
            GC::KeepAlive(this);
        }
        ~ExifExtractor()
        {
            this->!ExifExtractor();
        }
        !ExifExtractor()
        {
            delete m_pExifExtractor;
            delete[] m_pWorkBuffer;
        }

        JpegStatus Parse([Runtime::InteropServices::Out] Exif^ %out, array<Byte>^ raw)
        {
            pin_ptr<Byte> pinRawData = &raw[0];

            m_pExifExtractor->SetExifData(pinRawData, raw->Length);
            NN_JPEG_STATUS_DO(m_pExifExtractor->Analyze());


            out = gcnew Exif();

            size_t size;
            out->Maker = CreateStringIfNotNull(m_pExifExtractor->ExtractMaker(&size));
            out->Model = CreateStringIfNotNull(m_pExifExtractor->ExtractModel(&size));
            out->Software = CreateStringIfNotNull(m_pExifExtractor->ExtractSoftware(&size));
            out->DateTime = CreateStringIfNotNull(m_pExifExtractor->ExtractDateTime(&size));
            out->UniqueId = CreateStringIfNotNull(m_pExifExtractor->ExtractUniqueId(&size));

            nn::image::ExifOrientation orientation;
            if (m_pExifExtractor->ExtractOrientation(&orientation))
            {
                out->Orientation = orientation;
            }

            nn::image::Dimension dimension;
            if (m_pExifExtractor->ExtractEffectiveDimension(&dimension))
            {
                out->Dimension = gcnew Size(dimension.width, dimension.height);
            }

            const void* pSrc = m_pExifExtractor->ExtractMakerNote(&size);
            if (pSrc)
            {
                out->MakerNote = gcnew array<Byte>(size);
                pin_ptr<Byte> p = &out->MakerNote[0];
                std::memcpy(p, pSrc, size);
            }

            pSrc = m_pExifExtractor->ExtractThumbnail(&size);
            if (pSrc)
            {
                out->Thumbnail = gcnew array<Byte>(size);
                pin_ptr<Byte> p = &out->Thumbnail[0];
                std::memcpy(p, pSrc, size);
            }

            return Util::ReturnAndDeclareAlive(this, JpegStatus::Ok);
        }

    private:
        nn::image::ExifExtractor* m_pExifExtractor;
        Byte* m_pWorkBuffer;
    };

    struct RawExif
    {
        class StringBuffer {
        public:
            StringBuffer() : m_Empty(true) {}
            void Assign(const void* s) {
                if (s) {
                    m_Empty = false;
                    m_String = reinterpret_cast<const char*>(s);
                }
                else
                {
                    m_Empty = true;
                }
            }
            const char* GetBuffer() {
                return m_Empty ? nullptr : m_String.c_str();
            }
            size_t GetSize() {
                return m_Empty ? 0 : m_String.size() + 1;
            }
        private:
            std::string m_String;
            bool m_Empty;
        };
        StringBuffer maker;
        StringBuffer software;
        StringBuffer uniqueId;
        StringBuffer dateTime;
        std::unique_ptr<char[]> makerNote;
        size_t makerNoteSize;

    };

    public ref class ExifBuilder
    {
    public:
        ExifBuilder()
        {
            size_t workBufferSize = nn::image::ExifBuilder::GetWorkBufferSize();
            m_pWorkBuffer = new Byte[workBufferSize];
            m_pExifBuilder = new nn::image::ExifBuilder(m_pWorkBuffer, workBufferSize);
            m_pRawExif = new RawExif();
            GC::KeepAlive(this);
        }
        ~ExifBuilder()
        {
            this->!ExifBuilder();
        }
        !ExifBuilder()
        {
            delete m_pRawExif;
            delete m_pExifBuilder;
            delete[] m_pWorkBuffer;
        }

        JpegStatus SetExif(Exif^ exif)
        {
            pin_ptr<Byte> pUniqueId = exif->UniqueId ? &System::Text::Encoding::GetEncoding("utf-8")->GetBytes(exif->UniqueId)[0] : nullptr;
            pin_ptr<Byte> pDateTime = exif->DateTime ? &System::Text::Encoding::GetEncoding("utf-8")->GetBytes(exif->DateTime)[0] : nullptr;
            pin_ptr<Byte> pSoftware = exif->Software ? &System::Text::Encoding::GetEncoding("utf-8")->GetBytes(exif->Software)[0] : nullptr;
            nn::image::ExifOrientation orientation = static_cast<nn::image::ExifOrientation>(exif->Orientation);

            m_pRawExif->dateTime.Assign(pDateTime);
            m_pRawExif->uniqueId.Assign(pUniqueId);
            m_pRawExif->software.Assign(pSoftware);
            if (exif->MakerNote)
            {
                m_pRawExif->makerNoteSize = exif->MakerNote->Length;
                m_pRawExif->makerNote.reset(new char[m_pRawExif->makerNoteSize]);
                pin_ptr<Byte> pMakerNote = &exif->MakerNote[0];
                std::memcpy(m_pRawExif->makerNote.get(), pMakerNote, m_pRawExif->makerNoteSize);
            }

            m_pExifBuilder->SetDateTime(m_pRawExif->dateTime.GetBuffer(), m_pRawExif->dateTime.GetSize());
            m_pExifBuilder->SetMakerNote(m_pRawExif->makerNote.get(), m_pRawExif->makerNoteSize);
            m_pExifBuilder->SetOrientation(orientation);
            m_pExifBuilder->SetSoftware(m_pRawExif->software.GetBuffer(), m_pRawExif->software.GetSize());
            m_pExifBuilder->SetUniqueId(m_pRawExif->uniqueId.GetBuffer(), m_pRawExif->uniqueId.GetSize());
            m_pExifBuilder->SetThumbnail(nullptr, 0);
            NN_JPEG_STATUS_DO(m_pExifBuilder->Analyze());

            return Util::ReturnAndDeclareAlive(this, JpegStatus::Ok);
        }

        array<Byte>^ GetExifData(int32_t width, int32_t height)
        {
            nn::image::Dimension dimension = { width, height };

            auto out = gcnew array<Byte>(m_pExifBuilder->GetAnalyzedOutputSize());
            pin_ptr<Byte> pOut = &out[0];
            m_pExifBuilder->Build(pOut, out->Length, dimension);
            return Util::ReturnAndDeclareAlive(this, out);
        }

        nn::image::ExifBuilder* GetBuilderPointer() {
            return m_pExifBuilder;
        }

    private:
        nn::image::ExifBuilder* m_pExifBuilder;
        Byte* m_pWorkBuffer;
        RawExif* m_pRawExif;
    };


    public ref class JpegEncoder {
    public:
        JpegEncoder() :
            m_pPixelData(nullptr)
        {
            m_pEncoder = new nn::image::JpegEncoder();
            GC::KeepAlive(this);
        }
        ~JpegEncoder() {
            this->!JpegEncoder();
        }
        !JpegEncoder() {
            delete m_pEncoder;
            delete[] m_pPixelData;
        }

        JpegStatus SetPixelData(array<Byte>^ pixelData, PixelFormat pxFormat, Size dimension, int lineAlignment)
        {
            m_pPixelData = new Byte[pixelData->Length];
            pin_ptr<Byte> pinPixelData = &pixelData[0];
            std::memcpy(m_pPixelData, pinPixelData, pixelData->Length);

            nn::image::PixelFormat format;
            switch (pxFormat)
            {
            case PixelFormat::Format24bppRgb:
                format = nn::image::PixelFormat_Rgb24;
                break;
            case PixelFormat::Format32bppArgb:
                format = nn::image::PixelFormat_Rgba32;
                break;
            default:
                return JpegStatus::UnsupportedFormat;
                break;
            }

            nn::image::Dimension d = {
                static_cast<int32_t>(dimension.Width),
                static_cast<int32_t>(dimension.Height) };

            m_pEncoder->SetPixelData(m_pPixelData, format, d, lineAlignment);

            return Util::ReturnAndDeclareAlive(this, JpegStatus::Ok);
        }

        void SetSamplingRatio(SamplingRatio ratio) {
            m_pEncoder->SetSamplingRatio(ConvertSamplingRatio(ratio));
        }

        void SetQuality(int quality) {
            m_pEncoder->SetQuality(quality);
            GC::KeepAlive(this);
        }

        JpegStatus Encode([Runtime::InteropServices::Out] array<Byte>^ %outBuffer, Exif^ exif) {
            NN_JPEG_STATUS_DO(m_pEncoder->Analyze());

            auto workBufferSize = m_pEncoder->GetAnalyzedWorkBufferSize();
            std::unique_ptr<Byte[]> workBuffer(new Byte[workBufferSize]);

            // INFO: image_JpegEncoder.h によるとこのぐらいとっとけば OK とのこと
            size_t bufferSize = 3349964;
            std::unique_ptr<Byte[]> encodedBuffer(new Byte[bufferSize]);


            auto exifBuilder = gcnew ExifBuilder();
            exifBuilder->SetExif(exif);

            size_t outSize;
            NN_JPEG_STATUS_DO(m_pEncoder->Encode(&outSize, encodedBuffer.get(), bufferSize, workBuffer.get(), workBufferSize, exifBuilder->GetBuilderPointer()));

            GC::KeepAlive(exifBuilder);

            outBuffer = gcnew array<Byte>(outSize);
            pin_ptr<Byte> p = &outBuffer[0];
            memcpy(p, encodedBuffer.get(), outSize);


            return Util::ReturnAndDeclareAlive(this, JpegStatus::Ok);
        }

    private:
        nn::image::JpegSamplingRatio ConvertSamplingRatio(SamplingRatio ratio)
        {
            switch (ratio)
            {
            case SamplingRatio::Ratio_420:
                return nn::image::JpegSamplingRatio::JpegSamplingRatio_420;
            case SamplingRatio::Ratio_422:
                return nn::image::JpegSamplingRatio::JpegSamplingRatio_422;
            case SamplingRatio::Ratio_444:
                return nn::image::JpegSamplingRatio::JpegSamplingRatio_444;
            default:
                return nn::image::JpegSamplingRatio::JpegSamplingRatio_420;
            }
        }

    private:

        nn::image::JpegEncoder* m_pEncoder;
        Byte* m_pPixelData;
    };


    public ref class JpegDecoder
    {
    public:
        JpegDecoder() :
            m_pJpegData(nullptr)
        {
            m_pDecoder = new nn::image::JpegDecoder();
            GC::KeepAlive(this);
        }
        ~JpegDecoder()
        {
            this->!JpegDecoder();
        }
        !JpegDecoder()
        {
            delete m_pDecoder;
            delete[] m_pJpegData;
        }

        JpegStatus GetExifData([Runtime::InteropServices::Out] Int32 % outOffset, [Runtime::InteropServices::Out] array<Byte>^ %out, array<Byte>^ jpegData)
        {
            pin_ptr<Byte> p = &jpegData[0];

            const void* pExif;
            size_t exifSize;
            NN_JPEG_STATUS_DO(nn::image::JpegDecoder::GetExifData(&pExif, &exifSize, p, jpegData->Length));

            out = gcnew array<Byte>(exifSize);
            pin_ptr<Byte> pOut = &out[0];
            std::memcpy(pOut, pExif, exifSize);

            outOffset = reinterpret_cast<const Byte*>(pExif) - p;

            return Util::ReturnAndDeclareAlive(this, JpegStatus::Ok);
        }

        JpegStatus SetJpegData(array<Byte>^ jpegData)
        {
            NN_SDK_ASSERT(!m_pJpegData);

            m_pJpegData = new Byte[jpegData->Length];
            pin_ptr<Byte> pJpegData = &jpegData[0];
            std::memcpy(m_pJpegData, pJpegData, jpegData->Length);

            m_pDecoder->SetImageData(m_pJpegData, jpegData->Length);
            NN_JPEG_STATUS_DO(m_pDecoder->Analyze());

            return Util::ReturnAndDeclareAlive(this, JpegStatus::Ok);
        }

        Size GetDimension()
        {
            auto rawDimension = m_pDecoder->GetAnalyzedDimension();
            Size dimension(rawDimension.width, rawDimension.height);
            return Util::ReturnAndDeclareAlive(this, dimension);
        }
    private:
        nn::image::JpegDecoder* m_pDecoder;
        Byte* m_pJpegData;
    };

}}}
