﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/fs_NcaHeader.h>
#include <nn/fssystem/utilTool/fs_AesCtrCounterExtendedStorageBuilder.h>
#include <nn/fssystem/utilTool/fs_IndirectStorageBuilder.h>
#include <nn/fssystem/utilTool/fs_RelocatedIndirectStorageBuilder.h>
#include <nn/fssystem/utilTool/fs_RelocatedIndirectStorageUtils.h>
#include <nn/util/util_IntUtil.h>
#include "DefaultMemoryResource.h"
#include "RelocatedIndirectStorageStream.h"
#include "ManagedStreamStorage.h"
#include "../Util/DeclareAlive.h"

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::IO;
using namespace System::Runtime::InteropServices;

using namespace nn;
using namespace nn::fssystem;
using namespace nn::fssystem::utilTool;

private class RelocatedIndirectStorageStreamImpl
{
public:
    RelocatedIndirectStorageStreamImpl(array<Byte>^ oldFsHeader, int64_t oldFsStartOffset, IReadOnlySource^ oldSource, IReadOnlySource^ oldAesCtrExTableSource, array<Byte>^ newFsHeader, int64_t newFsStartOffset, IReadOnlySource^ newSource, int64_t outputFsStartOffset)
        : m_OldPatchInfo()
        , m_OldStartOffset(oldFsStartOffset)
        , m_OldStorage(oldSource)
        , m_OldAesCtrExTableStorage(oldAesCtrExTableSource)
        , m_NewPatchInfo()
        , m_NewStartOffset(newFsStartOffset)
        , m_NewStorage(newSource)
        , m_OutputStartOffset(outputFsStartOffset)
        , m_Allocator()
        , m_IndirectBuilder()
        , m_AesCtrExBuilder()
        , m_pBinaryRegionInfo()
    {
        // 前バージョンの FsEntry がない
        if( oldFsHeader == nullptr )
        {
            NN_SDK_REQUIRES_EQUAL(oldFsStartOffset, 0);
            NN_SDK_REQUIRES_EQUAL(oldSource, nullptr);

            std::memset(&m_OldPatchInfo, 0, sizeof(m_OldPatchInfo));
        }
        else
        {
            Marshal::Copy(oldFsHeader, NcaPatchInfo::Offset, IntPtr(&m_OldPatchInfo), sizeof(m_OldPatchInfo));
            NN_SDK_ASSERT(m_OldPatchInfo.IsExistIndirectStorage());
        }
        Marshal::Copy(newFsHeader, NcaPatchInfo::Offset, IntPtr(&m_NewPatchInfo), sizeof(m_NewPatchInfo));
        NN_SDK_ASSERT(m_NewPatchInfo.IsExistIndirectStorage());
    }

    void SetBinaryRegionInfo(String^ originalPath, String^ outputPath, array<Byte>^ rawHeader)
    {
        m_pBinaryRegionInfo.reset(new BinaryRegionInfo(originalPath, outputPath, rawHeader));
    }

    void Build(UInt32 shiftSize, UInt32 blockSize, UInt32 regionSize, Int64 matchSize, UInt32 generationForUpdatedRegion)
    {
        fs::MemoryStorage nullStorage(__nullptr, 0);
        fs::SubStorage zeroStorage(&nullStorage, 0, 0);

        RelocatedIndirectStorageBuilder::PatchInfo oldPatchInfo(0, 0, zeroStorage, zeroStorage, zeroStorage);
        if( m_OldPatchInfo.HasIndirectTable() )
        {
            auto result = oldPatchInfo.Initialize(m_OldPatchInfo, m_OldStartOffset, &m_OldStorage);
            if( result.IsFailure() )
            {
                throw gcnew System::Exception(
                    String::Format("Failed to Initialize PatchInfo 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
        }

        RelocatedIndirectStorageBuilder::PatchInfo newPatchInfo;
        auto result = newPatchInfo.Initialize(m_NewPatchInfo, m_NewStartOffset, &m_NewStorage);
        if( result.IsFailure() )
        {
            throw gcnew System::Exception(
                String::Format("Failed to Initialize PatchInfo 0x{0:X8}.", result.GetInnerValueForDebug()));
        }

        // IndirectStorage を生成
        {
            result = m_IndirectBuilder.Initialize(oldPatchInfo, newPatchInfo, m_OutputStartOffset);
            if( result.IsFailure() )
            {
                throw gcnew System::Exception(
                    String::Format("Failed to Initialize RelocatedIndirectStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            if( m_pBinaryRegionInfo != nullptr )
            {
                int64_t offset;
                result = m_IndirectBuilder.QueryBinaryRegionOffset(&offset, blockSize);
                if( result.IsFailure() )
                {
                    throw gcnew System::Exception(
                        String::Format("Failed to QueryBinaryRegionOffset RelocatedIndirectStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
                }

                result = m_pBinaryRegionInfo->Load(&oldPatchInfo.GetDataStorage(), offset, blockSize, regionSize);
                if( result.IsFailure() )
                {
                    throw gcnew System::Exception(
                        String::Format("Failed to Load BinaryRegionInfo 0x{0:X8}.", result.GetInnerValueForDebug()));
                }

                if( m_pBinaryRegionInfo->IsValid() )
                {
                    m_IndirectBuilder.SetBinaryRegion(m_pBinaryRegionInfo->GetRegionArray());
                }
            }

            result = m_IndirectBuilder.Build(&m_Allocator, shiftSize, blockSize, regionSize, matchSize);
            if( result.IsFailure() )
            {
                throw gcnew System::Exception(
                    String::Format("Failed to Build RelocatedIndirectStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            // int にキャストできるかチェック
            // NOTE: この制限はオーサリングツール側の制限で nca ファイルの制限ではない
            //       （IndirectStorage テーブルデータのストリーミングに対応していないため）
            const auto size = GetIndirectTableSize();
            if( !util::IsIntValueRepresentable<int>(size) )
            {
                throw gcnew System::Exception("IndirectStorage table too large.");
            }

            if( m_pBinaryRegionInfo != nullptr )
            {
                m_pBinaryRegionInfo->Save(m_IndirectBuilder.GetBinaryRegion());
            }
        }

        // AesCtrEx テーブル を生成
        {
            AesCtrCounterExtendedStorageBuilder::TableInfo aesCtrExTableInfo;
            if( m_OldPatchInfo.HasAesCtrExTable() )
            {
                result = aesCtrExTableInfo.Initialize(m_OldPatchInfo, &m_OldAesCtrExTableStorage);
                if( result.IsFailure() )
                {
                    throw gcnew System::Exception(
                        String::Format("Failed to Initialize TableInfo 0x{0:X8}.", result.GetInnerValueForDebug()));
                }
            }

            int64_t oldDataStorageSize;
            result = oldPatchInfo.GetDataStorage().GetSize(&oldDataStorageSize);
            if( result.IsFailure() )
            {
                throw gcnew System::Exception(
                    String::Format("Failed to GetSize old storage 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            int64_t newDataStorageSize;
            result = newPatchInfo.GetDataStorage().GetSize(&newDataStorageSize);
            if( result.IsFailure() )
            {
                throw gcnew System::Exception(
                    String::Format("Failed to GetSize new storage 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            m_AesCtrExBuilder.Initialize(
                aesCtrExTableInfo,
                oldPatchInfo.GetStorageOffset(),
                oldDataStorageSize,
                newPatchInfo.GetStorageOffset(),
                m_IndirectBuilder.QueryDataStorageSize()
            );

            result = m_AesCtrExBuilder.Build(
                &m_Allocator,
                m_IndirectBuilder.GetInvertedRelocationTable(),
                GetIndirectTableOffset(),
                GetIndirectTableSize(),
                generationForUpdatedRegion
            );
            if( result.IsFailure() )
            {
                throw gcnew System::Exception(
                    String::Format("Failed to Build AesCtrCounterExtendedStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            // int にキャストできるかチェック
            // NOTE: この制限はオーサリングツール側の制限で nca ファイルの制限ではない
            //       （AesCtrEx テーブルデータのストリーミングに対応していないため）
            const auto size = GetAesCtrExTableSize();
            if( !util::IsIntValueRepresentable<int>(size) )
            {
                throw gcnew System::Exception("AesCtrCounterExtendedStorage table too large.");
            }
        }
    } // NOLINT(impl/function_size)

    void WriteIndirectTable(array<byte>^ headerBuffer, array<byte>^ tableBuffer)
    {
        const auto headerSize = GetIndirectHeaderSize();
        const auto tableSize = static_cast<int>(GetIndirectTableSize());

        if( headerBuffer->GetLength(0) < GetIndirectHeaderSize() ||
            tableBuffer->GetLength(0) < GetIndirectTableSize() )
        {
            throw gcnew System::ArgumentException();
        }

        DefaultAlignedMemoryResource allocator;
        pin_ptr<unsigned char> headerPtr = &headerBuffer[0];
        fs::MemoryStorage headerStorage(headerPtr, headerSize);
        pin_ptr<unsigned char> tablePtr = &tableBuffer[0];
        fs::MemoryStorage tableStorage(tablePtr, tableSize);

        const int64_t nodeOffset = 0;
        const auto nodeSize = m_IndirectBuilder.QueryTableNodeStorageSize();
        const auto entryOffset = nodeOffset + nodeSize;
        const auto entrySize = m_IndirectBuilder.QueryTableEntryStorageSize();

        auto result = m_IndirectBuilder.WriteTable(
            &allocator,
            fs::SubStorage(&headerStorage, 0, headerSize),
            fs::SubStorage(&tableStorage, nodeOffset, nodeSize),
            fs::SubStorage(&tableStorage, entryOffset, entrySize)
        );

        headerPtr = nullptr;
        tablePtr = nullptr;

        if( result.IsFailure() )
        {
            throw gcnew System::Exception(
                String::Format("Failed to WriteTable RelocatedIndirectStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
        }
    }

    void WriteAesCtrExTable(array<byte>^ headerBuffer, array<byte>^ tableBuffer)
    {
        const auto headerSize = GetAesCtrExHeaderSize();
        const auto tableSize = static_cast<int>(GetAesCtrExTableSize());

        if( headerBuffer->GetLength(0) < GetAesCtrExHeaderSize() ||
            tableBuffer->GetLength(0) < GetAesCtrExTableSize() )
        {
            throw gcnew System::ArgumentException();
        }

        DefaultAlignedMemoryResource allocator;
        pin_ptr<unsigned char> headerPtr = &headerBuffer[0];
        fs::MemoryStorage headerStorage(headerPtr, headerSize);
        pin_ptr<unsigned char> tablePtr = &tableBuffer[0];
        fs::MemoryStorage tableStorage(tablePtr, tableSize);

        const int64_t nodeOffset = 0;
        const auto nodeSize = m_AesCtrExBuilder.QueryTableNodeStorageSize();
        const auto entryOffset = nodeOffset + nodeSize;
        const auto entrySize = m_AesCtrExBuilder.QueryTableEntryStorageSize();

        auto result = m_AesCtrExBuilder.WriteTable(
            &allocator,
            fs::SubStorage(&headerStorage, 0, headerSize),
            fs::SubStorage(&tableStorage, nodeOffset, nodeSize),
            fs::SubStorage(&tableStorage, entryOffset, entrySize)
        );

        headerPtr = nullptr;
        tablePtr = nullptr;

        if( result.IsFailure() )
        {
            throw gcnew System::Exception(
                String::Format("Failed to WriteTable AesCtrCounterExtendedStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
        }
    }

    void VerifyAesCtrExTable(array<Byte>^ headerBuffer, array<Byte>^ tableBuffer, UInt32 generationForUpdatedRegion)
    {
        pin_ptr<unsigned char> headerPtr = &headerBuffer[0];
        pin_ptr<unsigned char> tablePtr = &tableBuffer[0];
        fs::MemoryStorage tableStorage(tablePtr, tableBuffer->Length);

        DefaultAlignedMemoryResource allocator;
        auto aesCtrExHeader = m_OldPatchInfo.aesCtrExSize > 0
            ? m_OldPatchInfo.aesCtrExHeader : nullptr;

        Result result = VerifyAesCtrCounterExtendedTable(
            &allocator,
            m_OutputStartOffset,
            headerPtr, headerBuffer->Length,
            &tableStorage,
            GetIndirectTableOffset(),
            GetIndirectTableSize(),
            m_OldStartOffset,
            aesCtrExHeader, sizeof(m_OldPatchInfo.aesCtrExHeader),
            &m_OldAesCtrExTableStorage,
            generationForUpdatedRegion,
            m_IndirectBuilder.GetInvertedRelocationTable());

        headerPtr = nullptr;
        tablePtr = nullptr;

        if (result.IsFailure())
        {
            throw gcnew System::Exception(
                String::Format("Failed to VerifyAesCtrExTable 0x{0:X8}.", result.GetInnerValueForDebug()));
        }
    }

    array<byte>^ ReadData(Int64 offset, Int32 size)
    {
        array<byte>^ buffer = gcnew array<byte>(size);
        pin_ptr<unsigned char> ptr = &buffer[0];

        auto result = m_IndirectBuilder.ReadData(offset, ptr, size);
        ptr = nullptr;

        if( result.IsFailure() )
        {
            throw gcnew System::Exception(
                String::Format("Failed to ReadData RelocatedIndirectStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
        }

        return buffer;
    }

    void OutputBuildLog(Stream^ tableStream, Stream^ fragmentStream, Stream^ encryptionStream, Int64 startOffset)
    {
        OutputIndirectStorageBuildLog(tableStream, fragmentStream, startOffset);
        OutputAesCtrExStorageBuildLog(encryptionStream, startOffset);
    }

    IndirectStorageProgress GetProgress() const
    {
        IndirectStorageProgress progress;

        auto pProgress = m_IndirectBuilder.GetProgress();
        if( pProgress != __nullptr )
        {
            const auto info = pProgress->GetInfo();

            progress.phase = IndirectStorageProgress::Phase(info.phase);
            progress.total = info.total;
            progress.value = info.value;
        }
        return progress;
    }

    Int32 GetIndirectHeaderSize() const
    {
        return static_cast<int>(m_IndirectBuilder.QueryTableHeaderStorageSize());
    }

    Int64 GetIndirectTableSize() const
    {
        return util::align_up(m_IndirectBuilder.QueryTableNodeStorageSize() + m_IndirectBuilder.QueryTableEntryStorageSize(), Alignment);
    }

    Int64 GetIndirectTableOffset() const
    {
        return GetDataSize();
    }

    Int32 GetAesCtrExHeaderSize() const
    {
        return static_cast<int>(m_AesCtrExBuilder.QueryTableHeaderStorageSize());
    }

    Int64 GetAesCtrExTableSize() const
    {
        return util::align_up(m_AesCtrExBuilder.QueryTableNodeStorageSize() + m_AesCtrExBuilder.QueryTableEntryStorageSize(), Alignment);
    }

    Int64 GetAesCtrExTableOffset() const
    {
        return GetIndirectTableOffset() + GetIndirectTableSize();
    }

    Int64 GetDataSize() const
    {
        return util::align_up(m_IndirectBuilder.QueryDataStorageSize(), Alignment);
    }

private:
    static const size_t Alignment = NcaHeader::XtsBlockSize;

private:
    void OutputIndirectStorageBuildLog(Stream^ tableStream, Stream^ fragmentStream, Int64 startOffset)
    {
        auto nodeStream = gcnew MemoryStream();
        auto entryStream = gcnew MemoryStream();
        {
            ManagedStreamStorage tableStorage(tableStream);
            ManagedStreamStorage nodeStorage(nodeStream);
            ManagedStreamStorage entryStorage(entryStream);
            ManagedStreamStorage dataStorage(fragmentStream);
            DefaultAlignedMemoryResource allocator;

            auto result = m_IndirectBuilder.OutputBuildLog(&allocator, &tableStorage, &nodeStorage, &entryStorage, &dataStorage, startOffset);
            if( result.IsFailure() )
            {
                throw gcnew System::Exception(String::Format(
                    "Failed to OutputBuildLog IndirectStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
        }

        nodeStream->Position = 0;
        nodeStream->CopyTo(tableStream);
        entryStream->Position = 0;
        entryStream->CopyTo(tableStream);

        delete nodeStream;
        delete entryStream;
    }

    void OutputAesCtrExStorageBuildLog(Stream^ stream, Int64 startOffset)
    {
        DefaultAlignedMemoryResource allocator;

        auto nodeStream = gcnew MemoryStream();
        auto entryStream = gcnew MemoryStream();
        {
            ManagedStreamStorage tableStorage(stream);
            ManagedStreamStorage nodeStorage(nodeStream);
            ManagedStreamStorage entryStorage(entryStream);

            auto result = m_AesCtrExBuilder.OutputBuildLog(&allocator, &tableStorage, &nodeStorage, &entryStorage, startOffset, m_IndirectBuilder.GetInvertedRelocationTable());
            if( result.IsFailure() )
            {
                throw gcnew System::Exception(String::Format(
                    "Failed to OutputBuildLog AesCtrCounterExtendedStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
        }

        nodeStream->Position = 0;
        nodeStream->CopyTo(stream);
        entryStream->Position = 0;
        entryStream->CopyTo(stream);

        delete nodeStream;
        delete entryStream;
    }

private:
    NcaPatchInfo m_OldPatchInfo;
    Int64 m_OldStartOffset;
    ManagedSourceStorage m_OldStorage;
    ManagedSourceStorage m_OldAesCtrExTableStorage;
    NcaPatchInfo m_NewPatchInfo;
    Int64 m_NewStartOffset;
    ManagedSourceStorage m_NewStorage;
    Int64 m_OutputStartOffset;
    DefaultAlignedMemoryResource m_Allocator;
    RelocatedIndirectStorageBuilder m_IndirectBuilder;
    AesCtrCounterExtendedStorageBuilder m_AesCtrExBuilder;
    std::unique_ptr<BinaryRegionInfo> m_pBinaryRegionInfo;
};

RelocatedIndirectStorageStream::RelocatedIndirectStorageStream(
    array<Byte>^ oldFsHeader, int64_t oldFsStartOffset, IReadOnlySource^ oldSource, IReadOnlySource^ oldAesCtrExTableSource,
    array<Byte>^ newFsHeader, int64_t newFsStartOffset, IReadOnlySource^ newSource,
    int64_t outputFsStartOffset)
{
    m_pImpl = new RelocatedIndirectStorageStreamImpl(
        oldFsHeader, oldFsStartOffset, oldSource, oldAesCtrExTableSource,
        newFsHeader, newFsStartOffset, newSource,
        outputFsStartOffset);
    GC::KeepAlive(this);
}

RelocatedIndirectStorageStream::~RelocatedIndirectStorageStream()
{
    this->!RelocatedIndirectStorageStream();
}

RelocatedIndirectStorageStream::!RelocatedIndirectStorageStream()
{
    delete m_pImpl;
}

void RelocatedIndirectStorageStream::SetBinaryRegionInfo(String^ originalPath, String^ outputPath, array<Byte>^ rawHeader)
{
    m_pImpl->SetBinaryRegionInfo(originalPath, outputPath, rawHeader);
    GC::KeepAlive(this);
}

void RelocatedIndirectStorageStream::Build(UInt32 shiftSize, UInt32 blockSize, UInt32 regionSize, Int64 matchSize, UInt32 generationForUpdatedRegion)
{
    m_pImpl->Build(shiftSize, blockSize, regionSize, matchSize, generationForUpdatedRegion);
    GC::KeepAlive(this);
}

void RelocatedIndirectStorageStream::WriteIndirectTable(array<Byte>^ headerBuffer, array<Byte>^ tableBuffer)
{
    m_pImpl->WriteIndirectTable(headerBuffer, tableBuffer);
    GC::KeepAlive(this);
}

void RelocatedIndirectStorageStream::WriteAesCtrExTable(array<Byte>^ headerBuffer, array<Byte>^ tableBuffer)
{
    m_pImpl->WriteAesCtrExTable(headerBuffer, tableBuffer);
    GC::KeepAlive(this);
}

void RelocatedIndirectStorageStream::VerifyAesCtrExTable(array<Byte>^ headerBuffer, array<Byte>^ tableBuffer, UInt32 generationForUpdatedRegion)
{
    m_pImpl->VerifyAesCtrExTable(headerBuffer, tableBuffer, generationForUpdatedRegion);
    GC::KeepAlive(this);
}

array<Byte>^ RelocatedIndirectStorageStream::ReadData(Int64 offset, Int32 size)
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->ReadData(offset, size));
}

void RelocatedIndirectStorageStream::OutputBuildLog(Stream^ tableStream, Stream^ fragmentStream, Stream^ encryptionStream, Int64 startOffset)
{
    m_pImpl->OutputBuildLog(tableStream, fragmentStream, encryptionStream, startOffset);
    GC::KeepAlive(this);
}

IndirectStorageProgress RelocatedIndirectStorageStream::GetProgress()
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetProgress());
}

Int32 RelocatedIndirectStorageStream::GetIndirectHeaderSize()
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetIndirectHeaderSize());
}

Int32 RelocatedIndirectStorageStream::GetIndirectTableSize()
{
    return Util::ReturnAndDeclareAlive(this, static_cast<Int32>(m_pImpl->GetIndirectTableSize()));
}

Int64 RelocatedIndirectStorageStream::GetIndirectTableOffset()
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetIndirectTableOffset());
}

Int32 RelocatedIndirectStorageStream::GetAesCtrExHeaderSize()
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetAesCtrExHeaderSize());
}

Int32 RelocatedIndirectStorageStream::GetAesCtrExTableSize()
{
    return Util::ReturnAndDeclareAlive(this, static_cast<Int32>(m_pImpl->GetAesCtrExTableSize()));
}

Int64 RelocatedIndirectStorageStream::GetAesCtrExTableOffset()
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetAesCtrExTableOffset());
}

Int64 RelocatedIndirectStorageStream::GetDataSize()
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetDataSize());
}

}}}
