﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <fstream>
#include <msclr/marshal_cppstd.h>
#include "IndirectStorageStreamUtil.h"

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::IO;

BinaryRegionInfo::BinaryRegionInfo(String^ originalPath, String^ outputPath, array<Byte>^ rawHeader) NN_NOEXCEPT
    : m_Timestamp()
    , m_OutputPath()
    , m_RegionFile()
    , m_RegionBuffer()
    , m_pRegionData(nullptr)
    , m_RegionDataSize(0)
{
    if( String::IsNullOrEmpty(originalPath) || rawHeader == nullptr )
    {
        return;
    }

    // オリジナルファイルの作成時間を取得
    {
        auto originalFileInfo = gcnew FileInfo(originalPath);

        // NOTE: 作成日時ではうまくいかないので最終更新日時を使用する
        m_Timestamp = msclr::interop::marshal_as<std::wstring>(originalFileInfo->LastWriteTime.ToString());
    }

    // リージョンファイルのヘッダを設定
    {
        pin_ptr<unsigned char> headerPtr = &rawHeader[0];

        unsigned char* const buffer = headerPtr;
        const size_t size = rawHeader->GetLength(0);

        m_RegionFile.SetHeader(reinterpret_cast<char*>(buffer), size);

        headerPtr = nullptr;
    }

    m_OutputPath = msclr::interop::marshal_as<std::string>(outputPath + Path::DirectorySeparatorChar);
}

nn::Result BinaryRegionInfo::Load(nn::fs::IStorage* pStorage, int64_t storageOffset, size_t blockSize, size_t regionSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStorage);
    NN_SDK_REQUIRES_GREATER_EQUAL(storageOffset, 0);
    NN_SDK_REQUIRES_GREATER(blockSize, static_cast<size_t>(0));
    NN_SDK_REQUIRES_GREATER(regionSize, static_cast<size_t>(0));

    if( m_OutputPath.empty() )
    {
        NN_RESULT_SUCCESS;
    }

    std::string path;
    // NOTE: リージョンファイルのデータを設定するまでに関数を抜けた場合は Save() が実行されないようにする
    std::swap(path, m_OutputPath);

    // ストレージのサイズチェック
    {
        int64_t storageSize = 0;
        NN_RESULT_DO(pStorage->GetSize(&storageSize));

        // リージョンサイズより小さければ読み込まない
        if( storageSize < storageOffset + static_cast<int64_t>(regionSize) )
        {
            NN_RESULT_SUCCESS;
        }
    }

    // リージョンファイルのデータを設定
    {
        std::unique_ptr<char[]> dataBuffer(new char[regionSize]);

        NN_RESULT_DO(pStorage->Read(storageOffset, dataBuffer.get(), regionSize));

        m_RegionFile.SetData(storageOffset, dataBuffer.get(), regionSize);
    }

    // 出力ファイルを取得
    String^ outputPath;
    {
        char filename[128];
        m_RegionFile.GenerateFileName(filename, 128);

        std::swap(m_OutputPath, path);
        m_OutputPath += filename;

        outputPath = msclr::interop::marshal_as<String^>(m_OutputPath);
    }

    // タイムスタンプの確認＆読み込み
    {
        auto originalFileTime = DateTime::Parse(msclr::interop::marshal_as<String^>(m_Timestamp));
        auto outputFileInfo = gcnew FileInfo(outputPath);

        if( outputFileInfo->Exists && (originalFileTime < outputFileInfo->LastWriteTime) )
        {
            // 8 バイト境界に沿うようにメモリ確保
            m_RegionDataSize = static_cast<size_t>(outputFileInfo->Length);
            m_RegionBuffer.reset(new char[m_RegionDataSize + sizeof(int64_t)]);
            m_pRegionData = reinterpret_cast<char*>(
                nn::util::align_up(reinterpret_cast<uintptr_t>(m_RegionBuffer.get()), sizeof(int64_t)));

            // ファイル読み込み
            {
                std::ifstream file(m_OutputPath, std::ios::binary);
                file.read(m_pRegionData, m_RegionDataSize);
            }

            // 読み込んだリージョンファイルのチェック
            if( !m_RegionFile.CheckRegion(m_pRegionData, m_RegionDataSize) )
            {
                m_RegionBuffer.reset();
                m_pRegionData = nullptr;
            }
        }
    }

    // 不正なファイルを削除しておく
    if( m_pRegionData == nullptr )
    {
        File::Delete(outputPath);
    }

    NN_RESULT_SUCCESS;
}

void BinaryRegionInfo::Save(const nn::fssystem::utilTool::BinaryRegionArray& regions) NN_NOEXCEPT
{
    if( (m_pRegionData == nullptr) && !m_OutputPath.empty() && !regions.empty() )
    {
        m_RegionFile.SetRegion(regions);

        std::ofstream file(m_OutputPath, std::ios::binary);

        const auto headerPtr = &m_RegionFile.GetHeader();
        file.write(reinterpret_cast<const char*>(headerPtr), sizeof(RegionFile::Header));

        const auto regionPtr = regions.data();
        file.write(reinterpret_cast<const char*>(regionPtr), regions.GetBytes());
    }
}

void BucketTreeUtility::CopyHeader(nn::fssystem::BucketTree::Header* outValue, array<Byte>^ headerData)
{
    if( headerData->GetLength(0) < sizeof(nn::fssystem::BucketTree::Header) )
    {
        throw gcnew ArgumentException(String::Format("Failed to copy BucketTree header."));
    }

    System::Runtime::InteropServices::Marshal::Copy(headerData, 0, IntPtr(outValue), sizeof(nn::fssystem::BucketTree::Header));

    auto result = outValue->Verify();
    if( result.IsFailure() )
    {
        throw gcnew ArgumentException(String::Format("Failed to verify BucketTree header 0x{0:X8}.", result.GetInnerValueForDebug()));
    }
}

}}}
