﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>

#include <nn/fssystem/save/fs_HierarchicalIntegrityVerificationStorage.h>
#include <nn/fssystem/fs_IntegrityRomFsStorage.h>
#include <nn/os/os_Mutex.h>

#include "BufferPool.h"
#include "BufferManager.h"

namespace
{
    void GenerateRandomForAuthoringTool(void* pData, size_t size)
    {
        std::memset(pData, 0, size);
    }
}

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace nn;
using namespace System;

    class RomFsStorage : public nn::fs::IStorage
    {
    public:
        explicit RomFsStorage(int64_t bufferSize) NN_NOEXCEPT
            : m_BufferSize(bufferSize)
        {
            m_pBuffer = new unsigned char[static_cast<unsigned int>(bufferSize)];
            std::fill(m_pBuffer, &m_pBuffer[bufferSize], 0);
        }

        virtual ~RomFsStorage() NN_NOEXCEPT NN_OVERRIDE
        {
            delete m_pBuffer;
        }

        virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_SDK_REQUIRES_NOT_NULL(buffer);
            if( offset < 0 || m_BufferSize < static_cast<int64_t>(offset + size) )
            {
                return nn::fs::ResultOutOfRange();
            }

            memcpy(buffer, &m_pBuffer[offset], size);

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            if( offset < 0 || m_BufferSize < static_cast<int64_t>(offset + size) )
            {
                return nn::fs::ResultOutOfRange();
            }

            memcpy(&m_pBuffer[offset], buffer, size);

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = m_BufferSize;
            NN_RESULT_SUCCESS;
        }

    private:
        const int64_t m_BufferSize;
        unsigned char* m_pBuffer;
    };

    class DummyStorage : public nn::fs::IStorage
    {
    public:
        explicit DummyStorage(unsigned char* buffer, int64_t bufferSize) NN_NOEXCEPT
            : m_pBuffer(buffer)
            , m_BufferSize(bufferSize)
        {
        }

        virtual ~DummyStorage() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = m_BufferSize;
            NN_RESULT_SUCCESS;
        }

    private:
        unsigned char* const m_pBuffer;
        const int64_t m_BufferSize;
    };

    public ref class HierarchicalIntegrityVerificationStorage
    {
    public:
        nn::fssystem::save::HierarchicalIntegrityVerificationMetaInformation* m_pMetaInfo;
        nn::fssystem::save::HierarchicalIntegrityVerificationInformation* m_pInfoLevelHash;
        nn::fssystem::IntegrityRomFsStorage* m_pIntegrityRomFsStorage;
        nn::fssystem::FileSystemBufferManager* m_pBufferManager;
        nn::fs::SubStorage* m_pHashStorage;
        nn::fs::SubStorage* m_pBaseStorage;
        RomFsStorage* m_pHashStorageBuffer;
        nn::fs::IStorage* m_pBaseStorageBuffer;
        unsigned char* m_pCacheBuffer;

        static HierarchicalIntegrityVerificationStorage()
        {
            nn::fssystem::save::HierarchicalIntegrityVerificationStorage::SetGenerateRandomFunction(
                GenerateRandomForAuthoringTool
            );

            EnsureBufferPool();
        }

        HierarchicalIntegrityVerificationStorage()
            : m_pMetaInfo(nullptr)
            , m_pInfoLevelHash(nullptr)
            , m_pIntegrityRomFsStorage(nullptr)
            , m_pBufferManager(nullptr)
            , m_pHashStorage(nullptr)
            , m_pBaseStorage(nullptr)
            , m_pHashStorageBuffer(nullptr)
            , m_pBaseStorageBuffer(nullptr)
            , m_pCacheBuffer(nullptr)
        {
        }
        ~HierarchicalIntegrityVerificationStorage()
        {
            this->!HierarchicalIntegrityVerificationStorage();
        }
        !HierarchicalIntegrityVerificationStorage()
        {
            delete m_pMetaInfo;
            delete m_pInfoLevelHash;
            delete m_pIntegrityRomFsStorage;
            delete m_pHashStorage;
            delete m_pBaseStorage;
            delete m_pHashStorageBuffer;
            delete m_pBaseStorageBuffer;
        }
        void Initialize(int64_t sizeStorage)
        {
            Initialize(sizeStorage, nullptr, nullptr);
        }

        void Initialize(int64_t sizeStorage, array<Byte>^ hashData, array<Byte>^ masterHashData)
        {
            m_pIntegrityRomFsStorage = new nn::fssystem::IntegrityRomFsStorage();
            int layerCount = nn::fssystem::IntegrityLayerCountRomFs;

            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam inputParamIntegrity;
            for( int i = 0; i < layerCount - 1; ++i )
            {
                inputParamIntegrity.sizeBlockLevel[i] = nn::fssystem::IntegrityHashLayerBlockSize;
            }

            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea controlArea;

            nn::fssystem::save::HierarchicalIntegrityVerificationSizeSet sizes;
            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::QuerySize(
                &sizes,
                inputParamIntegrity,
                layerCount,
                sizeStorage
            );

            int64_t bufferSize = sizes.controlSize + sizes.masterHashSize;
            for( int i = 0; i < layerCount - 2; ++i )
            {
                bufferSize += sizes.layeredHashSizes[i];
            }
            m_pHashStorageBuffer = new RomFsStorage(bufferSize);
            m_pHashStorage = new nn::fs::SubStorage(m_pHashStorageBuffer, 0, bufferSize);
            m_pBaseStorageBuffer = new DummyStorage(nullptr, sizeStorage);
            m_pBaseStorage = new nn::fs::SubStorage(m_pBaseStorageBuffer, 0, sizeStorage);

            int64_t offsetHashInfo = 0;
            m_pMetaInfo = new nn::fssystem::save::HierarchicalIntegrityVerificationMetaInformation();
            m_pMetaInfo->Format();
            m_pMetaInfo->sizeMasterHash = static_cast<uint32_t>(sizes.masterHashSize);
            m_pMetaInfo->infoLevelHash.info[layerCount - 2].size = sizeStorage;
            m_pMetaInfo->infoLevelHash.maxLayers = layerCount;
            for( int level = 0; level < layerCount - 2; ++level )
            {
                const auto size = sizes.layeredHashSizes[level];
                m_pMetaInfo->infoLevelHash.info[level].offset = offsetHashInfo;
                m_pMetaInfo->infoLevelHash.info[level].size = size;
                m_pMetaInfo->infoLevelHash.info[level].orderBlock
                    = nn::fssystem::save::ILog2(
                        static_cast<uint32_t>(
                            inputParamIntegrity.sizeBlockLevel[level]
                        )
                    );
                offsetHashInfo += size;
            }
            m_pMetaInfo->infoLevelHash.info[layerCount - 2].offset = offsetHashInfo;
            m_pMetaInfo->infoLevelHash.info[layerCount - 2].orderBlock
                = nn::fssystem::save::ILog2(
                    static_cast<uint32_t>(
                        inputParamIntegrity.sizeBlockLevel[layerCount - 2]
                    )
                );

            nn::fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::Format(
                *m_pHashStorage,
                *m_pMetaInfo
            );

            m_pBufferManager = GetBufferManager();

            if( hashData != nullptr )
            {
                pin_ptr<unsigned char> ptr = &hashData[0];
                pin_ptr<unsigned char> ptrm = &masterHashData[0];
                m_pHashStorage->Write(sizeof(nn::fssystem::save::HierarchicalIntegrityVerificationMetaInformation), ptrm, static_cast<size_t>(m_pMetaInfo->sizeMasterHash));
                m_pHashStorage->Write(sizeof(nn::fssystem::save::HierarchicalIntegrityVerificationMetaInformation) + m_pMetaInfo->sizeMasterHash, ptr, static_cast<size_t>(hashData->Length));
                ptrm = nullptr;
                ptr = nullptr;
            }

            auto result = m_pIntegrityRomFsStorage->Initialize(*m_pHashStorage, *m_pBaseStorage, m_pBufferManager);
            if( result.IsFailure() )
            {
                throw gcnew ArgumentException(String::Format("Failed to initialize hierarchical integrity storage 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            controlArea.Initialize(*m_pHashStorage);

            m_pInfoLevelHash = new nn::fssystem::save::HierarchicalIntegrityVerificationInformation();
            controlArea.GetLevelHashInfo(m_pInfoLevelHash);

            NN_SDK_ASSERT(sizeStorage == GetBodySize());

            GC::KeepAlive(hashData);
        }

        void Read(int64_t offset, array<Byte>^ data)
        {
            pin_ptr<unsigned char> ptr = &data[0];
            auto result = m_pIntegrityRomFsStorage->Read(offset, ptr, data->GetLength(0));
            ptr = nullptr;
            if( result.IsFailure() )
            {
                throw gcnew ArgumentException(String::Format("Failed to read hierarchical integrity storage 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
        }

        void Write(int64_t offset, array<Byte>^ data)
        {
            pin_ptr<unsigned char> ptr = &data[0];
            auto result = m_pIntegrityRomFsStorage->Write(offset, ptr, data->GetLength(0));
            ptr = nullptr;
            if( result.IsFailure() )
            {
                throw gcnew ArgumentException(String::Format("Failed to write hierarchical integrity storage 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
        }

        void Commit()
        {
            auto result = m_pIntegrityRomFsStorage->Commit();
            if( result.IsFailure() )
            {
                throw gcnew ArgumentException(String::Format("Failed to commit hierarchical integrity storage 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
        }

        static int32_t GetHashBlockSize()
        {
            return nn::fssystem::IntegrityHashLayerBlockSize;
        }

        static int32_t GetMetaInfoSize()
        {
            return sizeof(nn::fssystem::save::HierarchicalIntegrityVerificationMetaInformation);
        }

        uint32_t GetMasterHashSize()
        {
            return m_pMetaInfo->sizeMasterHash;
        }

        int64_t GetLayerHashSize()
        {
            return m_pInfoLevelHash->GetDataOffset();
        }

        int64_t GetBodySize()
        {
            return m_pInfoLevelHash->GetDataSize();
        }

        array<Byte>^ GetMetaInfo()
        {
            int64_t size = sizeof(nn::fssystem::save::HierarchicalIntegrityVerificationMetaInformation);

            array<unsigned char>^ buf = gcnew array<unsigned char>(static_cast<int>(size));
            {
                pin_ptr<unsigned char> ptr = &buf[0];
                m_pHashStorage->Read(0, ptr, static_cast<size_t>(size));
                ptr = nullptr;
            }
            return buf;
        }

        array<Byte>^ GetMasterHash()
        {
            int64_t size = m_pMetaInfo->sizeMasterHash;
            int64_t offset = sizeof(nn::fssystem::save::HierarchicalIntegrityVerificationMetaInformation);

            array<unsigned char>^ buf = gcnew array<unsigned char>(static_cast<int>(size));
            {
                pin_ptr<unsigned char> ptr = &buf[0];
                m_pHashStorage->Read(offset, ptr, static_cast<size_t>(size));
                ptr = nullptr;
            }
            return buf;
        }

        array<Byte>^ GetLayerHash(int64_t offset, int64_t size)
        {
            int64_t pullSize = ((offset + size) > m_pInfoLevelHash->GetDataOffset()) ? m_pInfoLevelHash->GetDataOffset() - offset : size;
            int64_t pullOffset = sizeof(nn::fssystem::save::HierarchicalIntegrityVerificationMetaInformation) + m_pMetaInfo->sizeMasterHash + offset;

            array<unsigned char>^ buf = gcnew array<unsigned char>(static_cast<int>(pullSize));
            {
                pin_ptr<unsigned char> ptr = &buf[0];
                m_pHashStorage->Read(pullOffset, ptr, static_cast<size_t>(pullSize));
                ptr = nullptr;
            }
            return buf;
        }
    };

}}}
