﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <map>
#include <nn/nn_Allocator.h>
#include <nn/nn_Common.h>

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

    private class DefaultMemoryResource : public nn::MemoryResource
    {
    protected:
        virtual void* do_allocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(alignment);
            return std::malloc(bytes);
        }
        virtual void do_deallocate(void* p, std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(bytes);
            NN_UNUSED(alignment);
            return std::free(p);
        }
        virtual bool do_is_equal(const nn::MemoryResource& other) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(other);
            return false;
        }
    };

    private class DefaultAlignedMemoryResource : public nn::MemoryResource
    {
        NN_DISALLOW_COPY(DefaultAlignedMemoryResource);
    public:
        DefaultAlignedMemoryResource() NN_NOEXCEPT : m_MemoryMap() {}

    protected:
        virtual void* do_allocate(size_t size, size_t alignment) NN_NOEXCEPT NN_OVERRIDE
        {
            Memory memory(new char[size + alignment]);
            if( memory != nullptr )
            {
                void* ptr = reinterpret_cast<void*>(nn::util::align_up(
                                reinterpret_cast<uintptr_t>(memory.get()), alignment));
                if( m_MemoryMap.emplace(ptr, std::move(memory)).second )
                {
                    return ptr;
                }
            }
            return nullptr;
        }

        virtual void do_deallocate(void* ptr, size_t size, size_t alignment) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(size);
            NN_UNUSED(alignment);
            m_MemoryMap.erase(ptr);
        }

        virtual bool do_is_equal(const MemoryResource& other) const NN_NOEXCEPT NN_OVERRIDE
        {
            return this == &other;
        }

    private:
        typedef std::unique_ptr<char[]> Memory;

    private:
        std::map<void*, Memory> m_MemoryMap;
    };

}}}
