﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <vector>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/utilTool/fs_AesCtrCounterExtendedStorageBuilder.h>
#include "../Util/DeclareAlive.h"
#include "AesCtrExGenerationTable.h"
#include "DefaultMemoryResource.h"

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::IO;
using namespace System::Runtime::InteropServices;

using namespace nn;
using namespace nn::fssystem;

// AesCtrEx の世代テーブル
private class AesCtrExGenerationTableImpl
{
public:
    typedef AesCtrExGenerationTable::GenerationEntry GenerationEntry;

public:
    AesCtrExGenerationTableImpl()
        : m_IsInitialized(false)
        , m_Allocator()
        , m_TableBuffer()
        , m_pTableStorage()
        , m_BucketTree()
    {
    }

    ~AesCtrExGenerationTableImpl()
    {
        Finalize();
    }

    void Initialize(array<Byte>^ pHeaderBuffer, array<Byte>^ pTableBuffer)
    {
        if (m_IsInitialized)
        {
            throw gcnew Exception("already initialized.");
        }

        m_TableBuffer.resize(pTableBuffer->Length);
        Marshal::Copy(pTableBuffer, 0, IntPtr(&m_TableBuffer[0]), m_TableBuffer.size());

        m_pTableStorage.reset(new fs::MemoryStorage(&m_TableBuffer[0], m_TableBuffer.size()));

        utilTool::AesCtrCounterExtendedStorageBuilder::TableInfo info;
        {
            pin_ptr<unsigned char> pinHeaderBuffer = &pHeaderBuffer[0];
            auto result = info.Initialize(
                pinHeaderBuffer,
                pHeaderBuffer->Length,
                m_pTableStorage.get());
            pinHeaderBuffer = nullptr;

            if (result.IsFailure())
            {
                throw gcnew Exception(
                    String::Format(
                        "Failed to ExtractSubStoragesForAesCtrCounterExtendedStorage 0x{0:X8}.",
                        result.GetInnerValueForDebug()));
            }
        }

        {
            auto result = m_BucketTree.Initialize(
                &m_Allocator,
                info.GetNodeStorage(),
                info.GetEntryStorage(),
                AesCtrCounterExtendedStorage::NodeSize,
                sizeof(AesCtrCounterExtendedStorage::Entry),
                info.GetEntryCount());
            if (result.IsFailure())
            {
                throw gcnew System::Exception(
                    String::Format("Failed to Initialize BucketTree 0x{0:X8}.",
                                   result.GetInnerValueForDebug()));
            }
        }
        m_IsInitialized = true;
    }

    void Finalize()
    {
        if (m_IsInitialized == false)
        {
            return;
        }

        m_BucketTree.Finalize();
        m_pTableStorage.reset();
        m_TableBuffer.clear();
    }

    Collections::Generic::List<GenerationEntry>^ GetGenerations(Int64 offset, Int32 size,
                                                   UInt32 generationForOutOfTable) const
    {
        typedef AesCtrCounterExtendedStorage::Entry Entry;

        auto pEntries = gcnew Collections::Generic::List<GenerationEntry>();

        const auto searchBegin = std::max<int64_t>(m_BucketTree.GetBegin(), offset);
        const auto searchEnd = std::min<int64_t>(m_BucketTree.GetEnd(), offset + size);

        // 範囲外について追加(負のほうについて)
        {
            const auto outOfTableSize = std::min<int64_t>(searchBegin - offset, size);
            if (outOfTableSize > 0)
            {
                pEntries->Add(GenerationEntry(Int32(outOfTableSize), generationForOutOfTable));
            }
        }

        // テーブルを探索
        {
            BucketTree::Visitor visitor;
            m_BucketTree.Find(&visitor, searchBegin);

            while(NN_STATIC_CONDITION(true))
            {
                const auto entry = *(visitor.Get<Entry>());
                const auto entryOffset = std::max<int64_t>(entry.GetOffset(), searchBegin);

                if (entryOffset >= searchEnd)
                {
                    // 探索終了
                    break;
                }

                if (visitor.CanMoveNext() == false)
                {
                    // テーブルの最後まで来た
                    // 探索終了

                    // 最後の項目を追加
                    pEntries->Add(GenerationEntry(
                                      Int32(searchEnd - entryOffset),
                                      entry.generation));

                    break;
                }

                // 次の項目を取得 (visitor も移動する)
                int64_t nextEntryOffset;
                {
                    auto result = visitor.MoveNext();
                    if (result.IsFailure())
                    {
                        throw gcnew Exception(String::Format("visitor.MoveNext() failed.0x{0:X8}.", result.GetInnerValueForDebug()));
                    }
                    const auto& nextEntry = *(visitor.Get<Entry>());
                    nextEntryOffset = nextEntry.GetOffset();
                }

                // 項目を追加
                pEntries->Add(GenerationEntry(
                                  Int32(std::min<int64_t>(nextEntryOffset, searchEnd) - entryOffset),
                                  entry.generation));
            }
        }

        // 範囲外について追加(正のほうについて)
        {
            const auto outOfTableSize = std::min<int64_t>((offset + size) - searchEnd, size);
            if (outOfTableSize > 0)
            {
                pEntries->Add(GenerationEntry( Int32(outOfTableSize), generationForOutOfTable));
            }
        }

        return pEntries;
    }

private:
    bool m_IsInitialized;
    DefaultAlignedMemoryResource m_Allocator;
    std::vector<unsigned char> m_TableBuffer;
    std::unique_ptr<nn::fs::MemoryStorage> m_pTableStorage;
    BucketTree m_BucketTree;
};


AesCtrExGenerationTable::AesCtrExGenerationTable(array<Byte>^ headerBuffer, array<Byte>^ tableBuffer)
{
    m_pImpl = new AesCtrExGenerationTableImpl();
    m_pImpl->Initialize(headerBuffer, tableBuffer);
    GC::KeepAlive(this);
}

AesCtrExGenerationTable::~AesCtrExGenerationTable()
{
    this->!AesCtrExGenerationTable();
}

AesCtrExGenerationTable::!AesCtrExGenerationTable()
{
    m_pImpl->Finalize();
    delete m_pImpl;
}

Collections::Generic::List<AesCtrExGenerationTable::GenerationEntry>^ AesCtrExGenerationTable::GetGenerations(Int64 offset, Int32 size, UInt32 generationForOutOfTable)
{
    return Util::ReturnAndDeclareAlive(this, m_pImpl->GetGenerations(offset, size, generationForOutOfTable));
}

}}}
