﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/crypto/crypto_Aes128CbcEncryptor.h>
#include <nn/crypto/crypto_Aes128CbcDecryptor.h>
#include "IEncryptor.h"

namespace Nintendo { namespace Authoring { namespace CryptoLibrary {

using namespace System;
using namespace System::Security::Cryptography;

    public ref class Aes128CbcCryptoDriver : ICbcModeEncryptor
    {
    public:
        property int KeySize
        {
            virtual int get() { return 16; }
        };

        Aes128CbcCryptoDriver(array<Byte>^ key)
        {
            m_Key = key;
        }

        Aes128CbcCryptoDriver(Aes128KeyIndex keyIndex) : Aes128CbcCryptoDriver(keyIndex, 0, false)
        {
        }

        Aes128CbcCryptoDriver(Aes128KeyIndex keyIndex, bool useDev) : Aes128CbcCryptoDriver(keyIndex, 0, useDev)
        {
        }

        Aes128CbcCryptoDriver(Aes128KeyIndex keyIndex, Byte keyGeneration) : Aes128CbcCryptoDriver(keyIndex, keyGeneration, false)
        {
        }

        Aes128CbcCryptoDriver(Aes128KeyIndex keyIndex, Byte keyGeneration, bool useDev)
        {
            switch (keyIndex)
            {
            case Aes128KeyIndex::XciHeader:
                m_Key = useDev ? XciHeaderKeyForDev : XciHeaderKeyForProd;
                break;
            case Aes128KeyIndex::ETicketCommonKey:
                if (keyGeneration <= 1)
                {
                    m_Key = TicketCommonKey;
                }
                else if (keyGeneration == 2)
                {
                    m_Key = TicketCommonKeyGen2;
                }
                else if (keyGeneration == 3)
                {
                    m_Key = TicketCommonKeyGen3;
                }
                else if (keyGeneration == 4)
                {
                    m_Key = TicketCommonKeyGen4;
                }
                else if (keyGeneration == 5)
                {
                    m_Key = TicketCommonKeyGen5;
                }
                else
                {
                    m_Key = gcnew array<Byte>(16);
                    auto rng = gcnew RNGCryptoServiceProvider();
                    rng->GetBytes(m_Key);
                    delete rng;
                }
                break;
            default:
                throw gcnew ArgumentException("Invalid KeyIndex.");
            }
            GC::KeepAlive(this);
        }

        ~Aes128CbcCryptoDriver()
        {
            this->!Aes128CbcCryptoDriver();
        }

        !Aes128CbcCryptoDriver()
        {
        }

        virtual void EncryptBlock(array<Byte>^ iv, array<Byte>^ src, int srcOffset, int srcSize, array<Byte>^ dst, int dstOffset)
        {
            pin_ptr<unsigned char> pinKey = &m_Key[0];
            pin_ptr<unsigned char> pinSrc = &src[srcOffset];
            pin_ptr<unsigned char> pinIv = &iv[0];
            pin_ptr<unsigned char> pinDst = &dst[dstOffset];
            auto encryptedSize = nn::crypto::EncryptAes128Cbc(pinDst, srcSize, pinKey, KeySize, pinIv, KeySize, pinSrc, srcSize);
            pinKey = nullptr;
            pinSrc = nullptr;
            pinIv = nullptr;
            pinDst = nullptr;
            if (encryptedSize != srcSize)
            {
                throw gcnew InvalidOperationException();
            }
            GC::KeepAlive(this);
        }

        virtual void DecryptBlock(array<Byte>^ iv, array<Byte>^ src, int srcOffset, int srcSize, array<Byte>^ dst, int dstOffset)
        {
            pin_ptr<unsigned char> pinKey = &m_Key[0];
            pin_ptr<unsigned char> pinSrc = &src[srcOffset];
            pin_ptr<unsigned char> pinIv = &iv[0];
            pin_ptr<unsigned char> pinDst = &dst[dstOffset];
            auto encryptedSize = nn::crypto::DecryptAes128Cbc(pinDst, srcSize, pinKey, KeySize, pinIv, KeySize, pinSrc, srcSize);
            pinKey = nullptr;
            pinSrc = nullptr;
            pinIv = nullptr;
            pinDst = nullptr;
            if (encryptedSize != srcSize)
            {
                throw gcnew InvalidOperationException();
            }
            GC::KeepAlive(this);
        }

    private:
        array<Byte>^ m_Key;
        static array<Byte>^ XciHeaderKeyForProd =
        {
            0x01, 0xc5, 0x8f, 0xe7, 0x00, 0x2d, 0x13, 0x5a, 0xb2, 0x9a, 0x3f, 0x69, 0x33, 0x95, 0x74, 0xb1
        };
        static array<Byte>^ XciHeaderKeyForDev =
        {
            0xcb, 0xa7, 0xb8, 0x75, 0xeb, 0x67, 0x05, 0xfb, 0x46, 0x0a, 0x33, 0xfd, 0x34, 0x09, 0x13, 0xb4
        };
        static array<Byte>^ TicketCommonKey =
        {
            0xc4, 0x67, 0xd5, 0x3f, 0x4f, 0xdf, 0xeb, 0x19, 0x3f, 0x61, 0xbf, 0xda, 0x57, 0xf4, 0x89, 0xd0
        };
        static array<Byte>^ TicketCommonKeyGen2 =
        {
            0xb0, 0xea, 0xe3, 0x4f, 0x4e, 0x27, 0x08, 0x55, 0xb3, 0x24, 0x01, 0x65, 0x1d, 0x8a, 0xf6, 0x95
        };
        static array<Byte>^ TicketCommonKeyGen3 =
        {
            0x90, 0xe4, 0xf2, 0x2b, 0x9b, 0x9a, 0x46, 0xe3, 0x07, 0x4d, 0xa6, 0xd0, 0xae, 0x14, 0x9f, 0x8a
        };
        static array<Byte>^ TicketCommonKeyGen4 =
        {
            0xb2, 0x5a, 0x2b, 0xaf, 0x87, 0xa5, 0xb8, 0x3b, 0x4e, 0x5e, 0xd4, 0xc3, 0x25, 0x4b, 0x5c, 0xdc
        };
        static array<Byte>^ TicketCommonKeyGen5 =
        {
            0x7b, 0x3c, 0x64, 0xa2, 0xe9, 0x49, 0xf8, 0xdc, 0x8f, 0x10, 0xed, 0x85, 0x59, 0xf3, 0x18, 0x9a
        };
    };

}}}
