﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using YamlDotNet.RepresentationModel;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    using EntryFilterRule = Pair<FilterType, string>;
    using EntryFilterRuleRegex = Pair<FilterType, System.Text.RegularExpressions.Regex>;
    using DirectoryConnector = Pair<string, string>;
    using RomFsListFileInfo = Tuple<string, Int64>;

    public class RomFsAdfWriter
    {
        public const int AlignmentSize = 16; // offset のアライメント

        private string m_adfPath;
        private bool m_IsAdfWritten;

        public RomFsAdfWriter(string adfPath)
        {
            m_adfPath = adfPath;
            m_IsAdfWritten = false;
        }
        public void Write(string dirPath, List<Tuple<string, Int64>> originalRomFsListFileInfo = null)
        {
            var listDirPath = new List<DirectoryConnector>() { new DirectoryConnector(dirPath, null) };
            Write(listDirPath, null, originalRomFsListFileInfo);
        }
        public void Write(string dirPath, List<EntryFilterRule> filterRules, List<Tuple<string, Int64>> originalRomFsListFileInfo = null)
        {
            var listDirPath = new List<DirectoryConnector>() { new DirectoryConnector(dirPath, null) };
            Write(listDirPath, filterRules, originalRomFsListFileInfo);
        }
        public void Write(DirectoryConnector dirPath, List<EntryFilterRule> filterRules, List<Tuple<string, Int64>> originalRomFsListFileInfo = null)
        {
            var listDirPath = new List<DirectoryConnector>() { dirPath };
            Write(listDirPath, filterRules, originalRomFsListFileInfo);
        }

        private class RomFsListFileInfoComparer : IComparer<RomFsListFileInfo>
        {
            public int Compare(RomFsListFileInfo s1, RomFsListFileInfo s2)
            {
                return string.CompareOrdinal(s1.Item1, s2.Item1);
            }
        }

        public static List<string> GetTargetFileList(List<EntryFilterRule> filterRuleList, DirectoryConnector dirPath, SearchOption searchOption)
        {
            var targetFileList = Directory.EnumerateFiles(dirPath.first, "*", searchOption).ToList();
            targetFileList.Sort();
            // 0 番目が AbsoluteAdd でないということは、AbsoluteAdd がないということ
            if (filterRuleList == null || filterRuleList[0].first != FilterType.AbsoluteAdd)
            {
                return targetFileList;
            }
            // AbsoluteAdd がある場合は母集団は空からスタート
            var addedFileList = new List<string>();
            foreach (var filter in filterRuleList)
            {
                if (filter.first == FilterType.AbsoluteAdd)
                {
                    var absoluteAddFilelist = new List<string>();
                    string filePath = dirPath.first + "\\" + filter.second.Replace("/", "\\");
                    // Directory フラグが立っている場合は Directory 判定する。パスが存在していなければここで例外で止まる。
                    if ((File.GetAttributes(filePath) & FileAttributes.Directory) == FileAttributes.Directory)
                    {
                        absoluteAddFilelist = Directory.EnumerateFiles(filePath, "*", searchOption).ToList();
                    }
                    // それ以外はファイル
                    else
                    {
                        absoluteAddFilelist.Add(filePath);
                    }
                    foreach (var file in absoluteAddFilelist)
                    {
                        // 一応 targetFileList 内に同ファイルが存在していることを確認する
                        var index = targetFileList.BinarySearch(file);
                        System.Diagnostics.Trace.Assert(index >= 0);
                        addedFileList.Add(file);
                    }
                }
            }

            filterRuleList.RemoveAll(filter => filter.first == FilterType.AbsoluteAdd);
            return addedFileList;
        }

        public void Write(List<DirectoryConnector> dirPaths, List<EntryFilterRule> filterRules, List<Tuple<string, Int64>> originalRomFsListFileInfo = null)
        {
            if (m_IsAdfWritten)
            {
                throw new InvalidOperationException("RomFs Adf can be written only once.");
            }

            var memoryStream = new MemoryStream();
            using (var adf = new StreamWriter(memoryStream, Encoding.UTF8))
            {
                // ヘッダー部分の出力
                {
                    adf.WriteLine("formatType : {0}", NintendoContentFileSystemMetaConstant.FormatTypeRomFs);
                    adf.WriteLine("version : 0");
                    adf.WriteLine("entries :");
                }

                // ファイルリストを RomFs 上のパス順でソート
                var functionGetPathName = new Func<Pair<DirectoryConnector, string>, string>(delegate (Pair<DirectoryConnector, string> file)
                {
                    var dirPath = file.first;
                    var fileName = file.second;

                    string pathName = "";
                    if (dirPath.second != null)
                    {
                        pathName = dirPath.second.Replace("\\", "/") + "/";
                    }
                    var romFsPath = pathName + fileName.Replace("\\", "/").Replace(dirPath.first.Replace("\\", "/") + "/", string.Empty);
                    return romFsPath;
                });

                var functionGetDirName = new Func<string, string>(delegate (string path)
                {
                    var dirName = System.IO.Path.GetDirectoryName(path);
                    if (dirName == null)
                    {
                        return "";
                    }
                    return dirName.Replace("\\", "/");
                });

                var createDirList = new List<string>();
                // ディレクトリーを巡回し、ファイルリストを取得
                var listFiles = new List<Pair<DirectoryConnector, string>>();
                foreach (var dirPath in dirPaths)
                {
                    var targetFileNameList = GetTargetFileList(filterRules, dirPath, SearchOption.AllDirectories);
                    var filterRuleRegexList = FilterDescription.ConvertFilterRuleStringToRegex(filterRules);
                    foreach (var file in targetFileNameList)
                    {
                        if (filterRules != null && FilterDescription.IsEntryInFilterList(file, dirPath.first, filterRuleRegexList))
                        {
                            continue;
                        }
                        var pairPath = new Pair<DirectoryConnector, string>(dirPath, file);
                        listFiles.Add(pairPath);

                        if (GlobalSettings.NoCheckDirWarning == false)
                        {
                            for (var relPath = functionGetDirName(functionGetPathName(pairPath)); relPath != ""; relPath = functionGetDirName(relPath))
                            {
                                var path = dirPath.first + "/" + relPath;
                                if (createDirList.Contains(path))
                                {
                                    continue;
                                }

                                createDirList.Add(path);
                            }
                        }
                    }
                }

                if (GlobalSettings.NoCheckDirWarning == false)
                {
                    // RomFs に作成されないディレクトリーを検出
                    foreach (var dirPath in dirPaths)
                    {
                        var allDirList = Directory.EnumerateDirectories(dirPath.first, "*", SearchOption.AllDirectories).ToList();
                        foreach (var dir in allDirList)
                        {
                            var path = dir.Replace("\\", "/");
                            if (createDirList.Contains(path))
                            {
                                continue;
                            }

                            Log.Warning(string.Format("'{0}' is not archived because it is empty or filtered.", path));
                        }
                    }
                }

                listFiles.Sort(delegate (Pair<DirectoryConnector, string> fileLeft, Pair<DirectoryConnector, string> fileRight)
                {
                    return string.CompareOrdinal(
                        functionGetPathName(fileLeft),
                        functionGetPathName(fileRight)
                        );
                });

                if (originalRomFsListFileInfo != null)
                {
                    originalRomFsListFileInfo.Sort(new RomFsListFileInfoComparer());
                }

                // ファイルリストを出力
                long offset = 0;
                int foundIndex = 0;
                foreach (var file in listFiles)
                {
                    var dirPath = file.first;
                    var fileName = file.second;

                    adf.WriteLine("  - type : file");
                    adf.WriteLine("    name : \"{0}\"", functionGetPathName(file));

                    // 元 Rom が指定されている場合は同じファイル名のファイルオフセットのアラインメント mod 512 に調整
                    if (originalRomFsListFileInfo != null && foundIndex < originalRomFsListFileInfo.Count)
                    {
                        var target = Tuple.Create(functionGetPathName(file), offset);
                        var index = originalRomFsListFileInfo.BinarySearch(foundIndex, originalRomFsListFileInfo.Count - foundIndex, target, new RomFsListFileInfoComparer());
                        if (index >= 0)
                        {
                            foundIndex = index;
                            const long originalAlignmentSizeModulus = 512;
                            var originalAlignmentDiff = (int)(originalRomFsListFileInfo[index].Item2 % originalAlignmentSizeModulus);
                            var currentAlignmentDiff = (int)(offset % originalAlignmentSizeModulus);
                            offset += (originalAlignmentDiff >= currentAlignmentDiff) ? (originalAlignmentDiff - currentAlignmentDiff) : (originalAlignmentDiff + originalAlignmentSizeModulus - currentAlignmentDiff);
                        }
                    }

                    adf.WriteLine("    offset : {0}", offset);
                    FileInfo fi = new FileInfo(fileName);

                    offset += fi.Length;
                    offset = BitUtility.AlignUp(offset, AlignmentSize);

                    adf.WriteLine("    path : {0}", Path.GetFullPath(fileName.Replace("\\", "/")));
                }
            }

            // ファイルに出力
            {
                FileMode fileModeAdf;
                if (File.Exists(m_adfPath))
                {
                    fileModeAdf = FileMode.Truncate;
                }
                else
                {
                    fileModeAdf = FileMode.CreateNew;
                }

                using (var adf = File.Open(m_adfPath, fileModeAdf, FileAccess.Write, FileShare.None))
                {
                    var bufferMemoryStream = memoryStream.ToArray();
                    adf.Write(bufferMemoryStream, 0, bufferMemoryStream.Length);
                    m_IsAdfWritten = true;
                }
            }
        }
    }

    public class DirectoryList
    {
        public void AddAncestors(string directoryPath)
        {
            if (directoryPath.Contains("\\"))
            {
                throw new Exception("invalid directory path: " + directoryPath);
            }

            var parent = directoryPath;
            while (true)
            {
                if (parent == null ||
                    parent == string.Empty ||
                    // 既に追加済み
                    !directories.Add(parent))
                {
                    // これ以上の親ディレクトリは既存
                    return;
                }
                parent = RomFsFileSystemInfo.EntryInfo.GetDirectoryOnRomFs(parent);
            }
        }

        public void CalculateFileSystemDirectoryInfo(RomFsFileSystemInfo info)
        {
            info.directoryEntryCount = Count;
            info.directoryEntryBytes = TotalBytes;
        }

        private int Count
        {
            get { return directories.Count; }
        }

        private int TotalBytes
        {
            get
            {
                int totalBytes = 0;
                foreach (var dir in directories)
                {
                    int length = Encoding.UTF8.GetByteCount(RomFsFileSystemInfo.EntryInfo.GetFileNameOnRomFs(dir));
                    totalBytes += RomFsFileSystemInfo.GetAlignedDirectoryEntryBytes(length);
                }
                return totalBytes;
            }
        }

        private HashSet<string> directories = new HashSet<string>();
    }

    public class RomFsAdfReader
    {
        private string m_adfPath;

        public RomFsAdfReader(string adfPath)
        {
            m_adfPath = adfPath;
        }

        public RomFsFileSystemInfo GetFileSystemInfo()
        {
            RomFsFileSystemInfo fileSystemInfo = new RomFsFileSystemInfo();

            using (var adf = new StreamReader(m_adfPath, Encoding.UTF8))
            {
                var yamlStream = new YamlStream();
                yamlStream.Load(adf);

                YamlMappingNode rootNode;
                try
                {
                    rootNode = (YamlMappingNode)yamlStream.Documents[0].RootNode;
                    YamlScalarNode formatType = (YamlScalarNode)rootNode.Children[new YamlScalarNode("formatType")];
                    if (formatType.Value != NintendoContentFileSystemMetaConstant.FormatTypeRomFs)
                    {
                        throw new ArgumentException();
                    }
                }
                catch
                {
                    throw new ArgumentException("invalid format .adf file.");
                }

                // エントリ情報の読み込み
                YamlSequenceNode entries;
                try
                {
                    entries = (YamlSequenceNode)rootNode.Children[new YamlScalarNode("entries")];
                }
                catch
                {
                    throw new ArgumentException("invalid format .adf file. At least one file must be included in the data region.");
                }

                var directories = new DirectoryList();

                foreach (YamlMappingNode entry in entries)
                {
                    bool isSizeSpecified = false;

                    var entryInfo = new RomFsFileSystemInfo.EntryInfo();
                    foreach (var child in entry)
                    {
                        switch (((YamlScalarNode)child.Key).Value)
                        {
                            case "type":
                                entryInfo.type = ((YamlScalarNode)child.Value).Value;
                                break;
                            case "name":
                                entryInfo.name = "/" + ((YamlScalarNode)child.Value).Value;
                                break;
                            case "offset":
                                entryInfo.offset = Convert.ToUInt64(((YamlScalarNode)child.Value).Value);
                                break;
                            case "path":
                                entryInfo.path = ((YamlScalarNode)child.Value).Value;
                                break;
                            case "specifiedsize":
                                entryInfo.size = Convert.ToUInt64(((YamlScalarNode)child.Value).Value);
                                isSizeSpecified = true;
                                break;
                            case "sourceType":
                                entryInfo.sourceType = ((YamlScalarNode)child.Value).Value;
                                break;
                            default:
                                throw new ArgumentException("invalid format .adf file. invalid key is specified\n" + entry.ToString());
                        }
                    }

                    if (entryInfo.type == null)
                    {
                        throw new ArgumentException("invalid format .adf file. invalid \"type\"\n" + entry.ToString());
                    }
                    if (entryInfo.type == "file" && (entryInfo.name == null || entryInfo.path == null))
                    {
                        throw new ArgumentException("invalid format .adf file. \"type == file\" but \"name\" or \"path\" is not specified.\n" + entry.ToString());
                    }
                    if (entryInfo.type == "directory" && (entryInfo.name == null))
                    {
                        throw new ArgumentException("invalid format .adf file. \"type == directory\" but \"name\" is not specified.\n" + entry.ToString());
                    }

                    if (entryInfo.type == "file")
                    {
                        if (!isSizeSpecified)
                        {
                            FileInfo fi = new FileInfo(entryInfo.path);
                            entryInfo.size = (ulong)fi.Length;
                        }
                        fileSystemInfo.AddFileEntry(entryInfo);

                        // 新出のディレクトリをカウント
                        directories.AddAncestors(entryInfo.DirectoryOfEntryName);
                    }
                    else if (entryInfo.type == "directory")
                    {
                        fileSystemInfo.AddDirectoryEntry(entryInfo);

                        directories.AddAncestors(entryInfo.name);
                    }
                    else
                    {
                        throw new ArgumentException("invalid format .adf file. \"type\" should be \"directory\" or \"file\".\n" + entry.ToString());
                    }
                }

                directories.CalculateFileSystemDirectoryInfo(fileSystemInfo);
            }

            return fileSystemInfo;
        }
    }

    public class RomFsArchiveSource : ISource
    {
        public long Size { get; private set; }

        private ISource m_source;

        public RomFsArchiveSource(RomFsFileSystemInfo fileSystemInfo)
        {
            RomFsFileSystemMeta metaMgr = new RomFsFileSystemMeta();
            List<ConcatenatedSource.Element> elements = new List<ConcatenatedSource.Element>();
            long m_metaOffset;

            // 一括でメタデータを作成
            var metaInfo = metaMgr.Create(fileSystemInfo);

            // ヘッダーを先頭に配置
            {
                ConcatenatedSource.Element element = new ConcatenatedSource.Element(
                    new MemorySource(metaInfo.header, 0, metaInfo.header.Length), "meta_header", 0);
                elements.Add(element);
                m_metaOffset = element.Source.Size;
            }

            // ファイル実データを並べる
            foreach (var entryInfo in fileSystemInfo.entries)
            {
                if (entryInfo.type == "file")
                {
                    ISource fsource;
                    if (entryInfo.sourceType == "padding")
                    {
                        fsource = new DebugPaddingSource((long)entryInfo.size, entryInfo.name);
                    }
                    else
                    {
                        fsource = new FileSource(entryInfo.path, 0, (long)entryInfo.size);
                    }
                    ConcatenatedSource.Element element = new ConcatenatedSource.Element(fsource, entryInfo.name, (long)entryInfo.offset + m_metaOffset);
                    elements.Add(element);
                }
                else if (entryInfo.type == "source")
                {
                    ConcatenatedSource.Element element = new ConcatenatedSource.Element(
                        (ISource)entryInfo.sourceInterface, entryInfo.name, (long)entryInfo.offset + m_metaOffset);
                    elements.Add(element);
                }
            }

            // ファイル実データ～メタデータ間にパディングを追加
            {
                var lastEntry = fileSystemInfo.entries[fileSystemInfo.entries.Count - 1];
                var fileBodyEnd = m_metaOffset + (long)lastEntry.offset + (long)lastEntry.size;
                if (metaInfo.offsetData > fileBodyEnd)
                {
                    elements.Add(new ConcatenatedSource.Element(
                        new PaddingSource(metaInfo.offsetData - fileBodyEnd),
                        "romFsMetaBodyPadding",
                        fileBodyEnd
                        ));
                }
            }

            // メタの本体データを末尾に配置
            {
                ConcatenatedSource.Element element = new ConcatenatedSource.Element(
                    new MemorySource(metaInfo.data, 0, metaInfo.data.Length), "meta_data", metaInfo.offsetData);
                elements.Add(element);
            }

            m_source = new ConcatenatedSource(elements);
            Size = m_source.Size;
        }
        public ByteData PullData(long offset, int size)
        {
            return m_source.PullData(offset, size);
        }
        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
    }
}
