﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml.Serialization;
using YamlDotNet.RepresentationModel;
using Nintendo.Authoring.FileSystemMetaLibrary;
using Nintendo.Authoring.CryptoLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    internal class ProdEncryptionUtils
    {
        public static bool IsContentTypeEncryptedByExternalKey(string contentType)
        {
            return contentType == NintendoContentMetaConstant.ContentTypeProgram ||
                   contentType == NintendoContentMetaConstant.ContentTypeHtmlDocument ||
                   contentType == NintendoContentMetaConstant.ContentTypePublicData || // ここには来ないはずだが念のため
                   contentType == NintendoContentMetaConstant.ContentTypeData;
        }

        public static void SetRightsIdAndKeyGeneration(ref NintendoSubmissionPackageFileSystemInfo prodNspInfo, byte desiredSafeKeyGeneration, long sizeThresholdForKeyGenerationUpdate)
        {
            SetRightsIdAndKeyGeneration(ref prodNspInfo, new Dictionary<UInt64, byte>(), desiredSafeKeyGeneration, sizeThresholdForKeyGenerationUpdate);
        }

        public static void SetRightsIdAndKeyGeneration(ref NintendoSubmissionPackageFileSystemInfo prodNspInfo, Dictionary<UInt64, byte> keyGenerationMinList, byte desiredSafeKeyGeneration, long sizeThresholdForKeyGenerationUpdate)
        {
            for (int i = 0; i < prodNspInfo.Entries.Count; i++)
            {
                var entry = prodNspInfo.Entries[i];
                var contentMetaId = entry.ContentMetaInfo.Model.GetUInt64Id();
                if (entry.MetaType == NintendoContentMetaConstant.ContentMetaTypeApplication ||
                    entry.MetaType == NintendoContentMetaConstant.ContentMetaTypePatch ||
                    entry.MetaType == NintendoContentMetaConstant.ContentMetaTypeAddOnContent)
                {
                    // contentMetaId 毎に外部鍵で暗号化されるコンテンツのうち最新の鍵世代を記憶
                    byte keyGeneration = 0;
                    for (int j = 0; j < entry.Contents.Count; j++)
                    {
                        var content = entry.Contents[j];
                        if (IsContentTypeEncryptedByExternalKey(content.ContentType))
                        {
                            if (keyGeneration < (content.FsInfo as NintendoContentFileSystemInfo).keyGeneration)
                            {
                                keyGeneration = (content.FsInfo as NintendoContentFileSystemInfo).keyGeneration;
                            }
                        }
                    }

                    // 使用する鍵世代が指定されている場合はその鍵世代で強制的に上書きする
                    if (keyGenerationMinList.ContainsKey(contentMetaId))
                    {
                        keyGeneration = keyGenerationMinList[contentMetaId];
                    }
                    else
                    {
                        // アプリ・追加コンテンツ、初回パッチもしくはコンテンツサイズが sizeThresholdForKeyGenerationUpdate を超えないパッチであれば、
                        // 鍵世代が desiredSafeKeyGeneration 以上になるようにする
                        bool canUpdateKeyGeneration = false;
                        if (entry.ContentMetaInfo.Model.KeepGeneration ?? false)
                        {
                            // KeepGeneration が有効になっている場合は更新しない
                            canUpdateKeyGeneration = false;
                        }
                        else if (entry.MetaType != NintendoContentMetaConstant.ContentMetaTypePatch)
                        {
                            canUpdateKeyGeneration = true;
                        }
                        else
                        {
                            var model = entry.ContentMetaInfo.Model as PatchContentMetaModel;
                            if (!model.HistoryList.Any(x => x.Type == NintendoContentMetaConstant.ContentMetaTypePatch))
                            {
                                // 初回パッチ
                                canUpdateKeyGeneration = true;
                            }
                            else
                            {
                                // 鍵世代 0 -> 2 の更新はコモンチケットを置き換えられないため不可
                                if (desiredSafeKeyGeneration > 2 && !model.ContentList.Any(x => x.Size > sizeThresholdForKeyGenerationUpdate))
                                {
                                    canUpdateKeyGeneration = true;
                                }
                            }
                        }

                        if (canUpdateKeyGeneration && (keyGeneration < desiredSafeKeyGeneration))
                        {
                            keyGeneration = desiredSafeKeyGeneration;
                        }
                    }

                    for (int j = 0; j < entry.Contents.Count; j++)
                    {
                        var content = entry.Contents[j];
                        if (IsContentTypeEncryptedByExternalKey(content.ContentType))
                        {
                            (content.FsInfo as NintendoContentFileSystemInfo).rightsId = TicketUtility.CreateRightsId(contentMetaId, keyGeneration);
                        }
                        (content.FsInfo as NintendoContentFileSystemInfo).keyGeneration = keyGeneration; // 鍵世代を揃える

                        // W/A: HtmlDocument の SecureValue が設定されていない場合改めて設定する
                        // TODO: 適切な箇所で再設定を行う
                        if (content.ContentType == NintendoContentMetaConstant.ContentTypeHtmlDocument)
                        {
                            var fsEntries = (content.FsInfo as NintendoContentFileSystemInfo).fsEntries;
                            for (int k = 0; k < fsEntries.Count; k++)
                            {
                                var fsEntry = fsEntries[k];
                                if (NintendoContentArchiveUtil.NeedsAesCtrSecureValue(fsEntry) && fsEntry.secureValue == 0)
                                {
                                    fsEntry.secureValue = (uint)NintendoContentMeta.ConvertContentType(content.ContentType);
                                    fsEntries[k] = fsEntry;
                                }
                            }
                        }

                        content.SetFsInfo(content.ContentType, null, content.FsInfo);
                        entry.Contents[j] = content;
                    }
                    prodNspInfo.Entries[i] = entry;
                }
                // システム向け
                else if (entry.MetaType == NintendoContentMetaConstant.ContentMetaTypeSystemProgram ||
                         entry.MetaType == NintendoContentMetaConstant.ContentMetaTypeSystemData ||
                         entry.MetaType == NintendoContentMetaConstant.ContentMetaTypeSystemUpdate ||
                         entry.MetaType == NintendoContentMetaConstant.ContentMetaTypeBootImagePackage ||
                         entry.MetaType == NintendoContentMetaConstant.ContentMetaTypeBootImagePackageSafe)
                {
                    // システムタイトルはツールが対応する最新の鍵世代まで自動的に上げるようにする
                    var keepGeneration = entry.ContentMetaInfo.Model.KeepGeneration ?? false;
                    var keyGenerationListSpecified = keyGenerationMinList.ContainsKey(contentMetaId);
                    for (int j = 0; j < entry.Contents.Count; j++)
                    {
                        var content = entry.Contents[j];
                        var fsInfo = (content.FsInfo as NintendoContentFileSystemInfo);
                        if (keyGenerationListSpecified)
                        {
                            fsInfo.keyGeneration = keyGenerationMinList[contentMetaId];
                        }
                        else if (!keepGeneration && fsInfo.keyGeneration < NintendoContentFileSystemMetaConstant.CurrentKeyGeneration)
                        {
                            fsInfo.keyGeneration = NintendoContentFileSystemMetaConstant.CurrentKeyGeneration;
                        }
                        content.SetFsInfo(content.ContentType, null, content.FsInfo);
                        entry.Contents[j] = content;
                    }
                    prodNspInfo.Entries[i] = entry;
                }
            }
        }

        public static void SubstituteRequiredSystemVersion(ref NintendoSubmissionPackageFileSystemInfo prodNspInfo, uint requiredSystemVersion)
        {
            for (int i = 0; i < prodNspInfo.Entries.Count; i++)
            {
                var entry = prodNspInfo.Entries[i];
                switch (entry.MetaType)
                {
                    case NintendoContentMetaConstant.ContentMetaTypeApplication:
                        {
                            var model = entry.ContentMetaInfo.Model as ApplicationContentMetaModel;
                            if (model.RequiredSystemVersion > requiredSystemVersion)
                            {
                                throw new ArgumentException(string.Format("Specified required system version (= {0}) is older than that of application (= {1}).", requiredSystemVersion, model.RequiredSystemVersion));
                            }
                            model.RequiredSystemVersion = requiredSystemVersion;
                            prodNspInfo.Entries[i] = entry;
                        }
                        break;
                    case NintendoContentMetaConstant.ContentMetaTypePatch:
                        {
                            var model = entry.ContentMetaInfo.Model as PatchContentMetaModel;
                            if (model.RequiredSystemVersion > requiredSystemVersion)
                            {
                                throw new ArgumentException(string.Format("Specified required system version (= {0}) is older than that of patch (= {1}).", requiredSystemVersion, model.RequiredSystemVersion));
                            }
                            model.OriginalRequiredSystemVersion = model.RequiredSystemVersion;
                            model.RequiredSystemVersion = requiredSystemVersion;
                            prodNspInfo.Entries[i] = entry;
                        }
                        break;
                    default:
                        continue;
                }
            }
        }
    }

    internal class ProdEncryptedPatchableNintendoSubmissionPackageArchive : IConnector
    {
        public List<Connection> ConnectionList { get; private set; }
        private KeyConfiguration m_KeyConfig;

        public ProdEncryptedPatchableNintendoSubmissionPackageArchive(IReadableSink outSink, NintendoSubmissionPackageReader patchReader, NintendoSubmissionPackageReader originalReader, NintendoSubmissionPackageReader previousProdPatchReader, uint? requiredSystemVersionSubstitute, byte desiredSafeKeyGeneration, long sizeThresholdForKeyGenerationUpdate, KeyConfiguration keyConfig, DebugConfiguration debugConfig)
        {
            m_KeyConfig = keyConfig;

            var prodPatchNspInfo = ArchiveReconstructionUtils.GetProdNspInfo(patchReader, m_KeyConfig);

            // Patch 以外のエントリを削除（Delta 入りの場合を想定）
            RemoveOtherEntry(ref prodPatchNspInfo);

            // 前バージョンの鍵世代は無条件で引き継ぐ
            SetPreviousKeyGeneration(ref prodPatchNspInfo, previousProdPatchReader);

            // RightsId の埋め込み
            ProdEncryptionUtils.SetRightsIdAndKeyGeneration(ref prodPatchNspInfo, desiredSafeKeyGeneration, sizeThresholdForKeyGenerationUpdate);

            SetPatchFsInfo(ref prodPatchNspInfo, originalReader, previousProdPatchReader);

            // RequiredSystemVersion の上書き
            if (requiredSystemVersionSubstitute.HasValue)
            {
                ProdEncryptionUtils.SubstituteRequiredSystemVersion(ref prodPatchNspInfo, requiredSystemVersionSubstitute.Value);
            }

            var prodPatchNsp = new NintendoSubmissionPackageArchive(outSink, prodPatchNspInfo, m_KeyConfig);
            ConnectionList = new List<Connection>(prodPatchNsp.ConnectionList);
        }

        // 製品化済みのパッチとデルタをマージ TODO: 開発版と共通化
        public ProdEncryptedPatchableNintendoSubmissionPackageArchive(IReadableSink outSink, NintendoSubmissionPackageReader patchReader, NintendoSubmissionPackageReader deltaReader, KeyConfiguration keyConfig)
        {
            m_KeyConfig = keyConfig;

            var removeList = new List<string>() {
                @".*\.cnmt.nca",
                @".*\.cnmt.xml",
                @".*\.cnmt",
            };
            var removeListForDelta = new List<string>(removeList);
            removeListForDelta.Add(@"^authoringtoolinfo\.xml$");

            // Patch, Delta の cnmt 関連の除くエントリを読み込み
            var patchNspInfo = ArchiveReconstructionUtils.GetRootEntryRemovedNspInfo(patchReader, removeList);
            var deltaNspInfo = ArchiveReconstructionUtils.GetRootEntryRemovedNspInfo(deltaReader, removeListForDelta);

            // Patch, Delta のコンテンツメタを取得しマージ
            var patchModels = ArchiveReconstructionUtils.ReadContentMetaInNsp(patchReader).Where(p => p.Type == "Patch").Select(p => p as PatchContentMetaModel).ToList();
            var deltaModels = ArchiveReconstructionUtils.ReadContentMetaInNsp(deltaReader).Where(p => p.Type == "Delta").Select(p => p as DeltaContentMetaModel).ToList();
            for (int i = 0; i < patchModels.Count; i++)
            {
                var model = patchModels[i];
                model.ContentList.RemoveAll(p => p.Type == "Meta");
                var deltaModel = deltaModels.Find(p => p.Destination.PatchId == model.Id && p.Destination.Version == model.Version.Value);
                model.ContentList.AddRange(deltaModel.ContentList.Where(p => p.Type != "Meta"));
                MergeDeltaHistory(ref model, deltaModel);
                patchModels[i] = model;
            }

            // nsp の再構築
            var mergedNspInfo = new PartitionFileSystemInfo();
            mergedNspInfo.version = 0;
            ConnectionList = new List<Connection>();
            long headerSize = 0;
            {
                var entryNameList = new List<string>();
                entryNameList.AddRange(patchNspInfo.entries.Select(x => x.name));
                entryNameList.AddRange(deltaNspInfo.entries.Select(x => x.name));
                foreach (var model in patchModels)
                {
                    // Place Holder
                    entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cnmt.nca");
                    entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cnmt.xml");
                }
                headerSize = HashNameEntryPartitionFsHeaderSource<PartitionFileSystemMeta>.GetDummySize(entryNameList);
            }

            // Patch, Delta のコンテンツを ConnectionList に追加
            var hashSources = new List<ContentHashSource>();
            long offset = headerSize;
            foreach (var entry in new [] { patchNspInfo.entries, deltaNspInfo.entries }.SelectMany(_ => _))
            {
                {
                    var partFsEntry = PartitionFileSystemInfo.EntryInfo.Make(entry.name, entry.size, (ulong)(offset - headerSize));
                    mergedNspInfo.entries.Add(partFsEntry);
                }

                hashSources.Add(new ContentHashSource(null, "nochange"));

                var source = (ISource)entry.sourceInterface;
                outSink.SetSize(offset + source.Size);
                var sink = new SubSink(outSink, offset, source.Size);
                ConnectionList.Add(new Connection(source, sink));
                offset += source.Size;
            }

            // コンテンツメタ、コンテンツメタ XML を再生成し ConnectionList に追加
            foreach (var model in patchModels)
            {
                var metaBase = new NintendoContentMetaBase(model, true);
                var metaSource = new NintendoContentMetaArchiveSource(metaBase, 0, keyConfig, true, false, false);
                outSink.SetSize(offset + metaSource.Size);
                {
                    var partFsEntry = PartitionFileSystemInfo.EntryInfo.Make("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cnmt.nca", (ulong)metaSource.Size, (ulong)(offset - headerSize));
                    mergedNspInfo.entries.Add(partFsEntry);
                }

                var metaSink = new ReadableSubSink(outSink, offset, metaSource.Size);
                ConnectionList.Add(new Connection(metaSource, metaSink));
                var metaHashSource = new SinkLinkedSource(metaSink, new Sha256StreamHashSource(metaSink.ToSource()));
                hashSources.Add(new ContentHashSource(metaHashSource, ".cnmt.nca"));
                offset += metaSource.Size;

                var metaXmlSource = new NintendoContentMetaXmlSource(metaBase, metaHashSource, metaSource.Size);
                outSink.SetSize(offset + metaXmlSource.Size);
                {
                    var partFsEntry = PartitionFileSystemInfo.EntryInfo.Make("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cnmt.xml", (ulong)metaXmlSource.Size, (ulong)(offset - headerSize));
                    mergedNspInfo.entries.Add(partFsEntry);
                }

                ConnectionList.Add(new Connection(metaXmlSource, new SubSink(outSink, offset, metaXmlSource.Size)));
                hashSources.Add(new ContentHashSource(metaHashSource, ".cnmt.xml"));
                offset += metaXmlSource.Size;
            }

            var headerSource = new HashNameEntryPartitionFsHeaderSource<PartitionFileSystemMeta>(hashSources, mergedNspInfo, headerSize);
            ConnectionList.Add(new Connection(headerSource, new SubSink(outSink, 0, headerSource.Size)));
        }

        private void MergeDeltaHistory(ref PatchContentMetaModel patchModel, DeltaContentMetaModel deltaModel)
        {
            var patchDeltaModel = new PatchDeltaModel();
            {
                patchDeltaModel.ContentList = deltaModel.ContentList;
                patchDeltaModel.Source = deltaModel.Source;
                patchDeltaModel.Destination = deltaModel.Destination;
                patchDeltaModel.FragmentSetList = deltaModel.FragmentSetList.Where(p => p.FragmentTargetContentType != NintendoContentMetaConstant.ContentTypeMeta).ToList();
            }
            if (patchModel.DeltaHistoryList == null)
            {
                patchModel.DeltaHistoryList = new List<PatchDeltaHistoryModel>();
            }
            patchModel.DeltaHistoryList.Add(patchDeltaModel.GetHistory());
            if (patchModel.DeltaList == null)
            {
                patchModel.DeltaList = new List<PatchDeltaModel>();
            }
            patchModel.DeltaList.Add(patchDeltaModel);
        }

        private void RemoveOtherEntry(ref NintendoSubmissionPackageFileSystemInfo nspInfo)
        {
            nspInfo.Entries = nspInfo.Entries.Where(x => x.MetaType == NintendoContentMetaConstant.ContentMetaTypePatch).ToList();
            if (nspInfo.Entries.Count == 0)
            {
                throw new ArgumentException("This nsp is not patch.");
            }
            foreach (var entry in nspInfo.Entries)
            {
                entry.Contents.RemoveAll(x => x.ContentType == NintendoContentMetaConstant.ContentTypeDeltaFragment);
            }
        }

        // 連続性チェックと歴史情報の差し換え
        private void SetPatchHistory(ref NintendoSubmissionPackageFileSystemInfo.EntryInfo entry, NintendoSubmissionPackageReader previousPatchProdReader)
        {
            var model = entry.ContentMetaInfo.Model as PatchContentMetaModel;
            if (previousPatchProdReader == null)
            {
                // 初回パッチのみ許容
                if (model.HistoryList.Any(x => x.Type == NintendoContentMetaConstant.ContentMetaTypePatch))
                {
                    throw new ArgumentException("To prodencrypt this patch, previous prod patch must be specified.");
                }
            }
            else
            {
                var previousModel = ArchiveReconstructionUtils.ReadContentMetaInNsp(previousPatchProdReader).Find(p => p.Type == "Patch" && p.Id == model.Id) as PatchContentMetaModel;
                // 製品化対象のパッチの最適化元になったパッチの Digest が previousProd として指定されたパッチの Digest と一致しないとエラー
                if (model.HistoryList.Where(x => x.Type == NintendoContentMetaConstant.ContentMetaTypePatch).Last().Digest !=  previousModel.Digest)
                {
                    throw new ArgumentException("There is no continuity between this patch and previous prod patch.");
                }
                // 歴史情報の差し換え
                model.HistoryList = previousModel.HistoryList;
                {
                    var latestPatchModel = new PatchHistoryModel();
                    latestPatchModel.Type = previousModel.Type;
                    latestPatchModel.Id = previousModel.Id;
                    latestPatchModel.Version = previousModel.Version.Value;
                    latestPatchModel.Digest = previousModel.Digest;
                    latestPatchModel.ContentList = previousModel.ContentList.Where(x => x.Type != NintendoContentMetaConstant.ContentTypeDeltaFragment).ToList();
                    model.HistoryList.Add(latestPatchModel);
                }
                // TORIAEZU: Delta は後処理でマージ
                model.DeltaHistoryList = previousModel.DeltaHistoryList;
                model.DeltaList = null;
            }
        }

        // patch 側の未設定の情報を設定
        private void SetPatchFsInfo(ref NintendoSubmissionPackageFileSystemInfo patch, NintendoSubmissionPackageReader originalReader, NintendoSubmissionPackageReader previousPatchProdReader)
        {
            for (int i = 0; i < patch.Entries.Count; i++)
            {
                var entry = patch.Entries[i];
                if (entry.MetaType != NintendoContentMetaConstant.ContentMetaTypePatch)
                {
                    continue;
                }
                var applicationId = ((PatchContentMetaModel)entry.ContentMetaInfo.Model).GetUInt64ApplicationId();
                for (int j = 0; j < entry.Contents.Count; j++)
                {
                    var content = entry.Contents[j];
                    if (!ArchiveReconstructionUtils.IsPatchTargetNca(content.ContentType))
                    {
                        continue;
                    }
                    var ncaInfo = content.FsInfo as NintendoContentFileSystemInfo;
                    for (int k = 0; k < ncaInfo.fsEntries.Count; ++k)
                    {
                        var fsEntry = ncaInfo.fsEntries[k];
                        if (fsEntry.patchInfo == null)
                        {
                            continue;
                        }
                        // NintendoContentPatchFileSystemInfo の構築
                        {
                            var originalFsDataSource = GetOriginalFsDataSource(originalReader, applicationId, content.ContentType, content.IdOffset, fsEntry.partitionIndex);
                            var patchFsDataSource = fsEntry.sourceInterface as ISource;
                            var differenceSource = new SubSource(patchFsDataSource, 0, fsEntry.patchInfo.indirectOffset);
                            byte[] indirectTable;
                            {
                                var tableSource = new SubSource(patchFsDataSource, fsEntry.patchInfo.indirectOffset, fsEntry.patchInfo.indirectSize);
                                indirectTable = tableSource.PullData(0, (int)tableSource.Size).Buffer.Array;
                                Debug.Assert(indirectTable.Length == fsEntry.patchInfo.indirectSize);
                            }
                            byte[] aesCtrExTable;
                            {
                                var tableSource = new SubSource(patchFsDataSource, fsEntry.patchInfo.aesCtrExOffset, fsEntry.patchInfo.aesCtrExSize);
                                aesCtrExTable = tableSource.PullData(0, (int)tableSource.Size).Buffer.Array;
                                Debug.Assert(aesCtrExTable.Length == fsEntry.patchInfo.aesCtrExSize);
                            }

                            fsEntry.fileSystemInfo = new NintendoContentPatchFileSystemInfo(fsEntry.patchInfo.indirectHeader, indirectTable, aesCtrExTable, originalFsDataSource, differenceSource);
                            fsEntry.generationTable = new AesCtrExGenerationTable(fsEntry.patchInfo.aesCtrExHeader, aesCtrExTable);
                            var patchInfo = (fsEntry.fileSystemInfo as NintendoContentPatchFileSystemInfo);
                            patchInfo.HashTargetOffset = (long)fsEntry.hashTargetOffset;
                            fsEntry.sourceInterface = null; // 無効化しておく
                        }
                        ncaInfo.fsEntries[k] = fsEntry;
                    }
                    entry.Contents[j] = content;
                }
                SetPatchHistory(ref entry, previousPatchProdReader);
                patch.Entries[i] = entry;
            }
        }

        private void SetPreviousKeyGeneration(ref NintendoSubmissionPackageFileSystemInfo patch, NintendoSubmissionPackageReader previousProdPatchReader)
        {
            if (previousProdPatchReader == null)
            {
                return;
            }

            for (int i = 0; i < patch.Entries.Count; i++)
            {
                var entry = patch.Entries[i];
                var previousKeyGeneration = UpdatePartitionUtils.GetLatestKeyGeneration(previousProdPatchReader);
                for (int j = 0; j < entry.Contents.Count; j++)
                {
                    var content = entry.Contents[j];
                    if ((content.FsInfo as NintendoContentFileSystemInfo).keyGeneration < previousKeyGeneration)
                    {
                        (content.FsInfo as NintendoContentFileSystemInfo).keyGeneration = previousKeyGeneration;
                    }
                    entry.Contents[j] = content;
                }
                patch.Entries[i] = entry;
            }
        }

        private ISource GetOriginalFsDataSource(NintendoSubmissionPackageReader originalReader, UInt64 applicationId, string contentType, byte idOffset, int index)
        {
            var ncaReader = ArchiveReconstructionUtils.GetSpecifiedContentArchiveReader(originalReader, applicationId, contentType, idOffset, new NcaKeyGenerator(m_KeyConfig));
            if (ncaReader == null)
            {
                // original にパッチに対応する nca が無い場合はサイズ 0 のソースを返しておくことで IndirectStorage が対応できる
                return new PaddingSource(0);
            }
            var storageReader = ncaReader.OpenFsDataStorageArchiveReader(index);
            return new StorageArchiveSource(storageReader);
        }

        public ISource GetSource()
        {
            throw new NotImplementedException();
        }
    }

    internal class ProdEncryptedNintendoSubmissionPackageArchive : IConnector
    {
        public List<Connection> ConnectionList { get; private set; }
        private KeyConfiguration m_KeyConfig;

        public ProdEncryptedNintendoSubmissionPackageArchive(IReadableSink outSink, NintendoSubmissionPackageReader nspReader,  uint? requiredSystemVersionSubstitute, List<UInt64> excludeList, Dictionary<UInt64, byte> keyGenerationList, byte desiredSafeKeyGeneration, KeyConfiguration keyConfig, DebugConfiguration debugConfig)
        {
            m_KeyConfig = keyConfig;

            var prodNspInfo = ArchiveReconstructionUtils.GetProdNspInfo(nspReader, m_KeyConfig);
            if (prodNspInfo.OnCardAddOnContentInfo != null)
            {
                throw new ArgumentException("This nsp cannot be prod-encrypted.");
            }

            prodNspInfo.EnableContentMetaBinaryExport = debugConfig.EnableContentMetaBinaryExport;

            // 除外リストで指定されたコンテンツを除外
            prodNspInfo.Entries.RemoveAll((entry) =>
            {
                return excludeList.Contains(entry.ContentMetaInfo.Model.GetUInt64Id());
            });

            // RightsId の埋め込み
            ProdEncryptionUtils.SetRightsIdAndKeyGeneration(ref prodNspInfo, keyGenerationList, desiredSafeKeyGeneration, 0);

            // RequiredSystemVersion の上書き
            if (requiredSystemVersionSubstitute.HasValue)
            {
                ProdEncryptionUtils.SubstituteRequiredSystemVersion(ref prodNspInfo, requiredSystemVersionSubstitute.Value);
            }

            var prodNsp = new NintendoSubmissionPackageArchive(outSink, prodNspInfo, m_KeyConfig);
            ConnectionList = new List<Connection>(prodNsp.ConnectionList);
        }

        public ISource GetSource()
        {
            throw new NotImplementedException();
        }
    }
}
