﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;
using System.Collections.Generic;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class PatchedNintendoSubmissionPackageReader : NintendoSubmissionPackageReader
    {
        private NintendoSubmissionPackageReader m_OriginalNspReader;
        private List<ContentMetaModel> m_ContentMetaList;

        public PatchedNintendoSubmissionPackageReader(string path, NintendoSubmissionPackageReader originalNspReader) : base(path)
        {
            m_OriginalNspReader = originalNspReader;
            m_ContentMetaList = ArchiveReconstructionUtils.ReadContentMetaInNsp(this);
        }

        // Item1: patchNcaReader, Item2: originalNcaReader
        public Tuple<NintendoContentArchiveReader, NintendoContentArchiveReader> OpenPatchableNintendoContentArchiveReader(string fileName, IKeyGenerator keyGenerator)
        {
            string contentType;
            UInt64 contentMetaId;
            byte contentIdOffset;
            return OpenPatchableNintendoContentArchiveReader(fileName, keyGenerator, out contentType, out contentIdOffset, out contentMetaId);
        }

        // Item1: patchNcaReader, Item2: originalNcaReader
        public Tuple<NintendoContentArchiveReader, NintendoContentArchiveReader> OpenPatchableNintendoContentArchiveReader(string fileName, IKeyGenerator keyGenerator, out string contentType, out byte contentIdOffset, out UInt64 contentMetaId)
        {
            var contentId = fileName.Replace(".nca", string.Empty).Replace(".cnmt", string.Empty);

            var patchedNcaReader = base.OpenNintendoContentArchiveReader(fileName, keyGenerator);
            TicketUtility.SetExternalKey(ref patchedNcaReader, this);

            PatchContentMetaModel patchContentMeta;
            {
                var contentMeta = m_ContentMetaList.Find(x => x.ContentList.Where(y => y.Id == contentId).Any());
                if (contentMeta.Type != NintendoContentMetaConstant.ContentMetaTypePatch)
                {
                    contentType = "";
                    contentMetaId = 0;
                    contentIdOffset = (byte)0;
                    return Tuple.Create(patchedNcaReader, null as NintendoContentArchiveReader);
                }
                patchContentMeta = contentMeta as PatchContentMetaModel;
            }

            var contentInfo = patchContentMeta.ContentList.Where(x => x.Id == contentId).Single();
            contentType = contentInfo.Type;
            contentIdOffset = contentInfo.IdOffset;
            contentMetaId = patchContentMeta.GetUInt64Id();
            if (contentType != NintendoContentMetaConstant.ContentTypeProgram && contentType != NintendoContentMetaConstant.ContentTypeHtmlDocument)
            {
                return Tuple.Create(patchedNcaReader, null as NintendoContentArchiveReader);
            }

            var originalNcaReader = ArchiveReconstructionUtils.GetSpecifiedContentArchiveReader(m_OriginalNspReader, patchContentMeta.GetUInt64ApplicationId(), contentType, contentIdOffset, keyGenerator);
            return Tuple.Create(patchedNcaReader, originalNcaReader);
        }

        // TODO: 本来は patchModel.ApplicationId で探すべきだが、resultXml には出力されていないためその W/A 用
        static internal void SetContentListForXci(out List<Tuple<string, string>> patchableContentList, List<ContentMetaModel> patchModels, List<ContentMetaModel> applicationModels)
        {
            patchableContentList = new List<Tuple<string, string>>();
            foreach (var patchModel in patchModels)
            {
                var dest = applicationModels.Where(x => IdConverter.ConvertToPatchId(x.GetUInt64Id()) == patchModel.GetUInt64Id());
                if (!dest.Any())
                {
                    throw new ArgumentException("There is no application for patch {0}", patchModel.Id);
                }

                foreach (var content in patchModel.ContentList)
                {
                    if (content.Type == NintendoContentMetaConstant.ContentTypeProgram || content.Type == NintendoContentMetaConstant.ContentTypeHtmlDocument)
                    {
                        foreach (var appli in dest)
                        {
                            var originalContent = appli.ContentList.Find(x => x.Type == content.Type);
                            patchableContentList.Add(Tuple.Create(content.Id + ".nca", originalContent != null ? originalContent.Id + ".nca": null));
                        }
                    }
                }
            }
        }
    }
}
