﻿using System;
using System.Diagnostics;
using System.Linq;
using System.IO;
using System.Text;
using System.Collections.Generic;
using System.Xml.Serialization;
using Nintendo.Authoring.FileSystemMetaLibrary;
using System.Text.RegularExpressions;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class LegalInformationXml : ISource
    {
        public long Size { get; set; }
        private ISource m_source;

        public LegalInformationXml(NintendoSubmissionPackageFileSystemInfo.ContentInfo legalInfoContent, KeyConfiguration config)
        {
            Trace.Assert(legalInfoContent.ContentType == NintendoContentMetaConstant.ContentTypeLegalInformation);

            var legalInfoXmlSource = NintendoSubmissionPackageArchiveUtils.GetMemorySourceFromContent(legalInfoContent, config, "legalinfo.xml");
            if (legalInfoXmlSource == null)
            {
                throw new ArgumentException("legalinfo.xml is not found in the zip file.");
            }

            var legalInfoXmlBytes = legalInfoXmlSource.PullData(0, (int)legalInfoXmlSource.Size).Buffer.Array;

            // BOM 付き UTF-8 CRLF に変換
            byte[] xmlBytes;
            using (var dst = new MemoryStream())
            using (var src = new MemoryStream(legalInfoXmlBytes.Length))
            {
                src.Write(legalInfoXmlBytes, 0, legalInfoXmlBytes.Length);
                src.Seek(0, SeekOrigin.Begin);

                var sw = new StreamWriter(dst, Encoding.UTF8);
                var sr = new StreamReader(src, Encoding.UTF8);

                while (!sr.EndOfStream)
                {
                    var line = sr.ReadLine();
                    if (sr.EndOfStream)
                    {
                        // 最終行は改行しない
                        sw.Write(line);
                    }
                    else
                    {
                        sw.WriteLine(line);
                    }
                }

                sw.Flush();

                xmlBytes = dst.ToArray();
            }

            m_source = new MemorySource(xmlBytes, 0, xmlBytes.Length);
            Size = m_source.Size;
        }

        private LegalInformationXml(NintendoSubmissionPackageReader nspReader, string fileName, long size)
        {
            var data = nspReader.ReadFile(fileName, 0, size);
            m_source = new MemorySource(data, 0, data.Length);
            Size = m_source.Size;
        }

        static public LegalInformationXml LoadFromNsp(NintendoSubmissionPackageReader nspReader)
        {
            var entry = nspReader.ListFileInfo().Find(e => e.Item1.EndsWith(".legalinfo.xml"));
            if (entry != null)
            {
                return new LegalInformationXml(nspReader, entry.Item1, entry.Item2);
            }
            return null;
        }

        public LegalInformationXml(LegalInformationModel model)
        {
            var data = model.GetBytes();
            m_source = new MemorySource(data, 0, data.Length);
            Size = m_source.Size;
        }

        public ByteData PullData(long offset, int size)
        {
            return m_source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
    }

    [XmlRoot("ProductRegions", IsNullable = false)]
    public class LegalInformationProductRegionsModel
    {
        [XmlElement("Usa", IsNullable = false)]
        public bool Usa { get; set; }
        [XmlElement("Japan", IsNullable = false)]
        public bool Japan { get; set; }
        [XmlElement("Europe", IsNullable = false)]
        public bool Europe { get; set; }
        public LegalInformationProductRegionsModel() { }
    }

    [XmlRoot("CopyrightNotations", IsNullable = false)]
    public class LegalInformationCopyrightNotationsModel
    {
        [XmlElement("TechnologyName", IsNullable = false)]
        public List<string> TechnologyName { get; set; }
        public LegalInformationCopyrightNotationsModel() { }
    }

    [XmlRoot("DeclarationRequiredNotations", IsNullable = false)]
    public class LegalInformationDeclarationRequiredNotationsModel
    {
        [XmlElement("TechnologyName", IsNullable = false)]
        public List<string> TechnologyName { get; set; }
        public LegalInformationDeclarationRequiredNotationsModel() { }
    }

    [XmlRoot("SoftwareLegalInformation", IsNullable = false)]
    public class LegalInformationModel
    {
        [XmlElement("ProductRegions", IsNullable = false)]
        public LegalInformationProductRegionsModel ProductRegions { get; set; }
        [XmlElement("DataHash", IsNullable = false)]
        public string DataHash { get; set; }
        [XmlElement("CopyrightNotations", IsNullable = false)]
        public LegalInformationCopyrightNotationsModel CopyrightNotations { get; set; }
        [XmlElement("DeclarationRequiredNotations", IsNullable = false)]
        public LegalInformationDeclarationRequiredNotationsModel DeclarationRequiredNotations { get; set; }
        [XmlElement("ApplicationId", IsNullable = false)]
        public string ApplicationId { get; set; }
        [XmlElement("OutputDateTime", IsNullable = false)]
        public string OutputDateTime { get; set; }
        [XmlElement("FormatVersion", IsNullable = false)]
        public string FormatVersion { get; set; }
        public LegalInformationModel() { }

        public byte[] GetBytes()
        {
            var nameSpace = new XmlSerializerNamespaces();
            nameSpace.Add(string.Empty, string.Empty);

            using (var memoryStream = new MemoryStream())
            {
                var sw = new StreamWriter(memoryStream, Encoding.UTF8);
                var serializer = new XmlSerializer(typeof(LegalInformationModel));
                serializer.Serialize(sw, this, nameSpace);
                return memoryStream.ToArray();
            }
        }
    }
}
