﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Security.Cryptography;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Authoring.CryptoLibrary;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    // IndirectStorageStream から差分データを読み出す ISource
    internal class IndirectStorageDifferenceSource : ISource
    {
        public long Size { get; private set; }
        private IndirectStorageDifferenceStream m_Stream;
        private SourceStatus m_Status;

        public IndirectStorageDifferenceSource(IndirectStorageDifferenceStream stream)
        {
            m_Stream = stream;
            Size = m_Stream.GetDataSize();
            m_Status = new SourceStatus();
            m_Status.AvailableRangeList.MergingAdd(new Range(0, Size));
        }

        public ByteData PullData(long offset, int size)
        {
            int readSize = SourceUtil.GetReadableSize(this.Size, offset, size);
            if (readSize == 0)
            {
                return new ByteData(new ArraySegment<byte>());
            }

            var buffer = m_Stream.ReadData(offset, readSize);
            return new ByteData(new ArraySegment<byte>(buffer, 0, buffer.Length));
        }

        public SourceStatus QueryStatus()
        {
            return m_Status;
        }
    }

    // IndirectStorage の差分データを取り出すためのクラス
    public class IndirectStorageArchiveBuilder
    {
        public const int BlockSize = 16;
        public const int RegionSize = 16 * 1024;
        private const long MatchSize = 32 * 1024;
        private const long WindowSize = long.MaxValue;

        private IndirectStorageDifferenceStream m_Stream;
        private byte[] m_Header;
        private byte[] m_Table;

        public IndirectStorageArchiveBuilder(ISource oldSource, ISource newSource)
        {
            // oldSource, newSource は全領域が有効である必要がある
            // TODO: チェック or 解消
            var oldStream = new IndirectStorageReadOnlySource(oldSource, BlockSize);
            var newStream = new IndirectStorageReadOnlySource(newSource, BlockSize);
            m_Stream = new IndirectStorageDifferenceStream(oldStream, newStream);
        }

        // 更新前後 2 つのソースからテーブル情報を生成する
        public void Build(List<IndirectStorageStream.ExcludeRange> oldExcludeRanges, List<IndirectStorageStream.ExcludeRange> newExcludeRanges)
        {
            if (oldExcludeRanges != null && 0 < oldExcludeRanges.Count)
            {
                m_Stream.SetExcludeRangeForOldSource(oldExcludeRanges.ToArray());
            }
            if (newExcludeRanges != null && 0 < newExcludeRanges.Count)
            {
                m_Stream.SetExcludeRangeForNewSource(newExcludeRanges.ToArray());
            }

            m_Stream.Build(BlockSize, BlockSize, RegionSize, MatchSize, WindowSize);

            m_Header = new byte[m_Stream.GetIndirectHeaderSize()];
            m_Table = new byte[m_Stream.GetIndirectTableSize()];
            m_Stream.WriteIndirectTable(m_Header, m_Table);
        }

        // 生成済みのテーブル情報と任意の更新後ソースを注入する
        public void Import(byte[] headerData, byte[] tableData, ISource differenceSource)
        {
            m_Stream.Import(BlockSize, headerData, tableData, new SourceBasedStream(differenceSource));
            m_Header = headerData;
            m_Table = tableData;
        }

        public byte[] GetTable()
        {
            return m_Table;
        }

        public byte[] GetHeader()
        {
            return m_Header;
        }

        public ISource GetDifferenceSource()
        {
            Trace.Assert(m_Header != null && m_Table != null);
            return new IndirectStorageDifferenceSource(m_Stream);
        }
    }
}
