﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Security.Cryptography;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Authoring.CryptoLibrary;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class SourceStatus
    {
        /// <summary>
        /// ISource のデータのうち、有効なデータの領域を表すリストです。
        /// </summary>
        public RangeList AvailableRangeList { get; set; }
        public SourceStatus()
        {
            AvailableRangeList = new RangeList();
        }
    }

    public class SourceUtil
    {
        public static int GetReadableSize(long sourceSize, long offset, int sizeToRead)
        {
            if (offset < 0)
            {
                return 0;
            }
            if (offset >= sourceSize)
            {
                return 0;
            }
            return (int)Math.Min(sizeToRead, sourceSize - offset);
        }

        static internal T CastSource<T>(ISource source)
        {
            if (source is CliCompatibleSource)
            {
                System.Diagnostics.Debug.Assert(((CliCompatibleSource)source).Source is T);
                return (T)((CliCompatibleSource)source).Source;
            }
            else
            {
                System.Diagnostics.Debug.Assert(source is T);
                return (T)source;
            }
        }

        internal static ISource MakeMemoryCacheSource(ISource source)
        {
            var data = source.PullData(0, (int)source.Size);
            return new MemorySource(data.Buffer.Array, data.Buffer.Offset, data.Buffer.Count());
        }

        internal static ISource MakeFileCacheSource(ISource source, string cachePath)
        {
            const int MaxWorkBufferSize = 32 * 1024 * 1024;

            using (var file = new FileStream(cachePath, FileMode.Create, FileAccess.Write))
            {
                long offset = 0;
                while (offset < source.Size)
                {
                    var currentSize = (int)Math.Min(MaxWorkBufferSize, source.Size - offset);
                    var data = source.PullData(offset, currentSize);
                    file.Write(data.Buffer.Array, data.Buffer.Offset, data.Buffer.Count());
                    offset += currentSize;
                }
            }

            return new FileSource(cachePath, 0, source.Size);
        }
    }

    public class FileSource : ISource
    {
        public long Size { get; private set; }

        private string m_filePath;
        private long m_offset;
        private SourceStatus m_status;

        public FileSource(string filePath, long offset, long size)
        {
            m_filePath = filePath;
            m_offset = offset;
            Size = size;
            m_status = new SourceStatus();
            m_status.AvailableRangeList.MergingAdd(new Range(0, Size));
        }
        public ByteData PullData(long offset, int size)
        {
            int readSize = SourceUtil.GetReadableSize(this.Size, offset, size);
            if (readSize == 0)
            {
                return new ByteData(new ArraySegment<byte>());
            }
            byte[] buffer = new byte[readSize];
            int read = 0;
            using (Stream stream = new SubStream(new FileStream(m_filePath, FileMode.Open, FileAccess.Read, FileShare.Read, 4096, FileOptions.SequentialScan), m_offset, this.Size))
            {
                stream.Seek(offset, SeekOrigin.Begin);
                read = stream.Read(buffer, 0, readSize);
            }
            return new ByteData(new ArraySegment<byte>(buffer, 0, read));
        }
        public SourceStatus QueryStatus()
        {
            return m_status;
        }
    }

    public class MemorySource : ISource
    {
        public long Size { get; private set; }

        private int m_offset;
        private byte[] m_buffer;
        private SourceStatus m_status;

        public MemorySource(byte[] buffer)
        {
            m_buffer = buffer;
            m_offset = 0;
            Size = buffer.Length;
            m_status = new SourceStatus();
            m_status.AvailableRangeList.MergingAdd(new Range(0, Size));
        }
        public MemorySource(byte[] buffer, int offset, int size)
        {
            m_buffer = new byte[buffer.Length];
            Buffer.BlockCopy(buffer, 0, m_buffer, 0, m_buffer.Length);
            m_offset = offset;
            Size = size;
            m_status = new SourceStatus();
            m_status.AvailableRangeList.MergingAdd(new Range(0, Size));
        }
        public ByteData PullData(long offset, int size)
        {
            int readSize = SourceUtil.GetReadableSize(this.Size, offset, size);
            if (readSize == 0)
            {
                return new ByteData(new ArraySegment<byte>());
            }
            return new ByteData(new ArraySegment<byte>(m_buffer, m_offset + (int)offset, readSize));
        }
        public SourceStatus QueryStatus()
        {
            return m_status;
        }
    }

    public class UpdatableMemorySource : ISource
    {
        public long Size { get; private set; }

        private int m_offset;
        private byte[] m_buffer;
        private SourceStatus m_status;

        public UpdatableMemorySource(byte[] buffer, int offset, int size)
        {
            m_buffer = buffer;
            m_offset = offset;
            Size = size;
            m_status = new SourceStatus();
            m_status.AvailableRangeList.MergingAdd(new Range(0, Size));
        }
        public ByteData PullData(long offset, int size)
        {
            int readSize = SourceUtil.GetReadableSize(this.Size, offset, size);
            if (readSize == 0)
            {
                return new ByteData(new ArraySegment<byte>());
            }
            return new ByteData(new ArraySegment<byte>(m_buffer, m_offset + (int)offset, readSize));
        }
        public SourceStatus QueryStatus()
        {
            return m_status;
        }
    }

    public class StreamSource : ISource
    {
        public long Size { get; private set; }

        private Stream m_stream;
        private long m_offset;
        private SourceStatus m_status;

        public StreamSource(Stream stream, long offset, long size)
        {
            m_stream = stream;
            m_offset = offset;
            Size = size;
            m_status = new SourceStatus();
            m_status.AvailableRangeList.MergingAdd(new Range(0, Size));
        }
        public ByteData PullData(long offset, int size)
        {
            int readSize = SourceUtil.GetReadableSize(this.Size, offset, size);
            if (readSize == 0)
            {
                return new ByteData(new ArraySegment<byte>());
            }
            m_stream.Seek(offset, SeekOrigin.Begin);
            byte[] buffer = new byte[readSize];
            m_stream.Read(buffer, 0, readSize);
            return new ByteData(new ArraySegment<byte>(buffer, 0, readSize));
        }
        public SourceStatus QueryStatus()
        {
            return m_status;
        }
        public Stream GetStream()
        {
            return m_stream;
        }
    }

    public class PaddingSource : ISource
    {
        public long Size { get; private set; }

        private SourceStatus m_status;
        protected byte m_value;

        public PaddingSource(long size, byte value)
        {
            Size = size;
            m_status = new SourceStatus();
            m_status.AvailableRangeList.MergingAdd(new Range(0, Size));
            m_value = value;
        }

        public PaddingSource(long size) : this(size, 0x00)
        {
        }

        public virtual ByteData PullData(long offset, int size)
        {
            int readSize = SourceUtil.GetReadableSize(this.Size, offset, size);
            if (readSize == 0)
            {
                return new ByteData(new ArraySegment<byte>());
            }
            var data = new byte[readSize];
            if (m_value != 0x00)
            {
                for (int i = 0; i < data.Length; ++i)
                {
                    data[i] = m_value;
                }
            }
            return new ByteData(new ArraySegment<byte>(data));
        }
        public SourceStatus QueryStatus()
        {
            return m_status;
        }
    }

    public class DebugPaddingSource : PaddingSource
    {
        public string Path { get; private set; }

        public DebugPaddingSource(long size, string path) : this(size, 0x00, path)
        {
        }
        public DebugPaddingSource(long size, byte value, string path) : base(size, value)
        {
            Path = path;
        }

        public override ByteData PullData(long offset, int size)
        {
            int readSize = SourceUtil.GetReadableSize(this.Size, offset, size);
            if (readSize == 0)
            {
                return new ByteData(new ArraySegment<byte>());
            }
            var data = new byte[readSize];
            if (m_value != 0x00)
            {
                for (int i = 0; i < data.Length; ++i)
                {
                    data[i] = m_value;
                }
            }

            // Padding の先頭にパス名を上書き
            var bytePath = System.Text.Encoding.UTF8.GetBytes(Path);
            Array.Copy(bytePath, data, Math.Min(Path.Length, readSize));
            return new ByteData(new ArraySegment<byte>(data));
        }
    }

    public class ConcatenatedSource : ISource
    {
        public class Element
        {
            public ISource Source { get; private set; }
            public string Signature { get; private set; }
            public long Offset { get; private set; }

            public Element(ISource source, string signature, long offset)
            {
                Source = source;
                Signature = signature;
                Offset = offset;
            }
        }

        private class ElementOffsetComparer : IComparer<Element>
        {
            private bool m_compareSize;

            public ElementOffsetComparer(bool compareSize)
            {
                m_compareSize = compareSize;
            }

            public int Compare(Element s1, Element s2)
            {
                if (s1.Offset > s2.Offset)
                {
                    return 1;
                }
                else if (s1.Offset < s2.Offset)
                {
                    return -1;
                }
                else
                {
                    if (m_compareSize)
                    {
                        if (s1.Source.Size > s2.Source.Size)
                        {
                            return 1;
                        }
                        else if (s1.Source.Size < s2.Source.Size)
                        {
                            return -1;
                        }
                    }
                    else if (s1.Source.Size == 0)
                    {
                        // Size が 0 の場合は一致とみなさない（BinarySearch 用）
                        return -1;
                    }
                    return 0;
                }
            }
        }

        private Element GetElement(long offset)
        {
            var targetElement = new Element(new PaddingSource(0), string.Empty, offset);
            var index = Elements.BinarySearch(targetElement, new ElementOffsetComparer(false));
            if (index < 0)
            {
                index = ~index - 1;
            }
            return Elements[index];
        }

        public long Size { get; private set; }
        public List<Element> Elements { get; private set; }

        public ConcatenatedSource(List<Element> elements)
        {
            Elements = new List<Element>();
            foreach (var i in elements)
            {
                Elements.Add(i);
            }
            Elements.Sort(new ElementOffsetComparer(true));
            for (int i = 0; i < Elements.Count - 1; i++)
            {
                long paddingSize = Elements[i + 1].Offset - (Elements[i].Offset + Elements[i].Source.Size);
                if (paddingSize < 0)
                {
                    throw new ArgumentException("entries layout overlaps. please check .adf file.");
                }
                else if (paddingSize > 0)
                {
                    Element padding = new Element(
                        new PaddingSource(paddingSize), string.Empty, Elements[i].Offset + Elements[i].Source.Size);
                    Elements.Insert(i + 1, padding);
                }
            }
            Size = Elements.Last().Offset + Elements.Last().Source.Size;
        }
        public ByteData PullData(long offset, int size)
        {
            int pulled = 0;
            byte[] buffer = new byte[size];
            long currentOffset = offset;
            while (pulled != size)
            {
                Element target = GetElement(currentOffset);
                ByteData data = target.Source.PullData(currentOffset - target.Offset, size - pulled);
                if (data.Buffer.Count == 0)
                {
                    break;
                }
                Buffer.BlockCopy(data.Buffer.Array, data.Buffer.Offset, buffer, pulled, data.Buffer.Count);
                pulled += data.Buffer.Count;
                currentOffset += (long)data.Buffer.Count;
            }
            return new ByteData(new ArraySegment<byte>(buffer, 0, pulled));
        }
        public SourceStatus QueryStatus()
        {
            SourceStatus sourceStatus = new SourceStatus();
            foreach (var i in Elements)
            {
                foreach (var range in i.Source.QueryStatus().AvailableRangeList)
                {
                    sourceStatus.AvailableRangeList.MergingAdd(new Range(range.Offset + i.Offset, range.Size));
                }
            }
            return sourceStatus;
        }
    }

    // 指定した Sink が IsFilled == true の場合に指定した Source が有効になる ISource
    public class SinkLinkedSource : ISource
    {
        public long Size { get; private set; }

        private ISink m_sink;
        private ISource m_source;

        public SinkLinkedSource(ISink sink, ISource source)
        {
            m_sink = sink;
            m_source = source;
            Size = source.Size;
        }
        public ByteData PullData(long offset, int size)
        {
            if (!m_sink.QueryStatus().IsFilled)
            {
                return new ByteData(new ArraySegment<byte>());
            }
            return m_source.PullData(offset, size);
        }
        public SourceStatus QueryStatus()
        {
            SourceStatus sourceStatus = new SourceStatus();
            if (m_sink.QueryStatus().IsFilled)
            {
                sourceStatus.AvailableRangeList.MergingAdd(new Range(0, m_source.Size));
            }
            return sourceStatus;
        }
    }

    // 指定したアラインメントのオフセット・サイズでしか読み出せない ISource。
    // アラインメントを満たさないオフセットからの読出しは例外となる。
    // アラインメントを満たさないサイズを指定した場合、読出しサイズが調整される。
    // ISource の末尾がアラインメントに満たない場合、末尾まで読む場合に限って
    // サイズのアラインメントチェックがされず、読めた分だけ出力する。
    public class AlignedSource : ISource
    {
        public long Size { get { return m_source.Size; } }
        public int AlignmentSize { get; private set; }

        private ISource m_source;

        public AlignedSource(ISource source, int alignmentSize)
        {
            AlignmentSize = alignmentSize;
            m_source = source;
        }
        public ByteData PullData(long offset, int size)
        {
            CheckAlignment(offset);
            int pullSize = size;
            if (offset + pullSize < Size)
            {
                if ((pullSize & (AlignmentSize - 1)) != 0)
                {
                    pullSize = (pullSize / AlignmentSize) * AlignmentSize;
                }
            }
            ByteData data = m_source.PullData(offset, pullSize);
            if (offset + data.Buffer.Count < Size)
            {
                CheckAlignment(data.Buffer.Count);
            }
            return data;
        }
        public SourceStatus QueryStatus()
        {
            SourceStatus sourceStatus = new SourceStatus();
            // ベースの ISource の AvalilableRange をもとにアラインメント調整
            foreach (var range in m_source.QueryStatus().AvailableRangeList)
            {
                long newOffset = range.Offset;
                long newSize = range.Size;
                if ((newOffset & (AlignmentSize - 1)) != 0)
                {
                    newOffset = ((newOffset + AlignmentSize) / AlignmentSize) * AlignmentSize;
                    newSize -= newOffset - range.Offset;
                }
                if (newOffset + newSize != Size)
                {
                    if (newSize < AlignmentSize)
                    {
                        continue;
                    }
                    else if ((newSize & (AlignmentSize - 1)) != 0)
                    {
                        newSize = (newSize / AlignmentSize) * AlignmentSize;
                    }
                }
                sourceStatus.AvailableRangeList.MergingAdd(new Range(newOffset, newSize));
            }
            return sourceStatus;
        }
        private void CheckSize(long value)
        {
            // アラインメント違反は例外に
            if (value < (long)AlignmentSize)
            {
                throw new ArgumentException();
            }
        }
        private void CheckAlignment(long value)
        {
            // アラインメント違反は例外に
            if ((value & (AlignmentSize - 1)) != 0)
            {
                throw new ArgumentException();
            }
        }
    }

    // 指定した ISource を、指定した keyData の AES-CTR モードで暗号化しながら読み出す ISource
    public class Aes128CtrEncryptedSource : ISource
    {
        public long Size { get { return m_source.Size; } }
        public int BlockSize { get; private set; }
        public uint SecureValue { get; private set; }
        public uint Generation { get; private set; }
        public long BaseOffset { get; private set; }

        private ISource m_source;
        private ICtrModeEncryptor m_crypto;

        public Aes128CtrEncryptedSource(ISource source, ICtrModeEncryptor encryptor, uint secureValue, uint generation, long baseOffset)
        {
            m_crypto = encryptor;
            BlockSize = 16;
            SecureValue = secureValue;
            Generation = generation;
            BaseOffset = baseOffset;
            m_source = new AlignedSource(source, BlockSize);
        }
        public ByteData PullData(long offset, int size)
        {
            ByteData data = m_source.PullData(offset, size);
#if false // true にすると暗号化無し
            return data;
#else
            if (data.Buffer.Count == 0)
            {
                return new ByteData(new ArraySegment<byte>());
            }

            byte[] keyStream = SetupKeyStream(BaseOffset, BlockSize, m_crypto.KeySize, SecureValue, Generation, offset);
            byte[] buffer = new byte[data.Buffer.Count];
            m_crypto.EncryptBlock(keyStream, data.Buffer.Array, data.Buffer.Offset, data.Buffer.Count, buffer, 0);
            return new ByteData(new ArraySegment<byte>(buffer, 0, data.Buffer.Count));
#endif
        }
        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
        public static byte[] SetupKeyStream(long baseOffset, int blockSize, int keySize, uint secureValue, uint generation, long offset)
        {
            ulong ctr = (ulong)((baseOffset + offset) / blockSize);
            byte[] keyStream = new byte[keySize];
            for (int i = 0; i < 4; i++)
            {
                keyStream[3 - i] = (byte)((secureValue >> (i * 8)) & 0xFF);
            }
            for (int i = 0; i < 4; i++)
            {
                keyStream[7 - i] = (byte)((generation >> (i * 8)) & 0xFF);
            }
            for (int i = 0; i < 8; i++)
            {
                keyStream[15 - i] = (byte)((ctr >> (i * 8)) & 0xFF);
            }
            return keyStream;
        }
    }

    // 指定した ISource を、指定した keyData の AES-CTR モードで暗号化しながら読み出す ISource
    public class Aes128CtrExEncryptedSource : ISource
    {
        public long Size { get { return m_source.Size; } }
        public int BlockSize { get; private set; }
        public long BaseOffset { get; private set; }
        public uint SecureValue { get; private set; }
        public AesCtrExGenerationTable GenerationTable { get; private set; }
        public uint GenerationForOutOfTable { get; private set; }

        private ISource m_source;
        private ICtrModeEncryptor m_crypto;

        public Aes128CtrExEncryptedSource(ISource source, ICtrModeEncryptor encryptor, uint secureValue,
            AesCtrExGenerationTable generationTable, uint generationForOutOfTable, long baseOffset)
        {
            m_crypto = encryptor;
            BlockSize = 16;
            SecureValue = secureValue;
            GenerationTable = generationTable;
            GenerationForOutOfTable = generationForOutOfTable;
            BaseOffset = baseOffset;
            m_source = new AlignedSource(source, BlockSize);
        }
        public ByteData PullData(long offset, int size)
        {
            ByteData data = m_source.PullData(offset, size);
#if false // true にすると暗号化無し
            return data;
#else
            if (data.Buffer.Count == 0)
            {
                return new ByteData(new ArraySegment<byte>());
            }

            byte[] encryptedData = new byte[data.Buffer.Count];

            // 該当する範囲の世代を取得 (テーブルの外の領域は GenerationForOutOfTable になる)
            var generationEntries = GenerationTable.GetGenerations(offset, size, GenerationForOutOfTable);

            int encryptedSize = 0;
            foreach (var entry in generationEntries)
            {
                byte[] keyStream = Aes128CtrEncryptedSource.SetupKeyStream(BaseOffset, BlockSize, m_crypto.KeySize,
                    SecureValue, entry.generation, offset + encryptedSize);
                m_crypto.EncryptBlock(keyStream, data.Buffer.Array, data.Buffer.Offset + encryptedSize, entry.size, encryptedData, encryptedSize);

                encryptedSize += entry.size;
            }

            if (encryptedSize != data.Buffer.Count)
            {
                throw new Exception("encrypted Size != data.Buffer.Count");
            }

            return new ByteData(new ArraySegment<byte>(encryptedData, 0, data.Buffer.Count));
#endif
        }
        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
    }

    // 指定した ISource を、指定した keyData の AES-XTS モードで暗号化しながら読み出す ISource
    public class Aes128XtsEncryptedSource : ISource
    {
        public long Size { get { return m_source.Size; } }
        public int BlockSize { get; private set; }
        public long BaseOffset { get; private set; }

        private ISource m_source;
        private IXtsModeEncryptor m_crypto;

        public Aes128XtsEncryptedSource(ISource source, IXtsModeEncryptor encryptor)
            : this(source, encryptor, 0)
        {
        }
        public Aes128XtsEncryptedSource(ISource source, IXtsModeEncryptor encryptor, long baseOffset)
        {
            m_crypto = encryptor;
            BlockSize = 512;
            BaseOffset = baseOffset;
            m_source = new AlignedSource(source, BlockSize);
        }
        public ByteData PullData(long offset, int size)
        {
            ByteData data = m_source.PullData(offset, size);
            if (data.Buffer.Count % BlockSize != 0)
            {
                throw new InvalidOperationException();
            }
            if (offset % BlockSize != 0)
            {
                throw new InvalidOperationException();
            }
#if false // true にすると暗号化無し
            return data;
#else
            if (data.Buffer.Count == 0)
            {
                return new ByteData(new ArraySegment<byte>());
            }
            byte[] buffer = new byte[data.Buffer.Count];
            for (int i = 0; i < data.Buffer.Count; i += BlockSize)
            {
                m_crypto.EncryptBlock(GetInitialTweak(BaseOffset + offset + i), data.Buffer.Array, data.Buffer.Offset + i, BlockSize, buffer, i);
            }
            return new ByteData(new ArraySegment<byte>(buffer, 0, data.Buffer.Count));
#endif
        }
        private byte[] GetInitialTweak(long offset)
        {
            ulong offsetIndex = (ulong)(offset / BlockSize);
            byte[] tweak = new byte[m_crypto.KeySize];
            for (int i = 0; i < 8; i++)
            {
                tweak[tweak.Length - 1 - i] = (byte)((offsetIndex >> i * 8) & 0xFF);
            }
            return tweak;
        }
        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
        public static byte[] GetDefaultKey(int index)
        {
            return Aes128XtsCryptoDriver.GetDefaultKey(index);
        }
    }

    // 指定した ISource の全域を、RSA2048+SHA256 で署名する ISource
    // 署名データは、GetSignatureValueSource() で取得可能
    public class Rsa2048PssSha256SignedSource : ISource
    {
        public long Size { get { return m_source.Size; } }

        private ISource m_source;
        private ISigner[] m_signer = new ISigner[2];
        private MemorySink[] m_rsaValueSink = new MemorySink[2];

        public Rsa2048PssSha256SignedSource(ISource source, ISigner signer) : this(source, signer, null)
        {
        }

        public Rsa2048PssSha256SignedSource(ISource source, ISigner signer1, ISigner signer2)
        {
            m_signer[0] = signer1;
            m_signer[1] = signer2;
            m_source = source;
            m_rsaValueSink[0] = new MemorySink(Rsa2048PssSha256SignCryptoDriver.KeySize);
            m_rsaValueSink[1] = new MemorySink(Rsa2048PssSha256SignCryptoDriver.KeySize);
        }
        public ByteData PullData(long offset, int size)
        {
            // TORIAEZU: 分割読出しを禁止
            if (offset != 0 || (long)size < Size)
            {
                throw new ArgumentException();
            }
            ByteData data = m_source.PullData(offset, size);
            if (data.Buffer.Count != Size)
            {
                throw new InvalidOperationException();
            }

            for (int i = 0; i < 2; i++)
            {
                byte[] sign;
                if (m_signer[i] != null)
                {
                    sign = m_signer[i].SignBlock(data.Buffer.Array, data.Buffer.Offset, data.Buffer.Count);
#if false
                    if (!m_signer[i].VerifyBlock(data.Buffer.Array, data.Buffer.Offset, data.Buffer.Count, sign)) // 念のため確認
                    {
                        throw new InvalidOperationException();
                    }
#endif
                }
                else
                {
                    sign = new Byte[m_rsaValueSink[i].Size];
                }
                m_rsaValueSink[i].PushData(new ByteData(new ArraySegment<byte>(sign, 0, sign.Length)), 0);
            }

            return data;
        }
        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
        public ISource GetSignatureValueSource()
        {
            return GetSignatureValueSource(0);
        }
        public ISource GetSignatureValueSource(int index)
        {
            if (m_signer[index] != null)
            {
                return new SinkLinkedSource(m_rsaValueSink[index], m_rsaValueSink[index].ToSource());
            }
            else
            {
                return new PaddingSource(Rsa2048PssSha256SignCryptoDriver.KeySize);
            }
        }
    }

    // 指定した ISource を、指定した階層数・ブロックサイズで Sha256 ハッシュを計算しながら読み出す ISource
    // ハッシュ計算結果は、GetLayerHashSource(), GetMasterHashSource() で取得可能
    public class Sha256HierarchicalHashCalculatedSource : IHierarchicalHashCalculatedSource
    {
        public long Size { get; private set; }

        private int m_blockSize;
        private int m_hashLayerCount;

        private ISource m_source;
        private List<MemorySink> m_hierarchicalHashValueSinkList;

        private SHA256CryptoServiceProvider m_hashCalculator;
        private int HashSize = 32;

        public Sha256HierarchicalHashCalculatedSource(ISource source, int blockSize, int hashLayerCount)
        {
            Size = source.Size;

            m_blockSize = blockSize;
            if (hashLayerCount < 1 || hashLayerCount > 2) // 1 or 2
            {
                throw new ArgumentException();
            }
            m_hashLayerCount = hashLayerCount;

            m_source = new AlignedSource(source, blockSize);
            m_hierarchicalHashValueSinkList = new List<MemorySink>();

            // TODO: 3 段目以降に対応
            m_hierarchicalHashValueSinkList.Add(new MemorySink(HashSize)); // master hash
            if (hashLayerCount == 2)
            {
                int layer1HashValueSize = Math.Min(HashSize * (int)((source.Size + (blockSize - 1)) / blockSize), blockSize);
                m_hierarchicalHashValueSinkList.Add(new MemorySink(layer1HashValueSize));
            }

            m_hashCalculator = new SHA256CryptoServiceProvider();
        }
        private void CalculateAndStoreHash(ByteData srcData, long dstOffset, int hashLayerCount)
        {
            if (m_hierarchicalHashValueSinkList[hashLayerCount].QueryStatus().IsFilled)
            {
                throw new InvalidOperationException();
            }
            int pushed = 0;
            for (int i = 0; i < srcData.Buffer.Count; i += m_blockSize)
            {
                // ブロックサイズに満たない末尾部分は、1 バイト単位で末尾までのサイズで計算
                int hashTargetSize = Math.Min(srcData.Buffer.Count - i, m_blockSize);
                byte[] hash = m_hashCalculator.ComputeHash(srcData.Buffer.Array, srcData.Buffer.Offset + i, hashTargetSize);
                pushed += m_hierarchicalHashValueSinkList[hashLayerCount].PushData(new ByteData(new ArraySegment<byte>(hash, 0, hash.Length)), dstOffset + pushed);
            }
            if (m_hierarchicalHashValueSinkList[hashLayerCount].QueryStatus().IsFilled)
            {
                if (hashLayerCount == 0)
                {
                    return;
                }
                else
                {
                    ISource source = m_hierarchicalHashValueSinkList[hashLayerCount].ToSource();
                    ByteData data = source.PullData(0, (int)source.Size);
                    CalculateAndStoreHash(data, 0, hashLayerCount - 1);
                }
            }
        }
        public ByteData PullData(long offset, int size)
        {
            ByteData data = m_source.PullData(offset, size);
            long hashOffset = (long)HashSize * offset / m_blockSize;
            CalculateAndStoreHash(data, hashOffset, m_hashLayerCount - 1);
            return data;
        }
        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
        public ISource GetMasterHashSource()
        {
            return new SinkLinkedSource(m_hierarchicalHashValueSinkList[0], m_hierarchicalHashValueSinkList[0].ToSource());
        }
        public ISource GetLayerHashSource()
        {
            if (m_hashLayerCount == 1)
            {
                return GetMasterHashSource();
            }

            List<ConcatenatedSource.Element> elements = new List<ConcatenatedSource.Element>();
            long offset = 0;
            for (int i = 1; i < m_hashLayerCount; i++)
            {
                ConcatenatedSource.Element hierarchicalHashElement = new ConcatenatedSource.Element(
                    new SinkLinkedSource(m_hierarchicalHashValueSinkList[i], m_hierarchicalHashValueSinkList[i].ToSource()),
                    "Hash:L" + i, offset);
                offset += m_hierarchicalHashValueSinkList[i].Size;
                elements.Add(hierarchicalHashElement);
            }
            return new ConcatenatedSource(elements);
        }
        public ISource GetFsHeaderSource(NintendoContentFileSystemInfo.EntryInfo entryInfo)
        {
            var fsHeader = NintendoContentFileSystemMeta.CreateNcaFsHeader(entryInfo);
            var masterHashOffset = NintendoContentFileSystemMeta.HierarchicalSha256MasterHashOffset;
            var masterHashSource = GetMasterHashSource();

            return new AdaptedSource(
                new MemorySource(fsHeader, 0, fsHeader.Length), masterHashSource, masterHashOffset, masterHashSource.Size);
        }
    }

    // ISource に、指定した別の ISource を埋め込んで出力する ISource
    // 指定した ISource が利用できない場合、当該 ISource も利用できない
    // 埋め込む ISource をリストで指定した場合、リストの順に埋め込みが適用される（範囲の重複も可）
    public class AdaptedSource : ISource
    {
        public long Size { get; private set; }

        private ISource m_baseSource;
        public List<Tuple<ISource, long, long>> m_adaptSourceInfos;

        public AdaptedSource(ISource baseSource, ISource adaptSource, long offset, long size) : this(baseSource, new List<Tuple<ISource, long, long>>(){Tuple.Create(adaptSource, offset, size)})
        {
        }
        public AdaptedSource(ISource baseSource, List<Tuple<ISource, long, long>> adaptSourceInfos)
        {
            Size = baseSource.Size;
            m_baseSource = baseSource;
            foreach (var info in adaptSourceInfos)
            {
                var adaptSource = info.Item1;
                var offset = info.Item2;
                var size = info.Item3;
                if (offset < 0 || offset > Size || offset + size > Size || adaptSource.Size < size)
                {
                    throw new ArgumentException();
                }
            }
            m_adaptSourceInfos = adaptSourceInfos;
        }
        public ByteData PullData(long offset, int size)
        {
            ByteData data = m_baseSource.PullData(offset, size);

            foreach (var info in m_adaptSourceInfos)
            {
                var adaptSource = info.Item1;
                var adaptSourceOffset = info.Item2;
                var adaptSourceSize = info.Item3;

                if (offset >= (adaptSourceOffset + adaptSourceSize) || (offset + data.Buffer.Count) <= adaptSourceOffset)
                {
                    continue;
                }

                ByteData adaptData = adaptSource.PullData(0, (int)adaptSourceSize);
                if (adaptData.Buffer.Count == 0)
                {
                    // adaptSource がひとつでも利用不可な場合、空データを返す
                    return new ByteData(new ArraySegment<byte>());
                }
                if (adaptData.Buffer.Count != adaptSourceSize)
                {
                    throw new InvalidOperationException();
                }

                bool isEndInclude = (adaptSourceOffset + adaptSourceSize) <= (offset + data.Buffer.Count);
                bool isOffsetInclude = adaptSourceOffset >= offset;
                int baseOffset = 0;
                int adaptOffset = 0;
                int adaptSize = 0;
                if (isOffsetInclude && isEndInclude)
                {
                    baseOffset = (int)(adaptSourceOffset - offset);
                    adaptOffset = 0;
                    adaptSize = (int)adaptSourceSize;
                }
                else if (isOffsetInclude && !isEndInclude)
                {
                    baseOffset = (int)(adaptSourceOffset - offset);
                    adaptOffset = 0;
                    adaptSize = (int)(offset + data.Buffer.Count - adaptSourceOffset);
                }
                else if (!isOffsetInclude && isEndInclude)
                {
                    baseOffset = 0;
                    adaptOffset = (int)(offset - adaptSourceOffset);
                    adaptSize = (int)(adaptSourceSize - adaptOffset);
                }
                else if (!isOffsetInclude && !isEndInclude)
                {
                    baseOffset = 0;
                    adaptOffset = (int)(offset - adaptSourceOffset);
                    adaptSize = data.Buffer.Count;
                }
                Buffer.BlockCopy(adaptData.Buffer.Array, adaptData.Buffer.Offset + adaptOffset,
                                 data.Buffer.Array, data.Buffer.Offset + baseOffset, adaptSize);
            }

            return data;
        }
        public SourceStatus QueryStatus()
        {
            bool isAvailable = true;
            foreach (var info in m_adaptSourceInfos)
            {
                var adaptSource = info.Item1;
                RangeList rangeList = adaptSource.QueryStatus().AvailableRangeList;
                if (rangeList.Count != 1 || rangeList[0].Size != adaptSource.Size)
                {
                    isAvailable = false;
                }
            }

            if (isAvailable)
            {
                return m_baseSource.QueryStatus();
            }
            else
            {
                return new SourceStatus();
            }
        }
    }

    public class CliCompatibleSource : ISource, FileSystemMetaLibrary.SourceInterface
    {
        public long Size { get; private set; }
        public ISource Source { get; private set; }
        public CliCompatibleSource(ISource source)
        {
            if (source is CliCompatibleSource)
            {
                Source = ((CliCompatibleSource)source).Source;
            }
            else
            {
                Source = source;
            }

            Size = source.Size;
        }
        public ByteData PullData(long offset, int size)
        {
            return Source.PullData(offset, size);
        }
        public SourceStatus QueryStatus()
        {
            return Source.QueryStatus();
        }
    }

    public class SubSource : ISource
    {
        public long Size { get; private set; }
        private ISource m_source;
        private long m_offset;
        public SubSource(ISource source, long offset, long size)
        {
            // ここでは offset, size のチェックはしない
            m_source = source;
            m_offset = offset;
            Size = size;
        }
        public ByteData PullData(long offset, int size)
        {
            int pullSize = size;
            if (offset + pullSize > Size)
            {
                int restSize = (int)(Size - offset);
                pullSize = restSize > 0 ? restSize : 0;
            }
            return m_source.PullData(m_offset + offset, pullSize);
        }
        public SourceStatus QueryStatus()
        {
            SourceStatus sourceStatus = new SourceStatus();
            RangeList availableList = m_source.QueryStatus().AvailableRangeList;
            RangeList removedList = new RangeList();
            removedList.Add(new Range(0, m_offset));
            removedList.Add(new Range(m_offset + Size, m_source.Size - (m_offset + Size)));
            RangeList subAvailableList = availableList.GetDuplicatedDeletedList(removedList);
            foreach (var range in subAvailableList)
            {
                sourceStatus.AvailableRangeList.MergingAdd(new Range(range.Offset - m_offset, range.Size));
            }
            return sourceStatus;
        }

        public void GetInternalInformation(out ISource source, out long offset)
        {
            source = m_source;
            offset = m_offset;
        }
    }

    // 指定した ISource 全体の SHA256 ハッシュを計算した結果を読みだす ISource
    // 初回 PullData 時にハッシュを計算する。
    // ISource 全体が読み出せない場合はこの ISource も利用できない
    public class Sha256StreamHashSource : ISource
    {
        public long Size { get; private set; }

        private ISource m_source;
        private SHA256CryptoServiceProvider m_hashCalculator;
        private SourceStatus m_status;
        private byte[] m_hash;

        public Sha256StreamHashSource(ISource source)
        {
            Size = 32;
            m_source = source;
            m_hashCalculator = new SHA256CryptoServiceProvider();
            m_status = new SourceStatus();
            m_status.AvailableRangeList.MergingAdd(new Range(0, Size));
        }
        public ByteData PullData(long offset, int size)
        {
            if (m_hash == null)
            {
                using (Stream stream = new SourceBasedStream(m_source))
                {
                    stream.Seek(0, SeekOrigin.Begin);
                    m_hash = m_hashCalculator.ComputeHash(stream);
                }
            }
            ISource source = new MemorySource(m_hash, 0, m_hash.Length);
            return source.PullData(offset, size);
        }
        public SourceStatus QueryStatus()
        {
            return m_status;
        }
    }

    // 指定した ISource のデータを 16 進数文字列のバイト列に変換する
    public class HexStringConvertedSource : ISource
    {
        public long Size { get; private set; }
        private ISource m_source;

        public HexStringConvertedSource(ISource source)
        {
            Size = source.Size * 2;
            m_source = source;
        }
        public ByteData PullData(long offset, int size)
        {
            if (size % 2 != 0)
            {
                throw new ArgumentException();
            }
            ByteData data = m_source.PullData(offset, size / 2);
            byte[] buffer = new byte[data.Buffer.Count];
            Buffer.BlockCopy(data.Buffer.Array, data.Buffer.Offset, buffer, 0, data.Buffer.Count);
            string str = BitConverter.ToString(buffer).Replace("-", string.Empty);
            byte[] converted = Encoding.ASCII.GetBytes(str);
            return new ByteData(new ArraySegment<byte>(converted, 0, converted.Length));
        }
        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
    }

    // FileSystemArchive の指定したファイルを読み出す ISource
    public class FileSystemArchvieFileSource : ISource
    {
        public long Size { get; private set; }

        private IFileSystemArchiveReader m_Reader;
        private string m_FileName;
        private SourceStatus m_status;

        public FileSystemArchvieFileSource(IFileSystemArchiveReader reader, string fileName)
        {
            m_Reader = reader;
            m_FileName = fileName;
            Size = m_Reader.GetFileSize(m_FileName);
            m_status = new SourceStatus();
            m_status.AvailableRangeList.MergingAdd(new Range(0, Size));
        }
        public ByteData PullData(long offset, int size)
        {
            int pullSize = size;
            if (offset + pullSize > Size)
            {
                int restSize = (int)(Size - offset);
                pullSize = restSize > 0 ? restSize : 0;
            }
            var buffer = m_Reader.ReadFile(m_FileName, offset, pullSize);
            return new ByteData(new ArraySegment<byte>(buffer, 0, buffer.Length));
        }
        public SourceStatus QueryStatus()
        {
            return m_status;
        }
    }

    // FileSystemArchive 全体を読み出す ISource
    public class FileSystemArchvieBaseSource : ISource
    {
        public long Size { get; private set; }

        private IFileSystemArchiveReader m_Reader;
        private SourceStatus m_status;

        public FileSystemArchvieBaseSource(IFileSystemArchiveReader reader)
        {
            m_Reader = reader;
            Size = m_Reader.GetBaseSize();
            m_status = new SourceStatus();
            m_status.AvailableRangeList.MergingAdd(new Range(0, Size));
        }
        public ByteData PullData(long offset, int size)
        {
            int pullSize = size;
            if (offset + pullSize > Size)
            {
                int restSize = (int)(Size - offset);
                pullSize = restSize > 0 ? restSize : 0;
            }
            var buffer = m_Reader.ReadBase(offset, pullSize);
            return new ByteData(new ArraySegment<byte>(buffer, 0, buffer.Length));
        }
        public SourceStatus QueryStatus()
        {
            return m_status;
        }
        public IFileSystemArchiveReader GetReader()
        {
            return m_Reader;
        }
    }

    // StorageArchive を読み出す ISource
    public class StorageArchiveSource : ISource
    {
        public long Size { get; private set; }

        private IStorageArchiveReader m_Reader;
        private SourceStatus m_status;

        public StorageArchiveSource(IStorageArchiveReader reader)
        {
            m_Reader = reader;
            Size = m_Reader.GetSize();
            m_status = new SourceStatus();
            m_status.AvailableRangeList.MergingAdd(new Range(0, Size));
        }

        public ByteData PullData(long offset, int size)
        {
            int readSize = SourceUtil.GetReadableSize(this.Size, offset, size);
            if (readSize == 0)
            {
                return new ByteData(new ArraySegment<byte>());
            }
            var buffer = m_Reader.Read(offset, readSize);
            return new ByteData(new ArraySegment<byte>(buffer, 0, buffer.Length));
        }

        public SourceStatus QueryStatus()
        {
            return m_status;
        }
    }

    // パッチ情報を付加した FileSystemArchive 全体を読み出す SourceInterface
    public class FileSystemArchviePatchSource : SourceInterface
    {
        public struct PatchInfo
        {
            public byte[] originalHeader;
            public byte[] currentHeader;
            public IStorageArchiveReader originalStorageReader;
            public IStorageArchiveReader currentStorageReader;
            public IFileSystemArchiveReader originalFsReader;
            public IFileSystemArchiveReader currentFsReader;
            public string[] binaryMatchHintPaths;
            public IndirectStorageStream.BinaryMatchHint[] binaryMatchHints;
        }

        public PatchInfo ExtraInfo { get; private set; }

        public FileSystemArchviePatchSource(PatchInfo info)
        {
            ExtraInfo = info;
        }
    }

    // パッチ最適化のための情報を付加した ISourceInterface
    public class FileSystemArchiveIscPatchSource : SourceInterface
    {
        public struct IscPatchInfo
        {
            public byte[] previousRawEntryInfo;
            public Int64 previousFsStartOffset;
            public ISource previousSource;
            public ISource previousAesCtrExTable;
            public byte[] currentRawEntryInfo;
            public Int64 currentFsStartOffset;
            public ISource currentSource;
        }

        public IscPatchInfo ExtraInfo { get; private set; }

        public FileSystemArchiveIscPatchSource(IscPatchInfo info)
        {
            ExtraInfo = info;
        }
    }

    // スパース化のための情報を付加した SourceInterface
    public class FileSystemArchiveSparseSource : SourceInterface
    {
        public struct SparseInfo
        {
            public NintendoContentArchiveReader originalReader;
            public NintendoContentArchiveReader patchReader;
        }

        public SparseInfo ExtraInfo { get; private set; }

        public FileSystemArchiveSparseSource(SparseInfo info)
        {
            ExtraInfo = info;
        }
    }

    internal class IntegrityHierarchicalMasterHashSource : ISource
    {
        public long Size { get; private set; }

        private IntegrityHierarchicalStorageSink m_sink;

        public IntegrityHierarchicalMasterHashSource(IntegrityHierarchicalStorageSink sink)
        {
            m_sink = sink;
            Size = sink.MasterHashSize;
        }
        public ByteData PullData(long offset, int size)
        {
            var pullSize = SourceUtil.GetReadableSize(Size, offset, size);
            return m_sink.GetMasterHash(offset, pullSize);
        }
        public SourceStatus QueryStatus()
        {
            // SinkLinkedSource を利用するので実装しない
            throw new NotSupportedException();
        }
    }

    internal class IntegrityHierarchicalLayerHashSource : ISource
    {
        public long Size { get; private set; }

        private IntegrityHierarchicalStorageSink m_sink;

        public IntegrityHierarchicalLayerHashSource(IntegrityHierarchicalStorageSink sink)
        {
            m_sink = sink;
            Size = sink.LayerHashSize;
        }
        public ByteData PullData(long offset, int size)
        {
            var pullSize = SourceUtil.GetReadableSize(Size, offset, size);
            return m_sink.GetLayerHash(offset, pullSize);
        }
        public SourceStatus QueryStatus()
        {
            // SinkLinkedSource を利用するので実装しない
            throw new NotSupportedException();
        }
    }

    // 指定した ISource を階層的完全性検証のハッシュを計算しながら読み出す ISource
    public class IntegrityHierarchicalHashCalculatedSource : IHierarchicalHashCalculatedSource
    {
        public long Size { get; private set; }

        private const int IntegrityHashLayerBlockSize = 1024 * 4;

        private ISource m_source;
        private IntegrityHierarchicalStorageSink m_sink;

        public IntegrityHierarchicalHashCalculatedSource(ISource source)
        {
            Size = source.Size;
            m_source = new AlignedSource(source, HierarchicalIntegrityVerificationStorage.GetHashBlockSize());
            m_sink = new IntegrityHierarchicalStorageSink(Size);
        }
        private void CalculateHash(ByteData sourceData, long offset, int size)
        {
            if (m_sink.QueryStatus().IsFilled)
            {
                throw new InvalidOperationException();
            }
            m_sink.PushData(sourceData, offset);
        }
        public ByteData PullData(long offset, int size)
        {
            ByteData data = m_source.PullData(offset, size);
            CalculateHash(data, offset, data.Buffer.Count);
            return data;
        }
        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
        public ISource GetMasterHashSource()
        {
            return new SinkLinkedSource(m_sink, new IntegrityHierarchicalMasterHashSource(m_sink));
        }
        public ISource GetLayerHashSource()
        {
            return new SinkLinkedSource(m_sink, new IntegrityHierarchicalLayerHashSource(m_sink));
        }
        public byte[] GetMetaInfo()
        {
            return m_sink.GetMetaInfo();
        }
        public ISource GetFsHeaderSource(NintendoContentFileSystemInfo.EntryInfo entryInfo)
        {
            var fsHeader = NintendoContentFileSystemMeta.CreateNcaFsHeader(entryInfo);
            var masterHashOffset = NintendoContentFileSystemMeta.IntegrityMasterHashOffset;
            var masterHashSource = GetMasterHashSource();

            return new AdaptedSource(
                new MemorySource(fsHeader, 0, fsHeader.Length), masterHashSource, masterHashOffset, masterHashSource.Size);

        }
    }

    // IndirectStorageStream からデータを読み出す ISource
    internal class IndirectStorageDataSource : ISource
    {
        private SourceStatus m_Status;
        private IndirectStorageStream m_Stream;

        public IndirectStorageDataSource(IndirectStorageStream stream)
        {
            Size = stream.GetDataSize();
            m_Stream = stream;
            m_Status = new SourceStatus();
            m_Status.AvailableRangeList.MergingAdd(new Range(0, Size));
        }

        public long Size { get; private set; }

        public ByteData PullData(long offset, int size)
        {
            int readSize = SourceUtil.GetReadableSize(this.Size, offset, size);
            if (readSize == 0)
            {
                return new ByteData(new ArraySegment<byte>());
            }

            var buffer = m_Stream.ReadData(offset, readSize);
            return new ByteData(new ArraySegment<byte>(buffer, 0, buffer.Length));
        }

        public SourceStatus QueryStatus()
        {
            return m_Status;
        }
    }

    // IndirectStorage 向けのソース
    internal class IndirectStorageReadOnlySource : IReadOnlySource
    {
        private ISource m_Source;
        private int m_BlockSize;

        public IndirectStorageReadOnlySource(ISource source, int blockSize)
        {
            m_Source = source;
            m_BlockSize = blockSize;
        }

        public int Read(IntPtr address, long offset, int size)
        {
            var readSize = SourceUtil.GetReadableSize(m_Source.Size, offset, size);
            var data = m_Source.PullData(offset, readSize);
            readSize = Math.Min(readSize, data.Buffer.Count());
            Marshal.Copy(data.Buffer.Array, data.Buffer.Offset, address, readSize);
            return readSize;
        }

        public long GetSize()
        {
            return (m_Source.Size + (m_BlockSize - 1)) / m_BlockSize * m_BlockSize;
        }
    }

    // IndirectStorage の進捗を書き出すクラス
    public class IndirectStorageProgressWriter : IDisposable
    {
        private static readonly string[] Phase = new string[5] { "Making match data", "File base matching", "Binary matching 1st pass", "Binary matching 2nd pass", "Optimizing" };

        private string[] m_Paths;

        public IndirectStorageProgressWriter()
        {
            m_Paths = new string[0];
        }

        public IndirectStorageProgressWriter(string[] paths)
        {
            if (paths == null)
            {
                m_Paths = new string[0];
            }
            else
            {
                m_Paths = paths;
            }
        }

        public void Write(IndirectStorageProgress progress)
        {
            string output = string.Format("[{0}] {1} ... {2:0.00%}", DateTime.Now.ToString("HH:mm:ss"), Phase[(int)progress.phase], progress.rate);

            if (progress.phase == IndirectStorageProgress.Phase.FileBaseMatch && (int)progress.value < m_Paths.Length)
            {
                output += string.Format(" ({0})", m_Paths[(int)progress.value]);
            }

            Log.Progress(output);
        }

        public void Dispose()
        {
            Log.Progress("Match completed ... 100%");
        }
    }

    // 2 つのソースから差分データとそのテーブル情報を生成する ISource
    public class IndirectStorageSource : ISource, IDisposable
    {
        public enum CacheType
        {
            None,
            Memory,
            File
        }

        public const int BlockSize = 16;
        public const int MinimumRegionSize = IndirectStorageStream.RegionSizeMin;
        public const int DefaultRegionSize = 16 * 1024;
        public const long WindowSize = long.MaxValue;

        public static CacheType DataCacheType = CacheType.Memory;
        public static int ShiftSize = BlockSize;
        public static int RegionSize = DefaultRegionSize;
        public static int MatchSize = DefaultRegionSize * 2;

        private IndirectStorageStream m_Stream;
        private ConcatenatedSource m_Source;
        private bool m_WillUseCache;

        public IndirectStorageSource(bool willUseCache)
        {
            m_Stream = null;
            m_Source = null;
            m_WillUseCache = willUseCache;
        }

        public static void SetStronglyOptimizeSize(int shiftSize)
        {
            // 2 のべき乗
            if ((shiftSize & (shiftSize - 1)) != 0)
            {
                throw new ArgumentException("strongly optimize size should be power of 2.");
            }
            if (shiftSize <= 0 || BlockSize < shiftSize)
            {
                throw new ArgumentException(string.Format("strongly optimize size smaller than or equal to {0} bytes.", BlockSize));
            }
            ShiftSize = shiftSize;
        }

        public static void SetMinimumMatchingSize(int matchSize)
        {
            // 2 のべき乗
            if ((matchSize & (matchSize - 1)) != 0)
            {
                throw new ArgumentException("match size should be power of 2.");
            }
            if (matchSize < MinimumRegionSize * 2)
            {
                throw new ArgumentException(string.Format("match size should be large than or equal to {0}KiB.", MinimumRegionSize * 2 / 1024));
            }
            MatchSize = matchSize;
            RegionSize = MatchSize / 2;
        }

        public void Build(ISource oldSource, ISource newSource, byte[] originalHeader)
        {
            var oldReader = new IndirectStorageReadOnlySource(oldSource, BlockSize);
            var newReader = new IndirectStorageReadOnlySource(newSource, BlockSize);
            m_Stream = new IndirectStorageStream(oldReader, newReader, BinaryMatchHints);

            if (OldExcludeRange != null && 0 < OldExcludeRange.Count)
            {
                m_Stream.SetExcludeRangeForOldSource(OldExcludeRange.ToArray());
            }
            if (NewExcludeRange != null && 0 < NewExcludeRange.Count)
            {
                m_Stream.SetExcludeRangeForNewSource(NewExcludeRange.ToArray());
            }

            if (NintendoContentArchiveSource.BuildLog.NeedsOutputCache)
            {
                m_Stream.SetBinaryRegionInfo(NintendoContentArchiveSource.BuildLog.OriginalPath, NintendoContentArchiveSource.BuildLog.CacheDirectory, originalHeader);
            }

            using (var writer = new IndirectStorageProgressWriter(BinaryMatchHintPaths))
            {
                Task task = Task.Run(() => m_Stream.Build((uint)ShiftSize, BlockSize, (uint)RegionSize, MatchSize, WindowSize, m_WillUseCache));
                try
                {
                    while (!task.Wait(2000))
                    {
                        Progress = m_Stream.GetProgress();

                        writer.Write(Progress);
                    }
                }
                catch (AggregateException e)
                {
                    throw e.InnerException;
                }
            }

            // RomFs の情報を書き出す
            if (NintendoContentArchiveSource.BuildLog.NeedsOutputLog)
            {
                using (var tableStream = new FileStream(NintendoContentArchiveSource.BuildLog.GetOutputPath(".is_table.txt"), FileMode.Create))
                using (var fragmentStream = new FileStream(NintendoContentArchiveSource.BuildLog.GetOutputPath(".is_fragment.csv"), FileMode.Create))
                {
                    m_Stream.OutputBuildLog(tableStream, fragmentStream, 0);
                }
            }

            var indirectHeader = new byte[m_Stream.GetIndirectHeaderSize()];
            var indirectTable = new byte[m_Stream.GetIndirectTableSize()];
            m_Stream.WriteIndirectTable(indirectHeader, indirectTable);

            IndirectHeader = indirectHeader;
            IndirectTableSize = m_Stream.GetIndirectTableSize();
            IndirectTableOffset = m_Stream.GetDataSize();

            var aesCtrExHeader = new byte[m_Stream.GetAesCtrExHeaderSize()];
            var aesCtrExTable = new byte[m_Stream.GetAesCtrExTableSize()];
            m_Stream.WriteAesCtrExTable(aesCtrExHeader, aesCtrExTable);

            // 暗号化時に使用するテーブルを取得
            GenerationTable = new AesCtrExGenerationTable(aesCtrExHeader, aesCtrExTable);

            AesCtrExHeader = aesCtrExHeader;
            AesCtrExTableSize = m_Stream.GetAesCtrExTableSize();
            AesCtrExTableOffset = IndirectTableOffset + IndirectTableSize;

            var dataSource = new IndirectStorageDataSource(m_Stream);
            var indirectTableSource = new MemorySource(indirectTable, 0, indirectTable.Length);
            var aesCtrExTableSource = new MemorySource(aesCtrExTable, 0, aesCtrExTable.Length);

            var elements = new List<ConcatenatedSource.Element>();
            elements.Add(new ConcatenatedSource.Element(dataSource, "data", 0));
            elements.Add(new ConcatenatedSource.Element(indirectTableSource, "indirectTable", IndirectTableOffset));
            elements.Add(new ConcatenatedSource.Element(aesCtrExTableSource, "aesCtrExTable", AesCtrExTableOffset));

            m_Source = new ConcatenatedSource(elements);

            System.Diagnostics.Debug.Assert(m_Source.Size == m_Stream.GetIndirectTableSize() + m_Stream.GetAesCtrExTableSize() + m_Stream.GetDataSize());
        }

        public long Size { get { return m_Source.Size; } }
        public IndirectStorageProgress Progress { get; private set; } = new IndirectStorageProgress();
        public byte[] IndirectHeader { get; private set; }
        public long IndirectTableSize { get; private set; }
        public long IndirectTableOffset { get; private set; }
        public byte[] AesCtrExHeader { get; private set; }
        public long AesCtrExTableSize { get; private set; }
        public long AesCtrExTableOffset { get; private set; }
        public List<IndirectStorageStream.ExcludeRange> OldExcludeRange { get; set; }
        public List<IndirectStorageStream.ExcludeRange> NewExcludeRange { get; set; }
        public IndirectStorageStream.BinaryMatchHint[] BinaryMatchHints { get; set; }
        public string[] BinaryMatchHintPaths { get; set; }
        public AesCtrExGenerationTable GenerationTable { get; private set; }
        private bool Disposed { get; set; } = false;

        public ByteData PullData(long offset, int size)
        {
            return m_Source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_Source.QueryStatus();
        }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (Disposed)
            {
                return;
            }

            if (disposing)
            {
                // 保持しているマネージドなリソースの解放
                if (m_Stream != null)
                {
                    m_Stream.Dispose();
                }

                if (GenerationTable != null)
                {
                    GenerationTable.Dispose();
                }
            }

            // 解放終わり
            Disposed = true;
        }
    }


    internal class RelocatedIndirectStorageDataSource : ISource
    {
        private SourceStatus m_Status;
        private RelocatedIndirectStorageStream m_Stream;

        public RelocatedIndirectStorageDataSource(RelocatedIndirectStorageStream stream)
        {
            Size = stream.GetDataSize();
            m_Stream = stream;
            m_Status = new SourceStatus();
            m_Status.AvailableRangeList.MergingAdd(new Range(0, Size));
        }

        public long Size { get; private set; }

        public ByteData PullData(long offset, int size)
        {
            int readSize = SourceUtil.GetReadableSize(this.Size, offset, size);
            if (readSize == 0)
            {
                return new ByteData(new ArraySegment<byte>());
            }

            var buffer = m_Stream.ReadData(offset, readSize);
            return new ByteData(new ArraySegment<byte>(buffer, 0, buffer.Length));
        }

        public SourceStatus QueryStatus()
        {
            return m_Status;
        }
    }

    // 前バージョンのパッチと新バージョンのパッチより、最適化(前バージョンと差分が少ないように再配置)されたパッチを生成する
    // 2 つのソースから差分データとそのテーブル情報を生成する ISource
    public class RelocatedIndirectStorageSource : ISource, IDisposable
    {
        public const int BlockSize = IndirectStorageSource.BlockSize;
        public const int DefaultDefragmentMatchSize = 512 * 1024;

        public static int RegionSize = IndirectStorageSource.DefaultRegionSize;
        public static int MatchSize = IndirectStorageSource.DefaultRegionSize * 2;

        private RelocatedIndirectStorageStream m_Stream;
        private ConcatenatedSource m_Source;

        public static void SetDefragmentSize(int defragmentSize)
        {
            if (defragmentSize == 0)
            {
                MatchSize = DefaultDefragmentMatchSize;
            }
            else if (defragmentSize < IndirectStorageSource.MinimumRegionSize * 2)
            {
                throw new ArgumentException(string.Format("defragment size should be large than or equal to {0}KiB.", IndirectStorageSource.MinimumRegionSize * 2 / 1024));
            }
            else
            {
                MatchSize = defragmentSize;
            }

            // 2 の累乗に切り下げ
            RegionSize = (int)Math.Pow(2, Math.Floor(Math.Log(MatchSize / 2, 2)));
            System.Diagnostics.Debug.Assert(RegionSize <= MatchSize / 2);
        }

        public RelocatedIndirectStorageSource()
        {
            m_Stream = null;
            m_Source = null;
        }

        public void Build(byte[] oldFsHeader, long oldFsStartOffset, ISource oldSource, ISource oldAesCtrExTableSource, byte[] newFsHeader, long newFsStartOffset, ISource newSource, long outputFsStartOffset, uint keyGenerationForUpdatedRegion)
        {
            {
                IndirectStorageReadOnlySource newReader = new IndirectStorageReadOnlySource(newSource, BlockSize);
                if (oldFsHeader == null)
                {
                    // 前バージョンの FsEntry がない
                    m_Stream = new RelocatedIndirectStorageStream(null, 0, null, null, newFsHeader, newFsStartOffset, newReader, outputFsStartOffset);
                }
                else
                {
                    IndirectStorageReadOnlySource oldReader = new IndirectStorageReadOnlySource(oldSource, BlockSize);
                    IndirectStorageReadOnlySource oldAesCtrExTableReader = new IndirectStorageReadOnlySource(oldAesCtrExTableSource, BlockSize);
                    m_Stream = new RelocatedIndirectStorageStream(oldFsHeader, oldFsStartOffset, oldReader, oldAesCtrExTableReader, newFsHeader, newFsStartOffset, newReader, outputFsStartOffset);
                }
            }

            if (NintendoContentArchiveSource.BuildLog.NeedsOutputCache)
            {
                m_Stream.SetBinaryRegionInfo(NintendoContentArchiveSource.BuildLog.OriginalPath, NintendoContentArchiveSource.BuildLog.CacheDirectory, oldFsHeader);
            }

            using (var writer = new IndirectStorageProgressWriter())
            {
                Task task = Task.Run(() => m_Stream.Build((uint)IndirectStorageSource.ShiftSize, BlockSize, (uint)RegionSize, MatchSize, keyGenerationForUpdatedRegion));
                try
                {
                    while (!task.Wait(1000))
                    {
                        Progress = m_Stream.GetProgress();

                        writer.Write(Progress);
                    }
                }
                catch (AggregateException e)
                {
                    throw e.InnerException;
                }
            }

            // RomFs の情報を書き出す
            if (NintendoContentArchiveSource.BuildLog.NeedsOutputLog)
            {
                using (var tableStream = new FileStream(NintendoContentArchiveSource.BuildLog.GetOutputPath(".optimized_table.txt"), FileMode.Create))
                using (var fragmentStream = new FileStream(NintendoContentArchiveSource.BuildLog.GetOutputPath(".optimized_fragment.csv"), FileMode.Create))
                using (var encryptionStream = new FileStream(NintendoContentArchiveSource.BuildLog.GetOutputPath(".aesctrex_table.txt"), FileMode.Create))
                {
                    m_Stream.OutputBuildLog(tableStream, fragmentStream, encryptionStream, 0);
                }
            }

            // IndirectStorage のヘッダ、テーブル
            MemorySource indirectTableSource;
            {
                var indirectHeaderBuffer = new byte[m_Stream.GetIndirectHeaderSize()];
                var indirectTableBuffer = new byte[m_Stream.GetIndirectTableSize()];
                m_Stream.WriteIndirectTable(indirectHeaderBuffer, indirectTableBuffer);

                IndirectHeader = indirectHeaderBuffer;
                IndirectTableSize = m_Stream.GetIndirectTableSize();
                IndirectTableOffset = m_Stream.GetIndirectTableOffset();

                indirectTableSource = new MemorySource(indirectTableBuffer, 0, indirectTableBuffer.Length);
            }

            // AesCtrExStorage のヘッダ、テーブル
            MemorySource aesCtrExTableSource;
            {
                var aesCtrExHeaderBuffer = new byte[m_Stream.GetAesCtrExHeaderSize()];
                var aesCtrExTableBuffer = new byte[m_Stream.GetAesCtrExTableSize()];
                m_Stream.WriteAesCtrExTable(aesCtrExHeaderBuffer, aesCtrExTableBuffer);
                m_Stream.VerifyAesCtrExTable(aesCtrExHeaderBuffer, aesCtrExTableBuffer, keyGenerationForUpdatedRegion);

                AesCtrExHeader = aesCtrExHeaderBuffer;
                AesCtrExTableSize = m_Stream.GetAesCtrExTableSize();
                AesCtrExTableOffset = m_Stream.GetAesCtrExTableOffset();

                aesCtrExTableSource = new MemorySource(aesCtrExTableBuffer, 0, aesCtrExTableBuffer.Length);


                // 暗号化時に使用するテーブルを取得
                GenerationTable = new AesCtrExGenerationTable(aesCtrExHeaderBuffer, aesCtrExTableBuffer);
            }

            var dataSource = new RelocatedIndirectStorageDataSource(m_Stream);

            var elements = new List<ConcatenatedSource.Element>();
            elements.Add(new ConcatenatedSource.Element(dataSource, "data", 0));
            elements.Add(new ConcatenatedSource.Element(indirectTableSource, "indirecttable", IndirectTableOffset));
            elements.Add(new ConcatenatedSource.Element(aesCtrExTableSource, "aesctrextable", AesCtrExTableOffset));

            m_Source = new ConcatenatedSource(elements);

            System.Diagnostics.Debug.Assert(
                m_Source.Size == m_Stream.GetIndirectTableSize() + m_Stream.GetAesCtrExTableSize() + m_Stream.GetDataSize());
        }

        public long Size { get { return m_Source.Size; } }
        public IndirectStorageProgress Progress { get; private set; } = new IndirectStorageProgress();

        public byte[] IndirectHeader { get; private set; }
        public long IndirectTableSize { get; private set; }
        public long IndirectTableOffset { get; private set; }

        public byte[] AesCtrExHeader { get; private set; }
        public long AesCtrExTableSize { get; private set; }
        public long AesCtrExTableOffset { get; private set; }

        public AesCtrExGenerationTable GenerationTable { get; private set; }

        private bool Disposed { get; set; } = false;

        public ByteData PullData(long offset, int size)
        {
            return m_Source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_Source.QueryStatus();
        }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (Disposed)
            {
                return;
            }

            if (disposing)
            {
                // 保持しているマネージドなリソースの解放
                if (m_Stream != null)
                {
                    m_Stream.Dispose();
                }

                if (GenerationTable != null)
                {
                    GenerationTable.Dispose();
                }
            }

            // 解放終わり
            Disposed = true;
        }
    }

    // SparseStorage のデータとテーブル情報を生成する ISource
    public class SparseStorageSource : ISource, IDisposable
    {
        public const int MinimumBlockSize = SparseStorageStream.BlockSizeMin;
        public const int DefaultBlockSize = 64 * 1024;
        public const int DefaultEraseSize = 1024 * 1024;

        public static int BlockSize = DefaultBlockSize;
        public static int EraseSize = DefaultEraseSize;

        private SparseStorageStream m_Stream;
        private ISource m_Source;

        public static void SetOption(int blockSize, int eraseSize)
        {
            if (blockSize != 0)
            {
                if (blockSize < MinimumBlockSize)
                {
                    throw new ArgumentException(string.Format("block size larger than or equal to {0} bytes.", MinimumBlockSize));
                }
                // 2 のべき乗
                if ((blockSize & (blockSize - 1)) != 0)
                {
                    throw new ArgumentException("block size should be power of 2.");
                }
                BlockSize = blockSize;
            }
            if (eraseSize != 0)
            {
                if (eraseSize < BlockSize)
                {
                    throw new ArgumentException("minimum erase size larger than or equal to block size.");
                }
                EraseSize = eraseSize;
            }
            if (EraseSize % BlockSize != 0)
            {
                throw new ArgumentException("erase size should be an integral multiple of block size.");
            }
        }

        public void Build(NintendoContentArchiveReader targetReader, NintendoContentArchiveReader patchReader, int fsIndex, long physicalOffset, Aes128CtrCryptoDriver cryptoDriver)
        {
            m_Stream = new SparseStorageStream(BlockSize, EraseSize);

            var headerInfo = m_Stream.Build(targetReader, patchReader, fsIndex, physicalOffset);
            FsHeader = headerInfo.GetRawData();

            // SparseStorage が必要
            if (headerInfo.ExistsSparseLayer())
            {
                // SparseStorage の情報を書き出す
                if (NintendoContentArchiveSource.BuildLog.NeedsOutputLog)
                {
                    using (var tableStream = new FileStream(NintendoContentArchiveSource.BuildLog.GetOutputPath(string.Format(".fs{0}.sparse_table.txt", fsIndex)), FileMode.Create))
                    {
                        m_Stream.OutputBuildLog(tableStream);
                    }
                }

                NeedsUpdate = true;
                TableSize = m_Stream.GetTableSize();

                // テーブルなし（データもなし）
                if (TableSize == 0)
                {
                    m_Source = new MemorySource(new byte[0], 0, 0);
                }
                else
                {
                    TableOffset = m_Stream.GetDataSize();

                    var tableBuffer = new byte[TableSize];
                    m_Stream.WriteTable(tableBuffer);

                    var dataSource = new SparseStorageDataSource(m_Stream);
                    var tableSource = new Aes128CtrEncryptedSource(
                        new MemorySource(tableBuffer, 0, tableBuffer.Length), cryptoDriver, headerInfo.GetSecureValue(), headerInfo.GetSparseGeneration(), physicalOffset + dataSource.Size);

                    var elements = new List<ConcatenatedSource.Element>();
                    elements.Add(new ConcatenatedSource.Element(dataSource, "data", 0));
                    elements.Add(new ConcatenatedSource.Element(tableSource, "table", TableOffset));

                    m_Source = new ConcatenatedSource(elements);
                }
            }
            // SparseStorage は不要
            else
            {
                m_Source = new StorageArchiveSource(targetReader.OpenFsRawStorageArchiveReader(fsIndex));
            }
        }

        public long Size { get { return m_Source.Size; } }
        public bool NeedsUpdate { get; private set; }
        public byte[] FsHeader { get; private set; }
        public long TableSize { get; private set; }
        public long TableOffset { get; private set; }
        private bool Disposed { get; set; } = false;

        public ByteData PullData(long offset, int size)
        {
            return m_Source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_Source.QueryStatus();
        }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (Disposed)
            {
                return;
            }

            if (disposing)
            {
                // 保持しているマネージドなリソースの解放
                if (m_Stream != null)
                {
                    m_Stream.Dispose();
                }
            }

            // 解放終わり
            Disposed = true;
        }
    }

    // IndirectStorageStream からデータを読み出す ISource
    internal class SparseStorageDataSource : ISource
    {
        private SourceStatus m_Status;
        private SparseStorageStream m_Stream;

        public SparseStorageDataSource(SparseStorageStream stream)
        {
            Size = stream.GetDataSize();
            m_Stream = stream;
            m_Status = new SourceStatus();
            m_Status.AvailableRangeList.MergingAdd(new Range(0, Size));
        }

        public long Size { get; private set; }

        public ByteData PullData(long offset, int size)
        {
            int readSize = SourceUtil.GetReadableSize(this.Size, offset, size);
            if (readSize == 0)
            {
                return new ByteData(new ArraySegment<byte>());
            }
            return new ByteData(new ArraySegment<byte>(m_Stream.ReadData(offset, readSize)));
        }

        public SourceStatus QueryStatus()
        {
            return m_Status;
        }
    }

    public class SparseStorageArchiveSource : ISource
    {
        private SourceStatus m_Status;
        private SparseStorageArchiveStream m_Stream;

        public SparseStorageArchiveSource(SparseStorageArchiveStream stream)
        {
            Size = stream.GetDataSize();
            m_Stream = stream;
            m_Status = new SourceStatus();
            m_Status.AvailableRangeList.MergingAdd(new Range(0, Size));
        }

        public long Size { get; private set; }

        public ByteData PullData(long offset, int size)
        {
            int readSize = SourceUtil.GetReadableSize(this.Size, offset, size);
            if (readSize == 0)
            {
                return new ByteData(new ArraySegment<byte>());
            }

            var buffer = m_Stream.ReadData(offset, readSize);
            return new ByteData(new ArraySegment<byte>(buffer, 0, buffer.Length));
        }

        public SourceStatus QueryStatus()
        {
            return m_Status;
        }
    }

    public class UnreadableSource : ISource
    {
        public long Size { get; private set; }

        public UnreadableSource(ISource source)
        {
            Size = source.Size;
        }

        public ByteData PullData(long offset, int size)
        {
            throw new NotImplementedException();
        }

        public SourceStatus QueryStatus()
        {
            return new SourceStatus();
        }
    }
}
