﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace MakeOffsetHeader
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Text.RegularExpressions;
    using System.Threading;
    using System.Threading.Tasks;
    using AOCommonLibrary;

    /// <summary>
    /// エントリポイント
    /// </summary>
    internal class Program
    {
        private static readonly Regex OffsetRule = new Regex(@"^[\t\s]*([0-9a-zA-Z_]+)(:\s+.quad\s+[#]?)(-?[0-9]+)\s*[\r]?\n[\s]*ARCHSIZEOF_\1\2([0-9]+)[\r]?$", RegexOptions.Multiline);

        /// <summary>
        /// メイン関数
        /// </summary>
        /// <param name="args">コマンドライン引数</param>
        internal static void Main(string[] args)
        {
            Mutex mutex = new Mutex(false, "AssemblyOffsetTools");
            mutex.WaitOne();

            // 引数の解析
            var result = Parser.ParseCommandLineArgs(args);

            try
            {
                var contents = StreamIO.ReadData(
                                    result.ContainsKey("-i") ? result["-i"] : (dynamic)Console.In,
                                    result.ContainsKey("-w") ? int.Parse(result["-w"]) : 200);
                var translated = Translate(contents);
                StreamIO.WriteData(result.ContainsKey("-c") ? MakeCopyrightForLgpl() + translated : MakeCopyright() + translated, result.ContainsKey("-o") ? result["-o"] : (dynamic)Console.Out);
            }
            catch (Exception ex)
            {
                Console.Error.WriteLine(ex);
                Environment.Exit(1);
            }

            mutex.ReleaseMutex();
        }

        private static string MakeCopyright()
        {
            var copyright =
@"/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * このファイルを編集しないでください。
 * このファイルは自動生成されます。
 */

// NOLINT(build/header_guard)

";
            return copyright;
        }

        private static string MakeCopyrightForLgpl()
        {
            var copyright =
@"/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo. All rights reserved.

  These coded instructions, statements, and computer programs contain
  information of Nintendo and/or its licensed developers and are protected
  by national and international copyright laws.

  THE SOFTWARE IS PROVIDED ""AS IS"", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
  CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
  TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
  SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *--------------------------------------------------------------------------------*/

/*
 * このファイルを編集しないでください。
 * このファイルは自動生成されます。
 */

// NOLINT(build/header_guard)

";
            return copyright;
        }

        private static string Translate(string source)
        {
            var result = new StringBuilder();

            foreach (Match match in OffsetRule.Matches(source))
            {
                var offset = match.Groups[3].Value;
                var archSize = match.Groups[4].Value;

                if (archSize != "4" && archSize != "8")
                {
                    Console.WriteLine("WARNING: Unknown architecture size {0}. Assume 64bit architecture.", archSize);
                }

                string offsetHex = Convert.ToString((archSize == "4") ? (dynamic)int.Parse(offset) : long.Parse(offset), 16);

                result.AppendLine(OffsetRule.Replace(match.ToString(), string.Format("#define $1 0x{0} // NOLINT(readability/define)", offsetHex)));
            }

            return result.ToString();
        }
    }
}
