﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace MakeOffsetCpp
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Text.RegularExpressions;
    using System.Threading;
    using System.Threading.Tasks;
    using AOCommonLibrary;

    /// <summary>
    /// エントリポイント
    /// </summary>
    internal class Program
    {
        private static readonly Regex CommentRule = new Regex(@"#.*$");
        private static readonly Regex IncludeRule = new Regex("^include");
        private static readonly Regex DefineRule = new Regex(@"^define\s+(\S+)\s+(\S+|\S*\(.*\)\S*)\s*[\r]?$");
        private static readonly Regex SpaceRule = new Regex(@"^\s*$");

        /// <summary>
        /// メイン関数
        /// </summary>
        /// <param name="args">コマンドライン引数</param>
        internal static void Main(string[] args)
        {
            Mutex mutex = new Mutex(false, "AssemblyOffsetTools");
            mutex.WaitOne();

            // 引数の解析
            var result = Parser.ParseCommandLineArgs(args);
            try
            {
                var contents = StreamIO.ReadData(
                                    result.ContainsKey("-i") ? result["-i"] : (dynamic)Console.In,
                                    result.ContainsKey("-w") ? int.Parse(result["-w"]) : 200);
                var translated = Translate(contents);
                StreamIO.WriteData(translated, result.ContainsKey("-o") ? result["-o"] : (dynamic)Console.Out);
            }
            catch (Exception ex)
            {
                Console.Error.WriteLine(ex);
                Environment.Exit(1);
            }

            mutex.ReleaseMutex();
        }

        private static string Translate(string source)
        {
            var result = new StringBuilder();

            try
            {
                foreach (var line in source.Split('\n'))
                {
                    if (CommentRule.IsMatch(line))
                    {
                        continue; // コメントは無視
                    }
                    else if (IncludeRule.IsMatch(line))
                    {
                        result.AppendLine(IncludeRule.Replace(line, "#include"));  // include の変換
                    }
                    else if (DefineRule.IsMatch(line))
                    {
                        if (line.Contains("ARCHSIZEOF_"))
                        {
                            // ARCHSIZEOF_ は予約語なので使用してはいけない
                            throw new Exception("ERROR: ARCHSIZEOF_ で始まる名前を使用することはできません。\n" + line);
                        }

                        result.AppendLine(DefineRule.Replace(line, "void __$1(){asm volatile (\"$1: .quad %0\\nARCHSIZEOF_$1: .quad %1\"::\"n\"($2), \"n\"(sizeof($2)));}"));  // define の変換
                    }
                    else if (SpaceRule.IsMatch(line))
                    {
                        // 空行、空白のみの行は無視
                        result.AppendLine(string.Empty);
                    }
                    else
                    {
                        throw new Exception("ERROR: 解析できない文字列です。\n" + line);
                    }
                }
            }
            catch (Exception)
            {
                Console.Error.WriteLine("ERROR: 入力ファイルの書式が不正です。");
                throw;
            }

            return result.ToString();
        }
    }
}
