﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/fs_Base.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_Host.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include "Utility/systemInitializer_Allocator.h"
#include <nn/utilTool/utilTool_CommandFramework.h>
#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/utilTool/utilTool_ResultHandlingUtility.h>
#include <nn/utilTool/utilTool_StringUtility.h>
#include <nn/tma/tma.h>
#include <Partition/systemInitializer_ListPartitions.h>
#include <cstdlib>
#include "WriteToStorage.h"

extern "C" void nninitStartup()
{
}

class WriteToStorageCommand
{
public:
    WriteToStorageCommand() :
        m_ListPartitions(false),
        m_Size(0),
        m_BeginAddress("0")
    {}

    const char* Name()
    {
        return "WriteToStorage";
    }

    const char* Summary()
    {
        return "Write raw data to the build-in-storage of a target.";
    }

    const char* Description()
    {
        return "";
    }

    void Arguments(nn::utilTool::SingleCommandInterface &commandInterface)
    {
        commandInterface.AddKeywordArgument(&m_InputPath, 'i', "input", "Input path of image file.", false);
        commandInterface.AddKeywordArgument(&m_TargetType, "type", "Target partition type. (See --list)", false);
        commandInterface.AddKeywordArgument(&m_TargetName, "name", "Target partition name.", false);
        commandInterface.AddKeywordArgument(&m_Size, "size", "Write size.", false);
        commandInterface.AddKeywordArgument(&m_BeginAddress, "begin", "Begin address of target partition. example: '--begin ABCD0123'", false);
        commandInterface.AddFlagArgument(&m_ListPartitions, "list", "List partitions.");
    }

    int Main()
    {
        nn::fs::InitializeWithMultiSessionForTargetTool();
        nn::fs::SetAllocator(Allocate, Deallocate);

        if(m_ListPartitions)
        {
            NN_UTILTOOL_LOG_VERBOSE("List partitions.");

            nn::Result result = ListPartitions();

            if( result.IsFailure() )
            {
                NN_UTILTOOL_LOG_ERROR("Failed to list partitions. Result(module=%d, desc=%d)", result.GetModule(), result.GetDescription());
                return 1;
            }

            return 0;
        }
        else
        {
            NN_UTILTOOL_LOG_VERBOSE("Start to write partition.");
            NN_UTILTOOL_LOG_VERBOSE("input path: %s", m_InputPath.c_str());

            NN_UTILTOOL_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::MountHostRoot());

            int64_t beginAddress;

            if(m_BeginAddress == "")
            {
                beginAddress = 0;
            }
            else
            {
                beginAddress = nn::utilTool::StrToInt64(m_BeginAddress.c_str(), nullptr, 16);
            }

            nn::Result result = WriteToStorage(m_InputPath, m_TargetType, m_TargetName, m_Size, beginAddress);
            if( result.IsFailure() )
            {
                NN_UTILTOOL_LOG_ERROR("Failed to export partition. Result(module=%d, desc=%d)", result.GetModule(), result.GetDescription());
                return 1;
            }

            return 0;
        }
    }

private:
    std::string m_InputPath;
    std::string m_TargetType;
    std::string m_TargetName;
    bool m_ListPartitions;
    int64_t m_Size;
    std::string m_BeginAddress;
};

extern "C" void nnMain()
{
    nn::utilTool::BasicCommandFramework<WriteToStorageCommand>::Run(
        nn::os::GetHostArgc(),
        const_cast<const char**>(nn::os::GetHostArgv())
    );
}
