﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "Utility/systemInitializer_IFile.h"
#include <nn/fs_Base.h>
#include <memory>
#include "Utility/systemInitializer_FsFile.h"

struct Cmac
{
    static const int CMAC_SIZE = 16;
    uint8_t data[CMAC_SIZE];

    bool operator==(const Cmac &cmac) const;
};

struct Key128
{
    static const int KEY_SIZE = 16;
    static const int KEY_STRING_SIZE = 32;
    uint8_t data[KEY_SIZE];

    static bool IsValid(const char *hexString);
    static Key128 Make(uint8_t *data);
    static Key128 Make(const char *hexString);
    static Key128 EncryptKey(Key128 source, Key128 kek);
};

struct Key2048
{
    static const int KEY_SIZE = 256;
    static const int KEY_STRING_SIZE = 512;
    uint8_t data[KEY_SIZE];

    static bool IsValid(const char *hexString);
    static Key2048 Make(uint8_t *data);
    static Key2048 Make(const char *hexString);
};

struct CmacFileHeader
{
    static const size_t SIGNATURE_SIZE = 16;

    char signature[SIGNATURE_SIZE];
    Cmac headerCmac;
    Key128 encryptedCmacKey;

    uint64_t BlockSize;
    uint64_t NumberOfBlocks;
};

class CmacFile : public IFile
{
public:
    CmacFile();
    virtual ~CmacFile() {}

    virtual nn::Result OpenRead(const char *path);
    virtual nn::Result OpenRead(const char *path, Key128 kek);
    virtual nn::Result OpenWrite(const char *path);
    virtual void Close();
    virtual nn::Result Write(int64_t offset, const void* buffer, size_t size, bool flush);
    virtual nn::Result Read(size_t* pOut, int64_t offset, void* buffer, size_t size);
    virtual nn::Result GetSize(int64_t* pOut);
    virtual nn::Result Flush();
    virtual bool IsValid();

private:
    int64_t CalculateBlockOffset(int64_t blockIndex);
    nn::Result ReadBlock(int64_t blockIndex, uint8_t *data, size_t size);
    void VerifyBlock(int64_t blockIndex, uint8_t *data, size_t size);
    void VerifyCmac(int64_t blockIndex, const Cmac &cmac, uint8_t *data, size_t size);
    bool VerifyBlock2(int64_t blockIndex, uint8_t *data, size_t size);
    bool VerifyCmac2(int64_t blockIndex, const Cmac &cmac, uint8_t *data, size_t size);

private:
    FsFile m_File;
    CmacFileHeader m_Header;
    Key128 m_CmacKey;
    std::unique_ptr<uint8_t[]> m_Cache;
    int64_t m_CacheIndex;
    static const int64_t INVALID_CACHE_INDEX = -1;
};
