﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "systemInitializer_CmacFile.h"
#include <nn/nn_Abort.h>
#include <nn/util/util_BitUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs_Base.h>
#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/crypto/crypto_Aes128CmacGenerator.h>
#include <algorithm>
#include <memory>
#include <cstring>

namespace
{
    bool IsHexChar(char c)
    {
        return
            ('0' <= c && c <= '9') ||
            ('a' <= c && c <= 'f') ||
            ('A' <= c && c <= 'F');
    }

    int FromHexChar(char c)
    {
        if ('0' <= c && c <= '9')
        {
            return c - '0';
        }
        else if ('a' <= c && c <= 'f')
        {
            return c - 'a' + 0xA;
        }
        else if ('A' <= c && c <= 'F')
        {
            return c - 'A' + 0xA;
        }
        else
        {
            NN_ABORT("invalid hex char");
        }
    }
}

char ToHexChar(uint8_t h)
{
    if (0 <= h && h <= 9)
    {
        return h + '0';
    }
    else if (0xA <= h && h <= 0xF)
    {
        return h - 0xA + 'A';
    }
    else
    {
        NN_ABORT("invalid hex");
    }
}

uint8_t HiHex(uint8_t b)
{
    return (b >> 4) & 0xF;
}

uint8_t LowHex(uint8_t b)
{
    return b & 0xF;
}

bool Cmac::operator==(const Cmac &cmac) const
{
    bool equal = true;
    for (int i = 0; i < Cmac::CMAC_SIZE; i++)
    {
        if (this->data[i] != cmac.data[i])
        {
            equal = false;
        }
    }

    return equal;
}


bool Key128::IsValid(const char *hexString)
{
    if (strnlen(hexString, KEY_STRING_SIZE + 1) != KEY_STRING_SIZE)
    {
        return false;
    }

    for (int i = 0; i < KEY_STRING_SIZE; i++)
    {
        if (!IsHexChar(hexString[i]))
        {
            return false;
        }
    }

    return true;

}

Key128 Key128::Make(uint8_t *data)
{
    Key128 ret;
    for (int i = 0; i < KEY_SIZE; i++)
    {
        ret.data[i] = data[i];
    }
    return ret;
}

Key128 Key128::Make(const char *hexString)
{
    NN_ABORT_UNLESS(Key128::IsValid(hexString), "Invalid key string.");
    Key128 ret;
    for (int i = 0; i < KEY_SIZE; i++)
    {
        ret.data[i] = (FromHexChar(hexString[i * 2]) << 4) | FromHexChar(hexString[i * 2 + 1]);
    }
    return ret;
}

Key128 Key128::EncryptKey(Key128 source, Key128 kek)
{
    Key128 ret;

    for (int i = 0; i < KEY_SIZE; i++)
    {
        ret.data[i] = source.data[i] ^ kek.data[i];
    }

    return ret;
}


bool Key2048::IsValid(const char *hexString)
{
    if (strnlen(hexString, KEY_STRING_SIZE + 1) != KEY_STRING_SIZE)
    {
        return false;
    }

    for (int i = 0; i < KEY_STRING_SIZE; i++)
    {
        if (!IsHexChar(hexString[i]))
        {
            return false;
        }
    }

    return true;
}

Key2048 Key2048::Make(uint8_t *data)
{
    Key2048 ret;
    for (int i = 0; i < KEY_SIZE; i++)
    {
        ret.data[i] = data[i];
    }
    return ret;
}

Key2048 Key2048::Make(const char *hexString)
{
    NN_ABORT_UNLESS(Key2048::IsValid(hexString), "Invalid key string.");
    Key2048 ret;
    for (int i = 0; i < KEY_SIZE; i++)
    {
        ret.data[i] = (FromHexChar(hexString[i * 2]) << 4) | FromHexChar(hexString[i * 2 + 1]);
    }
    return ret;
}


CmacFile::CmacFile()
{
}

nn::Result CmacFile::OpenRead(const char *path, Key128 kek)
{
    NN_RESULT_DO(
        m_File.OpenRead(path));

    CmacFileHeader header;

    // ヘッダを読み込む
    size_t readSize;
    NN_RESULT_DO(
        m_File.Read(&readSize, 0, &header, sizeof(header)));

    // 鍵の暗号化を解く
    Key128 cmacKey = Key128::EncryptKey(header.encryptedCmacKey, kek);

    Cmac cmac = header.headerCmac;

    // Cmac の部分の値を潰す
    ::memset(&header.headerCmac, 0, sizeof(header.headerCmac));

    // 検証する
    nn::crypto::Aes128CmacGenerator cmacGenerator;
    cmacGenerator.Initialize(cmacKey.data, Key128::KEY_SIZE);

    cmacGenerator.Update(&header, sizeof(header));

    Cmac resultCmac = {};
    cmacGenerator.GetMac(resultCmac.data, Cmac::CMAC_SIZE);

    NN_ABORT_UNLESS(resultCmac == cmac, "invalid header");
    NN_ABORT_UNLESS(header.BlockSize == 1 * 1024 * 1024, "invalid block size");

    NN_UTILTOOL_LOG_VERBOSE("header:");
    NN_UTILTOOL_LOG_VERBOSE("  signature: %s", header.signature);
    NN_UTILTOOL_LOG_VERBOSE("  block size: %lld", header.BlockSize);
    NN_UTILTOOL_LOG_VERBOSE("  number of blocks: %lld", header.NumberOfBlocks);

    m_Header = header;
    m_CmacKey = cmacKey;
    m_Cache = std::unique_ptr<uint8_t[]>(new uint8_t[Cmac::CMAC_SIZE + header.BlockSize]);
    NN_ABORT_UNLESS_NOT_NULL(m_Cache.get());
    m_CacheIndex = INVALID_CACHE_INDEX;

    NN_RESULT_SUCCESS;
}

nn::Result CmacFile::OpenRead(const char *path)
{
    NN_RESULT_THROW(nn::fs::ResultNotImplemented());
}

nn::Result CmacFile::OpenWrite(const char *path)
{
    NN_RESULT_THROW(nn::fs::ResultNotImplemented());
}

void CmacFile::Close()
{
    m_File.Close();
}

nn::Result CmacFile::Write(int64_t offset, const void* buffer, size_t size, bool flush)
{
    NN_RESULT_THROW(nn::fs::ResultNotImplemented());
}

nn::Result CmacFile::Read(size_t* pOut, int64_t offset, void* buffer, size_t size)
{
    int64_t endOffset = offset + size;
    size_t totalRead = 0;

    for (int64_t currentOffset = offset; currentOffset < endOffset; )
    {
        int64_t blockIndex = currentOffset / m_Header.BlockSize;
        if (m_CacheIndex != blockIndex)
        {
            ReadBlock(blockIndex, m_Cache.get(), Cmac::CMAC_SIZE + m_Header.BlockSize);
            m_CacheIndex = blockIndex;
        }

        uint8_t *blockData = m_Cache.get() + Cmac::CMAC_SIZE;

        // ブロック内でのコピー元のオフセットとサイズを計算する
        int64_t startBlockOffset = nn::util::align_down(currentOffset, m_Header.BlockSize);
        int64_t endBlockOffset = startBlockOffset + m_Header.BlockSize;
        size_t readSize = std::min(endBlockOffset, endOffset) - currentOffset;

        size_t sourceOffset = currentOffset - startBlockOffset;

        // コピー先のオフセットとサイズを計算する
        size_t targetOffset = currentOffset - offset;

        NN_ABORT_UNLESS(readSize <= size && readSize <= m_Header.BlockSize);
        NN_ABORT_UNLESS(0 <= sourceOffset && sourceOffset <= m_Header.BlockSize);
        NN_ABORT_UNLESS(0 <= targetOffset && targetOffset <= size);

        // コピーする
        std::memcpy( reinterpret_cast<uint8_t*>(buffer) + targetOffset, blockData + sourceOffset, readSize);

        currentOffset += readSize;
        totalRead += readSize;
    }

    *pOut = totalRead;

    NN_RESULT_SUCCESS;
}

nn::Result CmacFile::GetSize(int64_t* pOut)
{
    NN_RESULT_DO(
        m_File.GetSize(pOut));

    NN_RESULT_SUCCESS;
}

nn::Result CmacFile::Flush()
{
    NN_RESULT_THROW(nn::fs::ResultNotImplemented());
}

bool CmacFile::IsValid()
{
    return m_File.IsValid();
}

int64_t CmacFile::CalculateBlockOffset(int64_t blockIndex)
{
    return sizeof(m_Header) + (Cmac::CMAC_SIZE + m_Header.BlockSize) * blockIndex;
}

nn::Result CmacFile::ReadBlock(int64_t blockIndex, uint8_t * data, size_t size)
{
    NN_ABORT_UNLESS(size == m_Header.BlockSize + Cmac::CMAC_SIZE);

    auto blockOffset = CalculateBlockOffset(blockIndex);
    size_t readSize = 0;
    NN_RESULT_DO(
        m_File.Read(&readSize, blockOffset, data, size));

    for(int i = 0; i < 10; i++)
    {
        if( !VerifyBlock2(blockIndex, data, size) )
        {
            NN_RESULT_DO(
                m_File.Read(&readSize, blockOffset, data, size));
        }
        else
        {
            break;
        }

        NN_UTILTOOL_LOG_VERBOSE("retry: count=%d", i);
    }

    VerifyBlock(blockIndex, data, size);

    NN_ABORT_UNLESS(readSize == (m_Header.BlockSize + Cmac::CMAC_SIZE));

    NN_RESULT_SUCCESS;
}

void CmacFile::VerifyBlock(int64_t blockIndex, uint8_t * data, size_t size)
{
    NN_ABORT_UNLESS(size == m_Header.BlockSize + Cmac::CMAC_SIZE);

    Cmac cmac;
    memcpy(cmac.data, data, Cmac::CMAC_SIZE);

    VerifyCmac(blockIndex, cmac, &data[Cmac::CMAC_SIZE], size - Cmac::CMAC_SIZE);
}

void CmacFile::VerifyCmac(int64_t blockIndex, const Cmac &cmac, uint8_t * data, size_t size)
{
    NN_ABORT_UNLESS(size == m_Header.BlockSize);

    int32_t blockIndex32 = static_cast<int32_t>(blockIndex);

    nn::crypto::Aes128CmacGenerator cmacGenerator;
    cmacGenerator.Initialize(m_CmacKey.data, Key128::KEY_SIZE);

    cmacGenerator.Update(&blockIndex32, sizeof(blockIndex32));
    cmacGenerator.Update(data, size);

    Cmac resultCmac = {};
    cmacGenerator.GetMac(resultCmac.data, Cmac::CMAC_SIZE);

    NN_ABORT_UNLESS(cmac == resultCmac, "invalid data block");
}

bool CmacFile::VerifyBlock2(int64_t blockIndex, uint8_t * data, size_t size)
{
    NN_ABORT_UNLESS(size == m_Header.BlockSize + Cmac::CMAC_SIZE);

    Cmac cmac;
    memcpy(cmac.data, data, Cmac::CMAC_SIZE);

    return VerifyCmac2(blockIndex, cmac, &data[Cmac::CMAC_SIZE], size - Cmac::CMAC_SIZE);
}

bool CmacFile::VerifyCmac2(int64_t blockIndex, const Cmac &cmac, uint8_t * data, size_t size)
{
    NN_ABORT_UNLESS(size == m_Header.BlockSize);

    int32_t blockIndex32 = static_cast<int32_t>(blockIndex);

    nn::crypto::Aes128CmacGenerator cmacGenerator;
    cmacGenerator.Initialize(m_CmacKey.data, Key128::KEY_SIZE);

    cmacGenerator.Update(&blockIndex32, sizeof(blockIndex32));
    cmacGenerator.Update(data, size);

    Cmac resultCmac = {};
    cmacGenerator.GetMac(resultCmac.data, Cmac::CMAC_SIZE);

    return cmac == resultCmac;
}
