﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkText.h>
#include <nn/fs_Base.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/ldr/ldr_Result.h>
#include <nn/spl/spl_Api.h>

#include "ldr_MetaLoader.h"
#include "ldr_DataTypes.h"
#include "ldr_AutoClose.h"


namespace nn { namespace ldr {

    extern bool g_IsDescProductionFlagRequired;

    namespace
    {
        const char NpdmPath[] = "code:/main.npdm";

        bool IsValidRegion(uint32_t offset, uint32_t size, uint32_t begin, uint32_t end, size_t minSize) NN_NOEXCEPT
        {
            return (begin <= offset)
                && (offset + size <= end)
                && (size <= end)
                && (size >= minSize);
        }

        Result LoadNpdm(size_t* pNpdmSize, void* pBuffer, size_t bufferSize) NN_NOEXCEPT
        {
            Result result;
            fs::FileHandle file;

            result = fs::OpenFile(&file, NpdmPath, fs::OpenMode_Read);
            if( result.IsFailure() )
            {
                return result;
            }

            {
                AutoCloseFile acf(file);

                int64_t fileSize;
                result = fs::GetFileSize(&fileSize, file);
                if( result.IsFailure() )
                {
                    return result;
                }

                if( fileSize >= static_cast<int64_t>(bufferSize) )
                {
                    return ResultMetaOverflow();
                }

                result = fs::ReadFile(file, 0, pBuffer, fileSize);
                if( result.IsFailure() )
                {
                    return result;
                }

                *pNpdmSize = fileSize;
            }

            return ResultSuccess();
        }

        bool IsValidMeta(const Meta* pMeta, size_t npdmSize) NN_NOEXCEPT
        {
            if( ! (pMeta->signature == Meta::SignatureValue) )
            {
                return false;
            }
            if( ! ((pMeta->flags & 0x0000FFF0) == 0) )
            {
                return false;
            }
            if( ! IsValidRegion(pMeta->aciOffset, pMeta->aciSize,
                                    sizeof(Meta), npdmSize, sizeof(AciHeader)) )
            {
                return false;
            }
            if( ! IsValidRegion(pMeta->acidOffset, pMeta->acidSize,
                                    sizeof(Meta), npdmSize, sizeof(AcidHeader)) )
            {
                return false;
            }

            return true;
        }

        bool IsValidDescVersion(uint8_t descVersion) NN_NOEXCEPT
        {
            // 5.0.0 NUP では descVersion が 1 以上 4 以下の場合無効とする
            // 通常の desc は descVersion が 0 なので動作する。
            //
            // 特別に 4.0.0 NUP 系本体では動作するが、
            // 5.0.0 NUP 以降では動作しない desc を作りたい場合に
            // descVersion 4 の desc を作成する。
            if (descVersion >= 1 && descVersion < 5)
            {
                return false;
            }
            return true;
        }

        bool IsValidAcidHeader(const AcidHeaderData* pAcid, size_t acidSize) NN_NOEXCEPT
        {
            if( ! (pAcid->signature == AcidHeaderData::SignatureValue) )
            {
                return false;
            }
            if (g_IsDescProductionFlagRequired)
            {
                // 製品で動作している場合
                util::BitPack32 acidFlags = { pAcid->flags };
                if( ! acidFlags.Get<AcidHeaderData::ProductionFlag>() )
                {
                    // 製品フラグが有効でない場合
                    return false;
                }
            }
            if ( ! IsValidDescVersion(pAcid->descVersion) )
            {
                return false;
            }
            if( ! IsValidRegion(pAcid->facdOffset, pAcid->facdSize,
                                    sizeof(AcidHeader), acidSize, 0) )
            {
                return false;
            }
            if( ! IsValidRegion(pAcid->sacdOffset, pAcid->sacdSize,
                                    sizeof(AcidHeader), acidSize, 0) )
            {
                return false;
            }
            if( ! IsValidRegion(pAcid->kcdOffset, pAcid->kcdSize,
                                    sizeof(AcidHeader), acidSize, 0) )
            {
                return false;
            }

            return true;
        }

        bool IsValidAciHeader(const AciHeader* pAci, size_t aciSize) NN_NOEXCEPT
        {
            if( ! (pAci->signature == AciHeader::SignatureValue) )
            {
                return false;
            }
            if( ! IsValidRegion(pAci->facOffset, pAci->facSize,
                                    sizeof(AciHeader), aciSize, 0) )
            {
                return false;
            }
            if( ! IsValidRegion(pAci->sacOffset, pAci->sacSize,
                                    sizeof(AciHeader), aciSize, 0) )
            {
                return false;
            }
            if( ! IsValidRegion(pAci->kcOffset, pAci->kcSize,
                                    sizeof(AciHeader), aciSize, 0) )
            {
                return false;
            }

            return true;
        }
    }


    nn::Result MetaLoader::Load(MetaLoader::Npdm* pNpdm, ncm::ProgramId id) NN_NOEXCEPT
    {
        Result result;
        size_t npdmSize;

        // キャッシュ無効化
        m_CachedProgramId.value = 0;

        // npdm を読み込み
        result = LoadNpdm(&npdmSize, m_NpdmBuffer, sizeof(m_NpdmBuffer));
        if( result.IsFailure() )
        {
            return result;
        }

        if( npdmSize < sizeof(Meta) )
        {
            return ResultInvalidMeta();
        }

        // 妥当性検査
        {
            const Meta* pMeta = reinterpret_cast<Meta*>(m_NpdmBuffer);

            if( ! IsValidMeta(pMeta, npdmSize) )
            {
                return ResultInvalidMeta();
            }

            const auto pAcid = reinterpret_cast<AcidHeader*>(m_NpdmBuffer + pMeta->acidOffset);
            const auto pAci  = reinterpret_cast<AciHeader* >(m_NpdmBuffer + pMeta->aciOffset);

            if( ! IsValidAcidHeader(&pAcid->data, pMeta->acidSize) )
            {
                return ResultInvalidMeta();
            }
            if( ! IsValidAciHeader(pAci, pMeta->aciSize) )
            {
                return ResultInvalidMeta();
            }

            m_NpdmCache.pMeta = pMeta;
            m_NpdmCache.pAcid = &pAcid->data;
            m_NpdmCache.pAci  = pAci;

            m_NpdmCache.pFacd = reinterpret_cast<Bit8*>(pAcid) + pAcid->data.facdOffset;
            m_NpdmCache.pSacd = reinterpret_cast<Bit8*>(pAcid) + pAcid->data.sacdOffset;
            m_NpdmCache.pKcd  = reinterpret_cast<Bit8*>(pAcid) + pAcid->data.kcdOffset;

            m_NpdmCache.pFac  = reinterpret_cast<Bit8*>(pAci) + pAci->facOffset;
            m_NpdmCache.pSac  = reinterpret_cast<Bit8*>(pAci) + pAci->sacOffset;
            m_NpdmCache.pKc   = reinterpret_cast<Bit8*>(pAci) + pAci->kcOffset;
        }


        // 署名検証
        {
            // TODO
        }


        m_CachedProgramId = id;
        *pNpdm = m_NpdmCache;

        return ResultSuccess();
    }

    nn::Result MetaLoader::GetCachedOrLoad(MetaLoader::Npdm* pNpdm, ncm::ProgramId id) NN_NOEXCEPT
    {
        if( id == m_CachedProgramId )
        {
            *pNpdm = m_NpdmCache;
            return ResultSuccess();
        }
        else
        {
            return Load(pNpdm, id);
        }
    }


}}  // nn::ldr
