﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>
#include <mutex>
#include <siglo/client.h>
#include <sys/malloc.h>
#include <sys/socket.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/socket/socket_ConstantsPrivate.h>
#include <nn/socket/socket_Types.h>
#include <nn/socket/private/ioctl.h>
#include <nn/socket/private/session.h>
#include <nn/socket/private/thread.h>
#include <nn/util/util_Exchange.h>
#include <nn/socket/resolver/private/resolver_PrivateApi.h>
#include <nn/socket/resolver/resolver_Client.h>
#include "bsdsocket_ClientImpl.h"

namespace nn { namespace socket { namespace detail { namespace
{
    struct MapEntry
    {
        Bit64 processId;
        ClientImpl* pClient;
    };
    MapEntry g_MapEntries[16] = { };

    struct StaticMutex
    {
        os::MutexType _mutex;

        void lock() NN_NOEXCEPT
        {
            os::LockMutex(&_mutex);
        }

        void unlock() NN_NOEXCEPT
        {
            os::UnlockMutex(&_mutex);
        }
    };
    StaticMutex g_MapMutex = { NN_OS_MUTEX_INITIALIZER(false) };

    void AddToMap(Bit64 processId, ClientImpl* pClient) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pClient);
        std::lock_guard<decltype(g_MapMutex)> lk(g_MapMutex);
        for (auto&& entry : g_MapEntries)
        {
            if (entry.pClient == nullptr)
            {
                entry.processId = processId;
                entry.pClient = pClient;
                return;
            }
        }
    }

    ClientImpl* GetFromMap(Bit64 processId) NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_MapMutex)> lk(g_MapMutex);
        for (auto&& entry : g_MapEntries)
        {
            if (entry.processId == processId && entry.pClient != nullptr)
            {
                return util::Exchange(&entry.pClient, nullptr);
            }
        }
        return nullptr;
    }

    inline const char* GetPointer(const nn::sf::InBuffer& buffer) NN_NOEXCEPT
    {
        return buffer.GetSize() == 0 ? nullptr : buffer.GetPointerUnsafe();
    }

    inline char* GetPointer(const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
    {
        return buffer.GetSize() == 0 ? nullptr : buffer.GetPointerUnsafe();
    }

    template <typename T>
    inline const void* GetArrayData(const nn::sf::InArray<T>& inArray) NN_NOEXCEPT
    {
        return inArray.GetLength() == 0 ? nullptr : inArray.GetData();
    }

    struct TimeVal32
    {
        int32_t tv_sec;
        int32_t tv_usec;
    };

    struct TimeVal64
    {
        int64_t tv_sec;
        int64_t tv_usec;
    };

    union MyTimeVal
    {
        TimeVal32 tv32;
        TimeVal64 tv64;
    };

}}}} // namespace nn::socket::detail::<unnamed>

namespace nn { namespace socket { namespace detail
{
    // keep this re-definition local,
    // remove once libc support for errno is ready
    #ifdef  errno
    #undef  errno
    #endif
    #define errno (* NetworkThreadErrno())

    ClientImpl::ClientImpl(int gid) NN_NOEXCEPT
        : m_ProcessId(0),
          m_IsRegistered(false),
          m_CoreStackPermissionGroupId(gid)
    {
    }

    ClientImpl::~ClientImpl() NN_NOEXCEPT
    {
        if (m_pMonitorTarget)
        {
            m_pMonitorTarget->CancelAndWait();
        }
    }

    Result ClientImpl::RegisterClient(
        nn::sf::Out<int> pOutRet,
        Bit64 pid,
        nn::sf::NativeHandle memoryHandle,
        uint64_t memorySize,
        const nn::socket::sf::LibraryConfigData configData) NN_NOEXCEPT
    {
        auto* transferMemory = reinterpret_cast<nn::os::TransferMemoryType*>(
            malloc(sizeof(nn::os::TransferMemoryType), M_DEVBUF, 0));
        NN_ABORT_UNLESS_NOT_NULL(transferMemory);

        nn::os::AttachTransferMemory(
            transferMemory, memorySize, memoryHandle.GetOsHandle(), memoryHandle.IsManaged());
        memoryHandle.Detach();

        void* memoryPool;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::MapTransferMemory(
            &memoryPool, transferMemory, nn::os::MemoryPermission_None));

        client_config cfg = {0};
        cfg.version                      = configData.version;
        cfg.gid                          = m_CoreStackPermissionGroupId;
        cfg.tcpInitialSendBufferSize     = configData.tcpInitialSendBufferSize;
        cfg.tcpInitialReceiveBufferSize  = configData.tcpInitialReceiveBufferSize;
        cfg.tcpAutoSendBufferSizeMax     = configData.tcpAutoSendBufferSizeMax;
        cfg.tcpAutoReceiveBufferSizeMax  = configData.tcpAutoReceiveBufferSizeMax;
        cfg.udpSendBufferSize            = configData.udpSendBufferSize;
        cfg.udpReceiveBufferSize         = configData.udpReceiveBufferSize;
        cfg.socketBufferEfficiency       = configData.socketBufferEfficiency;
        *pOutRet = clientCreate(
            pid, memoryPool, memorySize, static_cast<void*>(transferMemory), &cfg);
        if (0 <= *pOutRet)
        {
            // FIXME: Multiple objects may be created for one client process because
            //        clientCreate() returns zero even if the process has already been registered.
            AddReference();
            AddToMap(pid, this);
            m_ProcessId = pid;
            m_IsRegistered = true;
        }
        else
        {
            nn::os::UnmapTransferMemory(transferMemory);
            nn::os::DestroyTransferMemory(transferMemory);
            free(transferMemory, M_DEVBUF);
        }
        return ResultSuccess();
    }

    Result ClientImpl::StartMonitoring(Bit64 pid) NN_NOEXCEPT
    {
        auto pMonitorTarget = GetFromMap(pid);
        if (pMonitorTarget != nullptr)
        {
            m_pMonitorTarget = decltype(m_pMonitorTarget)(pMonitorTarget, false);
        }
        NN_RESULT_SUCCESS;
    }

    Result ClientImpl::Socket(
        nn::sf::Out<int> pOutResult,
        nn::sf::Out<int> pOutError,
        int domain,
        int type,
        int protocol) NN_NOEXCEPT
    {
        *pOutResult = socketsocket(domain, type, protocol, m_ProcessId);

        if (*pOutResult < 0)
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }
    Result ClientImpl::SocketExempt(
        nn::sf::Out<int> pOutResult,
        nn::sf::Out<int> pOutError,
        int domain,
        int type,
        int protocol) NN_NOEXCEPT
    {
        *pOutResult = socketsocketexempt(domain, type, protocol, m_ProcessId);

        if (*pOutResult < 0)
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }
    Result ClientImpl::Open(
        nn::sf::Out<int> pOutResult,
        nn::sf::Out<int> pOutError,
        const nn::sf::InBuffer& devicePath,
        int flags) NN_NOEXCEPT
    {
        const char*  path = GetPointer(devicePath);
        size_t pathLength = devicePath.GetSize();

        if (pathLength <= NetworkDevicePathLength)
        {
            *pOutResult = socketopen(path, flags, m_ProcessId);
            if (*pOutResult < 0)
            {
                *pOutError = errno;
            }
        }
        else
        {
            *pOutError = EINVAL;
            *pOutResult = -1;
        }

        return ResultSuccess();
    }

    Result ClientImpl::Select(
            nn::sf::Out<int> pOutRet,
            nn::sf::Out<int> pOutError,
            int numberOfDescriptors,
            const nn::sf::InBuffer& inReadDescriptors,
            const nn::sf::InBuffer& inWriteDescriptors,
            const nn::sf::InBuffer& inExceptDescriptors,
            const nn::sf::OutBuffer& outReadDescriptors,
            const nn::sf::OutBuffer& outWriteDescriptors,
            const nn::sf::OutBuffer& outExceptDescriptors,
            const nn::socket::sf::SelectTimeval& selTimeout) NN_NOEXCEPT
    {
        fd_set*  outRead       = reinterpret_cast<fd_set*>(GetPointer(outReadDescriptors));
        fd_set*  outWrite      = reinterpret_cast<fd_set*>(GetPointer(outWriteDescriptors));
        fd_set*  outExcept     = reinterpret_cast<fd_set*>(GetPointer(outExceptDescriptors));
        fd_set   read;
        fd_set   write;
        fd_set   except;
        timeval  timeout;

        std::memcpy(
                reinterpret_cast<void*>(&read),
                GetPointer(inReadDescriptors),
                std::min(sizeof(read), inReadDescriptors.GetSize())
        );

        std::memcpy(
                reinterpret_cast<void*>(&write),
                GetPointer(inWriteDescriptors),
                std::min(sizeof(write), inWriteDescriptors.GetSize())
        );

        std::memcpy(
                reinterpret_cast<void*>(&except),
                GetPointer(inExceptDescriptors),
                std::min(sizeof(except), inExceptDescriptors.GetSize())
        );

        timeout.tv_sec = static_cast<time_t>(selTimeout.sec);
        timeout.tv_usec = static_cast<suseconds_t>(selTimeout.usec);

        *pOutRet = socketselect(
                    numberOfDescriptors,
                    inReadDescriptors.GetSize()   != 0     ? &read    : nullptr,
                    inWriteDescriptors.GetSize()  != 0     ? &write   : nullptr,
                    inExceptDescriptors.GetSize() != 0     ? &except  : nullptr,
                    selTimeout.wasNull            == false ? &timeout : nullptr,
                    m_ProcessId);

        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }
        else
        {
            std::memcpy(
                    reinterpret_cast<void*>(outRead),
                    &read,
                    std::min(sizeof(read), outReadDescriptors.GetSize())
            );

            std::memcpy(
                    reinterpret_cast<void*>(outWrite),
                    &write,
                    std::min(sizeof(write), outWriteDescriptors.GetSize())
            );

            std::memcpy(
                    reinterpret_cast<void*>(outExcept),
                    &except,
                    std::min(sizeof(except), outExceptDescriptors.GetSize())
            );
        }

        return ResultSuccess();
    }

    Result ClientImpl::Poll(
            nn::sf::Out<int> pOutRet,
            nn::sf::Out<int> pOutError,
            const nn::sf::InBuffer&  inDescriptors,
            const nn::sf::OutBuffer& outDescriptors,
            int numberOfDescriptors,
            int timeoutMilliseconds) NN_NOEXCEPT
    {
        pollfd* socketDescriptors       = reinterpret_cast<pollfd*>(GetPointer(outDescriptors));
        size_t  socketDescriptorsLength = outDescriptors.GetSize();

        if (numberOfDescriptors  * sizeof(pollfd) <= socketDescriptorsLength)
        {
            std::memcpy(
                reinterpret_cast<void*>(socketDescriptors),
                GetPointer(inDescriptors),
                std::min(socketDescriptorsLength, inDescriptors.GetSize())
            );

            *pOutRet = socketpoll(
                socketDescriptors, numberOfDescriptors, timeoutMilliseconds, m_ProcessId);
            if (*pOutRet < 0)
            {
                *pOutError = errno;
            }
        }
        else
        {
            *pOutError = EINVAL;
        }

        return ResultSuccess();
    }

    Result ClientImpl::Sysctl(
            nn::sf::Out<int> pOutRet,
            nn::sf::Out<int> pOutError,
            const nn::sf::InBuffer& mib,
            const nn::sf::OutBuffer& outValue,
            nn::sf::Out<unsigned int> pOutValueLength,
            const nn::sf::InBuffer& inValue) NN_NOEXCEPT
    {
        int*   pMibEntries    = reinterpret_cast<int*>(const_cast<char*>(GetPointer(mib)));
        void*  pOldValue      = GetPointer(outValue);
        char*  pNewValue      = const_cast<char*>(GetPointer(inValue));
        size_t mibEntryCount  = mib.GetSize() / sizeof(int);
        size_t oldValueLength = outValue.GetSize();
        size_t newValueLength = inValue.GetSize();

        *pOutRet = socketsysctl(
            pMibEntries, mibEntryCount, pOldValue, &oldValueLength,
            pNewValue, newValueLength, m_ProcessId);

        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }
        else
        {
            *pOutValueLength = oldValueLength;
        }

        return ResultSuccess();
    }

    Result ClientImpl::Recv(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        const nn::sf::OutBuffer& outBuffer,
        int flags) NN_NOEXCEPT
    {
        void*  buffer       = GetPointer(outBuffer);
        size_t bufferLength = outBuffer.GetSize();

        *pOutRet = socketrecv(sock, buffer, bufferLength, flags, m_ProcessId);
        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::RecvFrom(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        const nn::sf::OutBuffer& outBuffer,
        int flags,
        const nn::sf::OutBuffer& pOutAddress,
        nn::sf::Out<unsigned int> pOutAddressLength) NN_NOEXCEPT
    {
        void*     buffer           = GetPointer(outBuffer);
        sockaddr* address          = reinterpret_cast<sockaddr*>(GetPointer(pOutAddress));
        size_t    bufferLength     = outBuffer.GetSize();
        socklen_t addressLength    = pOutAddress.GetSize();

        *pOutRet = socketrecvfrom(
            sock, buffer, bufferLength, flags,
            address, &addressLength, m_ProcessId);
        if (0 <= *pOutRet)
        {
            *pOutAddressLength = addressLength;
        }
        else
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::RecvMMsg(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        const nn::sf::OutBuffer& outBuffer,
        size_t vlen,
        int flags,
        const nn::socket::sf::Timespec& timeout) NN_NOEXCEPT
    {
        size_t          numReceivedMessages = 0;
        const size_t    maxMessages         = vlen;
        timespec        ts                  = {static_cast<time_t>(timeout.sec),
                                               static_cast<long>(timeout.nsec)};
        mmsghdr *       pMmsgHdr            = new mmsghdr[maxMessages]();
        iovec *         pIovecs             = new iovec[maxMessages]();
        char **         pBufs               = new char *[maxMessages];
        sockaddr_in *   pNames              = new struct sockaddr_in[maxMessages]();

        if( pMmsgHdr == nullptr || pIovecs == nullptr || pBufs == nullptr || pNames == nullptr )
        {
            *pOutRet = -1;
            *pOutError = ENOMEM;
            goto exit;
        }

        for( int i = 0; i < maxMessages; i++ )
        {
            pBufs[i] = new char[nn::socket::MaxMessageSize]();

            if( pBufs[i] == nullptr )
            {
                *pOutRet = -1;
                *pOutError = ENOMEM;
                goto exit;
            }

            // this is the reason that MaxMessageSize and MaxNumMessages needed to be introduced
            // instead of only having a MaxMessagesBytes - we need to know where to point each iovec to
            pIovecs[i].iov_base             = pBufs[i];
            pIovecs[i].iov_len              = nn::socket::MaxMessageSize;
            pMmsgHdr[i].msg_hdr.msg_iov     = &pIovecs[i];
            pMmsgHdr[i].msg_hdr.msg_iovlen  = 1;
            pMmsgHdr[i].msg_hdr.msg_name    = &pNames[i];
            pMmsgHdr[i].msg_hdr.msg_namelen = sizeof(struct sockaddr_in);
        }

        *pOutRet = numReceivedMessages = socketrecvmmsg(sock, pMmsgHdr, maxMessages, flags, &ts, m_ProcessId);

        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }
        else if (*pOutRet > 0)
        {
            char*   pBuf            = GetPointer(outBuffer);
            size_t  bufferLength    = outBuffer.GetSize();
            char*   pBufEnd         = pBuf + bufferLength;

            for( int msgidx = 0; msgidx < numReceivedMessages; msgidx++ )
            {
                // make sure we are not about to write off the end of the SF buffer
                if( pBuf + pMmsgHdr[msgidx].msg_hdr.msg_namelen
                         + sizeof(pMmsgHdr[msgidx].msg_len)
                         + pMmsgHdr[msgidx].msg_len
                    > pBufEnd )
                {
                    // only return how many messages we copied
                    *pOutRet = msgidx;
                    break;
                }

                std::memcpy(pBuf, &pMmsgHdr[msgidx].msg_hdr.msg_namelen, sizeof(pMmsgHdr[msgidx].msg_hdr.msg_namelen));
                pBuf += sizeof(pMmsgHdr[msgidx].msg_hdr.msg_namelen);
                std::memcpy(pBuf, pMmsgHdr[msgidx].msg_hdr.msg_name, pMmsgHdr[msgidx].msg_hdr.msg_namelen);
                pBuf += pMmsgHdr[msgidx].msg_hdr.msg_namelen;
                std::memcpy(pBuf, &pMmsgHdr[msgidx].msg_len, sizeof(pMmsgHdr[msgidx].msg_len));
                pBuf += sizeof(pMmsgHdr[msgidx].msg_len);
                std::memcpy(pBuf, pIovecs[msgidx].iov_base, pMmsgHdr[msgidx].msg_len);
                pBuf += pMmsgHdr[msgidx].msg_len;

                // TODO: research - do we care about pMmsgHdr[i].msg_hdr.msg_control?
            }
        }

    exit:
        if( pBufs != nullptr )
        {
            for( int i = 0; i < maxMessages; i++ )
            {
                delete[] pBufs[i];
                pBufs[i] = nullptr;
            }
        }

        delete[] pNames;
        pNames = nullptr;
        delete[] pBufs;
        pBufs = nullptr;
        delete[] pIovecs;
        pIovecs = nullptr;
        delete[] pMmsgHdr;
        pMmsgHdr = nullptr;

        return ResultSuccess();
    }

    Result ClientImpl::Send(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        const nn::sf::InBuffer& inBuffer,
        int flags) NN_NOEXCEPT
    {
        const void* buffer       = GetPointer(inBuffer);
        size_t      bufferLength = inBuffer.GetSize();

        *pOutRet = socketsend(
            sock, buffer, bufferLength, flags, m_ProcessId);

        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::SendTo(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        const nn::sf::InBuffer& inBuffer,
        int flags,
        const nn::sf::InBuffer& inAddress) NN_NOEXCEPT
    {
        const void*     buffer        = GetPointer(inBuffer);
        const sockaddr* address       = reinterpret_cast<const sockaddr*>(GetPointer(inAddress));
        size_t          bufferLength  = inBuffer.GetSize();
        socklen_t       addressLength = inAddress.GetSize();

        *pOutRet = socketsendto(
            sock, buffer, bufferLength, flags,
            address, addressLength, m_ProcessId);

        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::SendMMsg(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        const nn::sf::InArray<nn::socket::sf::MsgHdr>& inArrayMsgHdr,
        size_t vlen,
        const nn::sf::InBuffer& inBufferPackedMsgHdrData,
        int flags) NN_NOEXCEPT
    {
        const nn::socket::sf::MsgHdr *  pSfMsghdr           = reinterpret_cast<const nn::socket::sf::MsgHdr *>(GetArrayData(inArrayMsgHdr));
        const char *                    pPackedMsgHdrData   = GetPointer(inBufferPackedMsgHdrData);
        const size_t                    numMessages         = vlen;

        mmsghdr *pMmsgHdr = new mmsghdr[numMessages]();

        if( pMmsgHdr == nullptr )
        {
            *pOutRet = -1;
            *pOutError = ENOMEM;
            goto exit;
        }

        for( int msghdridx = 0; msghdridx < numMessages; msghdridx++ )
        {
            msghdr *pMsgHdr = &pMmsgHdr[msghdridx].msg_hdr;

            pMsgHdr->msg_namelen = pSfMsghdr[msghdridx].msg_namelen;
            pMsgHdr->msg_name = pSfMsghdr[msghdridx].msg_namelen > 0 ?
                                const_cast<char *>(&pPackedMsgHdrData[pSfMsghdr[msghdridx].msg_name])
                                : nullptr;

            pMsgHdr->msg_controllen = pSfMsghdr[msghdridx].msg_controllen;
            pMsgHdr->msg_control = pSfMsghdr[msghdridx].msg_controllen > 0 ?
                                   const_cast<char *>(&pPackedMsgHdrData[pSfMsghdr[msghdridx].msg_control])
                                   : nullptr;

            pMsgHdr->msg_iovlen = pSfMsghdr[msghdridx].msg_iovlen;
            pMsgHdr->msg_iov = new iovec[pMsgHdr->msg_iovlen];
            if( pMsgHdr->msg_iov == nullptr )
            {
                *pOutRet = -1;
                *pOutError = ENOMEM;
                goto exit;
            }

            for( int iovidx = 0; iovidx < pMsgHdr->msg_iovlen; iovidx++ )
            {
                pMsgHdr->msg_iov[iovidx].iov_len = pSfMsghdr[msghdridx].msg_iov[iovidx].iov_len;
                pMsgHdr->msg_iov[iovidx].iov_base = const_cast<char *>(&pPackedMsgHdrData[pSfMsghdr[msghdridx].msg_iov[iovidx].iov_base]);
            }
        }

        *pOutRet = socketsendmmsg(sock, pMmsgHdr, vlen, flags, m_ProcessId);

        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }

    exit:
        if( pMmsgHdr != nullptr )
        {
            for( int msghdridx = 0; msghdridx < numMessages; msghdridx++ )
            {
                delete[] pMmsgHdr[msghdridx].msg_hdr.msg_iov;
                pMmsgHdr[msghdridx].msg_hdr.msg_iov = nullptr;
            }
        }

        delete[] pMmsgHdr;
        pMmsgHdr = nullptr;

        return ResultSuccess();
    }

    Result ClientImpl::Accept(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        const nn::sf::OutBuffer& pOutAddress,
        nn::sf::Out<unsigned int> pOutAddressLength) NN_NOEXCEPT
    {
        sockaddr* address       = reinterpret_cast<sockaddr*>(GetPointer(pOutAddress));
        socklen_t addressLength = pOutAddress.GetSize();

        *pOutRet = socketaccept(
            sock, address, &addressLength, m_ProcessId);

        if (0 <= *pOutRet && *pOutRet < SOMAXCONN)
        {
            *pOutAddressLength = addressLength;
        }
        else
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::Bind(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        const nn::sf::InBuffer& inAddress) NN_NOEXCEPT
    {
        const sockaddr* address = reinterpret_cast<const sockaddr*>(GetPointer(inAddress));
        socklen_t addressLength = inAddress.GetSize();

        *pOutRet = socketbind(sock, address, addressLength, m_ProcessId);

        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }


    Result ClientImpl::Connect(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        const nn::sf::InBuffer& inAddress) NN_NOEXCEPT
    {
        const sockaddr* address = reinterpret_cast<const sockaddr*>(GetPointer(inAddress));
        socklen_t addressLength = inAddress.GetSize();

        *pOutRet = socketconnect(sock, address, addressLength, m_ProcessId);

        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::GetPeerName(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        const nn::sf::OutBuffer& pOutAddress,
        nn::sf::Out<unsigned int> pOutAddressLength) NN_NOEXCEPT
    {
        sockaddr* address       = reinterpret_cast<sockaddr*>(GetPointer(pOutAddress));
        socklen_t addressLength = pOutAddress.GetSize();

        *pOutRet = socketgetpeername(
            sock, address, &addressLength, m_ProcessId);

        if (0 <= *pOutRet)
        {
            *pOutAddressLength = addressLength;
        }
        else
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::GetSockName(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        const nn::sf::OutBuffer& pOutAddress,
        nn::sf::Out<unsigned int> pOutAddressLength) NN_NOEXCEPT
    {
        sockaddr* address       = reinterpret_cast<sockaddr*>(GetPointer(pOutAddress));
        socklen_t addressLength = pOutAddress.GetSize();

        *pOutRet = socketgetsockname(
            sock, address, &addressLength, m_ProcessId);

        if (0 <= *pOutRet)
        {
            *pOutAddressLength = addressLength;
        }
        else
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::GetSockOpt(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        int level,
        int optionName,
        const nn::sf::OutBuffer& pOutOptionValue,
        nn::sf::Out<unsigned int> pOutOptionLength) NN_NOEXCEPT
    {
        void*     pOptionValue  = GetPointer(pOutOptionValue);
        socklen_t optionLength = pOutOptionValue.GetSize();
        MyTimeVal tv = { };
        bool      convertToTimeVal32 = false,
                  convertToTimeVal64 = false;

        if( optionName == SO_SNDTIMEO || optionName == SO_RCVTIMEO )
        {
            if( optionLength == sizeof(TimeVal32) && sizeof(timeval) == sizeof(TimeVal64) )
            {
                convertToTimeVal64 = true;
            }
            else if (optionLength == sizeof(TimeVal64) && sizeof(timeval) == sizeof(TimeVal32) )
            {
                convertToTimeVal32 = true;
            }
        }

        if( true == convertToTimeVal64 )
        {
            pOptionValue = &tv.tv64;
            optionLength = sizeof(tv.tv64);
        }
        else if( true == convertToTimeVal32 )
        {
            pOptionValue = &tv.tv32;
            optionLength = sizeof(tv.tv32);
        }

        *pOutRet = socketgetsockopt(
            sock, level, optionName, pOptionValue, &optionLength, m_ProcessId);

        if( true == convertToTimeVal64 )
        {
            // Make sure there will be no truncation when down-casting to 32 bits.
            // Note: We only need to check tv_sec because tv_usec is always within the range 0-999999.
            if( tv.tv64.tv_sec < INT_MIN || tv.tv64.tv_sec > INT_MAX )
            {
                *pOutRet = -1;
                // EDOM is also the error that is thrown in the case of tv_usec not being within 0-999999
                errno = EDOM;
                goto exit;
            }

            pOptionValue = GetPointer(pOutOptionValue);
            static_cast<TimeVal32 *>(pOptionValue)->tv_sec = static_cast<int32_t>(tv.tv64.tv_sec);
            static_cast<TimeVal32 *>(pOptionValue)->tv_usec = static_cast<int32_t>(tv.tv64.tv_usec);
            optionLength = sizeof(TimeVal32);
        }
        else if( true == convertToTimeVal32 )
        {
            pOptionValue = GetPointer(pOutOptionValue);
            static_cast<TimeVal64 *>(pOptionValue)->tv_sec = tv.tv32.tv_sec;
            static_cast<TimeVal64 *>(pOptionValue)->tv_usec = tv.tv32.tv_usec;
            optionLength = sizeof(TimeVal64);
        }

exit:
        if (0 <= *pOutRet)
        {
            *pOutOptionLength = optionLength;
        }
        else
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::Listen(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        int backlog) NN_NOEXCEPT
    {
        *pOutRet = socketlisten(sock, backlog, m_ProcessId);

        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::Ioctl(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        uint32_t command,
        const nn::sf::InBuffer& dataInSlot0,
        const nn::sf::InBuffer& dataInSlot1,
        const nn::sf::InBuffer& dataInSlot2,
        const nn::sf::InBuffer& dataInSlot3,
        const nn::sf::OutBuffer& dataOutSlot0,
        const nn::sf::OutBuffer& dataOutSlot1,
        const nn::sf::OutBuffer& dataOutSlot2,
        const nn::sf::OutBuffer& dataOutSlot3,
        int embeddedSegmentCount) NN_NOEXCEPT
    {
        const uint32_t SiocGIfConf32    = 0xc0086924;
        const uint32_t SiocGIfConf64    = 0xc0106924;
        const uint32_t SiocGIfMedia32   = 0xc0286938;
        const uint32_t SiocGIfMedia64   = 0xc0306938;

        ifconf ifc = {};
        ifmediareq ifmr = {};
        bool convertIfConfSize = false,
             convertIfMediaReqSize = false;

        // No support for exchange buffers, copy 'in' into 'out',
        // use 'out' in the ioctl call...

        std::memcpy(
                GetPointer(dataOutSlot0),
                GetPointer(dataInSlot0),
                std::min(dataOutSlot0.GetSize(), dataInSlot0.GetSize())
        );

        std::memcpy(
                GetPointer(dataOutSlot1),
                GetPointer(dataInSlot1),
                std::min(dataOutSlot1.GetSize(), dataInSlot1.GetSize())
        );

        std::memcpy(
                GetPointer(dataOutSlot2),
                GetPointer(dataInSlot2),
                std::min(dataOutSlot2.GetSize(), dataInSlot2.GetSize())
        );

        std::memcpy(
                GetPointer(dataOutSlot3),
                GetPointer(dataInSlot3),
                std::min(dataOutSlot3.GetSize(), dataInSlot3.GetSize())
        );

        void*  embeddedBuffers[IoctlEmbeddedSegmentCount] = {
                reinterpret_cast<void*>(GetPointer(dataOutSlot0)),
                reinterpret_cast<void*>(GetPointer(dataOutSlot1)),
                reinterpret_cast<void*>(GetPointer(dataOutSlot2)),
                reinterpret_cast<void*>(GetPointer(dataOutSlot3))
               };

        size_t embeddedBufferLengths[IoctlEmbeddedSegmentCount] = {
                dataOutSlot0.GetSize(),
                dataOutSlot1.GetSize(),
                dataOutSlot2.GetSize(),
                dataOutSlot3.GetSize()
               };

        // if we are a 64-bit process and this is a SIOCGIFCONF command coming from a 32-bit application
        if( SiocGIfConf32 == command && sizeof(ifconf) == IOCPARM_LEN(SiocGIfConf64) )
        {
            command = SiocGIfConf64;
            convertIfConfSize = true;
        }
        // if we are a 32-bit process and this is a SIOCGIFCONF command coming from a 64-bit application
        else if( SiocGIfConf64 == command && sizeof(ifconf) == IOCPARM_LEN(SiocGIfConf32) )
        {
            command = SiocGIfConf32;
            convertIfConfSize = true;
        }
        // if we are a 64-bit process and this is a SIOCGIFMEDIA command coming from a 32-bit application
        else if( SiocGIfMedia32 == command && sizeof(ifmediareq) == IOCPARM_LEN(SiocGIfMedia64) )
        {
            command = SiocGIfMedia64;
            convertIfMediaReqSize = true;
        }
        // if we are a 32-bit process and this is a SIOCGIFMEDIA command coming from a 64-bit application
        else if( SiocGIfMedia64 == command && sizeof(ifmediareq) == IOCPARM_LEN(SiocGIfMedia32) )
        {
            command = SiocGIfMedia32;
            convertIfMediaReqSize = true;
        }

        if( convertIfConfSize )
        {
            ifc.ifc_len = embeddedBufferLengths[1];
            ifc.ifc_buf = static_cast<char *>(embeddedBuffers[1]);

            embeddedBuffers[0] = &ifc;
            embeddedBufferLengths[0] = sizeof(ifc);
        }
        else if( convertIfMediaReqSize )
        {
            std::memcpy(&ifmr, embeddedBuffers[0], sizeof(ifmr) - sizeof(ifmr.ifm_ulist));
            embeddedBuffers[0] = &ifmr;
            embeddedBufferLengths[0] = sizeof(ifmr);
        }

        if (0 <= RestoreIoctlEmbeddedPointers(command, embeddedBuffers, embeddedBufferLengths,
                                              embeddedSegmentCount))
        {
            *pOutRet = socketioctl(
                sock, command, embeddedBuffers[0], embeddedBufferLengths[0], m_ProcessId);
            if (*pOutRet < 0)
            {
                *pOutError = errno;
            }

            if( convertIfMediaReqSize )
            {
                std::memcpy(embeddedBuffers[0], &ifmr, embeddedBufferLengths[0]);
            }
        }
        else
        {
            *pOutRet = -1;
            *pOutError = ENOIOCTL;
        }

        return ResultSuccess();
    }

    Result ClientImpl::Fcntl(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        int command,
        int iocmd) NN_NOEXCEPT
    {
        *pOutRet = socketfcntl(sock, command, iocmd, m_ProcessId);

        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::SetSockOpt(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        int level,
        int optionName,
        const nn::sf::InBuffer& inOptionValue) NN_NOEXCEPT
    {
        const void* pOptionValue  = GetPointer(inOptionValue);
        socklen_t optionLength = inOptionValue.GetSize();
        MyTimeVal tv = { };

        if( optionName == SO_SNDTIMEO || optionName == SO_RCVTIMEO )
        {
            if( optionLength == sizeof(TimeVal32) && sizeof(timeval) == sizeof(TimeVal64) )
            {
                tv.tv64.tv_sec = static_cast<const TimeVal32 *>(pOptionValue)->tv_sec;
                tv.tv64.tv_usec = static_cast<const TimeVal32 *>(pOptionValue)->tv_usec;
                pOptionValue = &tv.tv64;
                optionLength = sizeof(TimeVal64);
            }
            else if (optionLength == sizeof(TimeVal64) && sizeof(timeval) == sizeof(TimeVal32) )
            {
                // Make sure there will be no truncation when down-casting to 32 bits.
                // Note: We only need to check tv_sec because tv_usec is always within the range 0-999999.
                int64_t tv_sec = static_cast<const TimeVal64 *>(pOptionValue)->tv_sec;
                if( tv_sec < INT_MIN || tv_sec > INT_MAX )
                {
                    *pOutRet = -1;
                    // EDOM is also the error that is thrown in the case of tv_usec not being within 0-999999
                    errno = EDOM;
                    goto exit;
                }

                tv.tv32.tv_sec = static_cast<int32_t>(tv_sec);
                tv.tv32.tv_usec = static_cast<int32_t>(static_cast<const TimeVal64 *>(pOptionValue)->tv_usec);
                pOptionValue = &tv.tv32;
                optionLength = sizeof(TimeVal32);
            }
        }

        *pOutRet = socketsetsockopt(
            sock, level, optionName, pOptionValue, optionLength, m_ProcessId);

exit:
        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::Shutdown(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        int how) NN_NOEXCEPT
    {
        *pOutRet = socketshutdown(sock, how, m_ProcessId);

        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::ShutdownAllSockets(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int forced) NN_NOEXCEPT
    {
        int ret;
        *pOutError = clientShutdownAllSockets(&ret, forced);
        if (*pOutError < 0)
        {
            *pOutRet = -1;
        }
        else
        {
            *pOutRet = ret;
        }

        return ResultSuccess();
    }

    Result ClientImpl::Write(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        const nn::sf::InBuffer& inBuffer) NN_NOEXCEPT
    {
        const void* buffer       = GetPointer(inBuffer);
        size_t      bufferLength = inBuffer.GetSize();

        *pOutRet = socketwrite(sock, buffer, bufferLength, m_ProcessId);

        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::Read(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
    {
        void*  buffer       = GetPointer(outBuffer);
        size_t bufferLength = outBuffer.GetSize();

        *pOutRet = socketread(sock, buffer, bufferLength, m_ProcessId);

        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::Close(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock) NN_NOEXCEPT
    {
        *pOutRet = socketclose(sock, m_ProcessId);

        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::DuplicateSocket(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        int sock,
        Bit64 ownerProcessId) NN_NOEXCEPT
    {
        *pOutRet = socketduplicate(sock, ownerProcessId, m_ProcessId);

        if (*pOutRet < 0)
        {
            *pOutError = errno;
        }

        return ResultSuccess();
    }

    Result ClientImpl::GetResourceStatistics(
        nn::sf::Out<int> pOutRet,
        nn::sf::Out<int> pOutError,
        Bit64 pid,
        int32_t type,
        const nn::sf::OutBuffer& outBuffer,
        uint32_t options) NN_NOEXCEPT
    {
        void*  buffer       = GetPointer(outBuffer);
        size_t bufferLength = outBuffer.GetSize();

        *pOutRet = client_get_resource_statistics(pid, (nn::socket::StatisticsType)type, buffer, bufferLength, options);
        if (*pOutRet < 0)
        {
            *pOutError = EINVAL;   // only errors should be invalid type or invalid pid arguments
        }

        return ResultSuccess();
    }

    void ClientImpl::CancelAndWait() NN_NOEXCEPT
    {
        if (m_IsRegistered)
        {
            nn::socket::resolver::CancelAll(m_ProcessId);

            void* privateDate;
            int   result;

            result = clientTerminate(m_ProcessId, &privateDate);
            NN_ABORT_UNLESS_EQUAL(result, 0);
            NN_ABORT_UNLESS_NOT_NULL(privateDate);

            auto* transferMemory = reinterpret_cast<nn::os::TransferMemoryType*>(privateDate);
            if (transferMemory != nullptr)
            {
                nn::os::UnmapTransferMemory(transferMemory);
                nn::os::DestroyTransferMemory(transferMemory);
                free(privateDate, M_DEVBUF);
            }

            m_IsRegistered = false;
        }
    }

}}} // namespace nn::socket::detail
