﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/os/os_SdkThreadCommon.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/wlan/wlan_SocketApi.h>
#include <nn/eth/eth_Result.public.h>
#include <nn/psc.h>
#include <net/if_dl.h>

#include "bsdsocket_Wlan.h"

namespace nn   {
namespace wlan {
#include <nn/wlan/wlan_Result.public.h>  // for nn::wlan::ResultNotAuthorized
#include <nn/wlan/wlan_Result.private.h> // for nn::wlan::ResultRxEntryIsMuted
}}

namespace nn   {
namespace eth  {
namespace wlan {
namespace      {

NN_OS_ALIGNAS_THREAD_STACK Wlan g_Wlan;
WorkerThread g_InitThread;
nn::os::EventType g_TerminateEvent;

} // namespace

void Wlan::SetLinkUp() NN_NOEXCEPT
{
    m_LinkState = LINK_STATE_UP;
    if_link_state_change(m_pIfnet, m_LinkState);
    if_up(m_pIfnet);
    nn::os::SignalEvent(&m_ReceiveEvent);
    NN_SDK_LOG("[wlan]: up request\n");
}

void Wlan::SetLinkDown() NN_NOEXCEPT
{
    if( m_LinkState == LINK_STATE_UP )
    {
        nn::wlan::Socket::CancelGetFrame(m_RxId);
    }
    m_LinkState = LINK_STATE_DOWN;

    if_link_state_change(m_pIfnet, m_LinkState);
    if_down(m_pIfnet);
    NN_SDK_LOG("[wlan]: down request\n");
}

int Wlan::Ioctl(uint32_t command, char* pData)
NN_NOEXCEPT
{
    int result = 0;

    switch (command)
    {
    case SIOCSIFMTU:
        m_pIfnet->if_mtu = (reinterpret_cast<ifreq *>(pData))->ifr_mtu;
        break;
    case SIOCSIFFLAGS:
        if ((reinterpret_cast<ifreq *>(pData))->ifr_flags & IFF_UP)
        {
            if (!(m_pIfnet->if_drv_flags & IFF_DRV_RUNNING))
            {
                m_pIfnet->if_drv_flags &= ~IFF_DRV_OACTIVE;
                m_pIfnet->if_drv_flags |=  IFF_DRV_RUNNING;
            }
            SetLinkUp();
        }
        else
        {
            if (m_pIfnet->if_drv_flags & IFF_DRV_RUNNING)
            {
                m_pIfnet->if_drv_flags &= ~(IFF_DRV_RUNNING | IFF_DRV_OACTIVE);
            }
            SetLinkDown();
        }
        break;
    case SIOCADDMULTI:
    case SIOCDELMULTI:
    {
        struct ifmultiaddr *ifma;
        uint8_t *address;
        nn::wlan::MulticastFilterInfo filterInfo;

        filterInfo.count = 0;
        TAILQ_FOREACH(ifma, &m_pIfnet->if_multiaddrs, ifma_link)
        {
            if (ifma->ifma_addr->sa_family != AF_LINK)
            {
                continue;
            }
            address = reinterpret_cast<uint8_t *>(LLADDR(reinterpret_cast<struct sockaddr_dl *>(ifma->ifma_addr)));
            filterInfo.address[filterInfo.count].Set(address);
            filterInfo.count++;
        }
        nn::wlan::Socket::SetMulticastFilter(filterInfo);
    }
        break;
    case SIOCGIFMEDIA:
        {
            struct ifmediareq *p_ifmr = reinterpret_cast<struct ifmediareq *>(pData);
            p_ifmr->ifm_status = IFM_AVALID | ((m_LinkState == LINK_STATE_UP) ? IFM_ACTIVE : 0);
            p_ifmr->ifm_active = IFM_ETHER;
            p_ifmr->ifm_count  = 0;
            p_ifmr->ifm_ulist  = nullptr;
            break;
        }
    case SIOCSIFMEDIA:
        break;
    case SIOCSIFCAP:
        break;
    default:
        result = ether_ioctl(m_pIfnet, command, pData);
        break;
    }

    return result;
}

void Wlan::Start()
NN_NOEXCEPT
{
    nn::os::SignalEvent(&m_SendEvent);
}

void Wlan::Init()
NN_NOEXCEPT
{
    nn::os::SignalEvent(&m_ReceiveEvent);
}

nn::Result Wlan::SendFrameToStack(uint8_t* pData, uint32_t length)
NN_NOEXCEPT
{
    struct mbuf* pMbuf;

    pMbuf = m_devget(reinterpret_cast<caddr_t>(pData), length, ETHER_ALIGN, m_pIfnet, NULL);

    if (pMbuf != NULL)
    {
        m_pIfnet->if_input(m_pIfnet, pMbuf);
        m_pIfnet->if_ipackets++;
    }

    return ResultSuccess();
}

nn::Result Wlan::GetFrameFromStack(uint32_t* pLengthOut, uint8_t* pData, uint32_t length)
NN_NOEXCEPT
{
    struct mbuf* pMbuf;

    *pLengthOut = 0;

    IFQ_DRV_LOCK(&m_pIfnet->if_snd);
    if (!IFQ_DRV_IS_EMPTY(&m_pIfnet->if_snd))
    {
        IFQ_DRV_DEQUEUE(&m_pIfnet->if_snd, pMbuf);

        if (pMbuf != nullptr)
        {
            if (pMbuf->m_pkthdr.len <= length)
            {
                *pLengthOut = pMbuf->m_pkthdr.len;
                m_copydata(pMbuf, 0, pMbuf->m_pkthdr.len, reinterpret_cast<caddr_t>(pData));
            }
            m_freem(pMbuf);
        }
    }
    IFQ_DRV_UNLOCK(&m_pIfnet->if_snd);

    return ResultSuccess();
}

#if ENABLE_SHARED_MEMORY

void Wlan::ReceiveFunction()
NN_NOEXCEPT
{
    nn::Result       result;
    nn::cbuf::Frame* pFrame;

    while (m_Running)
    {
        nn::os::WaitEvent(&m_ReceiveEvent);

        while (m_Running && m_LinkState == LINK_STATE_UP)
        {
            m_RxPacketEvent.Wait();

            while ((pFrame = m_pRxCbuf->GetReadPointer()) != nullptr)
            {
                SendFrameToStack(pFrame->data, pFrame->size);
                m_pRxCbuf->AcknowledgeRead();
                m_TxCbufEvent.Signal();
            }
        }
    }
}

void Wlan::SendFunction()
NN_NOEXCEPT
{
    nn::Result result;
    uint32_t   length;

    while (m_Running)
    {
        nn::os::WaitEvent(&m_SendEvent);

        while (m_Running && m_LinkState == LINK_STATE_UP)
        {
            nn::cbuf::Frame* pFrame = m_pTxCbuf->GetWritePointer();

            if (pFrame == nullptr)
            {
                m_TxPacketEvent.Signal();
                m_RxCbufEvent.Wait();
                continue;
            }

            if (GetFrameFromStack(
                        &length,
                        pFrame->data,
                        sizeof(pFrame->data)
                        ).IsFailure() || length == 0)
            {
                m_TxPacketEvent.Signal();
                break;
            }

            pFrame->size = length;
            m_pTxCbuf->CommitWrite();
        }
    }
}

#else

void Wlan::ReceiveFunction()
NN_NOEXCEPT
{
    nn::Result       result;
    uintptr_t    packetLength;
    struct mbuf* pMbuf;

    while (m_Running)
    {
        nn::os::WaitEvent(&m_ReceiveEvent);

        while (m_Running && m_LinkState == LINK_STATE_UP)
        {
            result = nn::wlan::Socket::GetFrameRaw(m_ReceiveBuffer, sizeof(m_ReceiveBuffer), &packetLength, m_RxId);
            if (result <= nn::wlan::ResultRxEntryIsMuted())
            {
                // link is down, stop processing packets
                NN_SDK_LOG("[wlan]: link is down\n");
                break;
            }
            else if (result <= nn::wlan::ResultGetFrameCancelled())
            {
                // can only happen when link is forced to LINK_STATE_DOWN,
                // which will drop us out of the innner loop and will block thread on
                // receive event
                continue;
            }
            else if (result.IsFailure())
            {
                // all other WLAN errors are recoverable, ignore and continue...
                NN_SDK_LOG("[wlan]: receive returned %d:%d\n", result.GetModule(), result.GetDescription());
                m_pIfnet->if_ierrors++;
                continue;
            }

            pMbuf = m_devget((char*)(m_ReceiveBuffer), packetLength, ETHER_ALIGN, m_pIfnet, nullptr);
            if (pMbuf != nullptr)
            {
                m_pIfnet->if_input(m_pIfnet, pMbuf);
                m_pIfnet->if_ipackets++;
            }
            else
            {
                NN_SDK_LOG("[wlan]: out of mbufs\n");
                m_pIfnet->if_ierrors++;
            }
        }
    }
}

void Wlan::SendFunction()
NN_NOEXCEPT
{
    struct mbuf* pMbuf;
    uint32_t     packetLength;
    nn::Result   result;

    while (m_Running)
    {
        // allow stack to call Start()
        m_pIfnet->if_drv_flags &= ~IFF_DRV_OACTIVE;

        nn::os::WaitEvent(&m_SendEvent);

        while (m_Running && m_LinkState == LINK_STATE_UP)
        {
            // prevent stack from calling Start() while we are processing packets
            m_pIfnet->if_drv_flags |= IFF_DRV_OACTIVE;

            if (IFQ_DRV_IS_EMPTY(&m_pIfnet->if_snd))
            {
                break;
            }

            IFQ_DRV_DEQUEUE(&m_pIfnet->if_snd, pMbuf);

            if (pMbuf == nullptr)
            {
                break;
            }

            packetLength = pMbuf->m_pkthdr.len;
            NN_ABORT_UNLESS(packetLength <= sizeof(m_SendBuffer));

            m_copydata(pMbuf, 0, packetLength, (caddr_t)m_SendBuffer);
            m_freem(pMbuf);
            m_pIfnet->if_opackets++;

            result = nn::wlan::Socket::PutFrameRaw(m_SendBuffer, packetLength);
            if (result <= nn::wlan::ResultNotAuthorized())
            {
                // link is down, stop processing packets
                NN_SDK_LOG("[wlan]: link is down\n");
                break;
            }
            else if (result.IsFailure())
            {
                // all other WLAN errors are recoverable, ignore and continue...
                NN_SDK_LOG("[wlan]: send returned %d:%d\n", result.GetModule(), result.GetDescription());
                m_pIfnet->if_oerrors++;
            }
        }
    }
}

#endif

nn::Result Wlan::ReadMacAddress()
NN_NOEXCEPT
{
    const int ReadMacPollPeriodMs  = 500;
    const int ReadMacPollTimeoutMs = 5000;

    int macReadRetries = ReadMacPollTimeoutMs / ReadMacPollPeriodMs;
    nn::Result result;

    do
    {
        result = nn::wlan::Socket::GetMacAddress(&m_MacAddress);

        NN_ABORT_UNLESS(
            result.IsSuccess(),
            "[wlan]: failed to get mac address - %d:%d\n",
            result.GetModule(),
            result.GetDescription()
        );

        m_pMacAddressData = m_MacAddress.GetMacAddressData();

        if (!(m_pMacAddressData[0] == 0 &&
              m_pMacAddressData[1] == 0 &&
              m_pMacAddressData[2] == 0 &&
              m_pMacAddressData[3] == 0 &&
              m_pMacAddressData[4] == 0 &&
              m_pMacAddressData[5] == 0))
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ReadMacPollPeriodMs));

        if (macReadRetries-- == 0)
        {
            // did not get valid MAC from wlan process
            return ResultInvalidMac();
        }

    } while (NN_STATIC_CONDITION(true));

    return ResultSuccess();
}

nn::Result Wlan::Initialize()
NN_NOEXCEPT
{
    nn::Result       result;

    if (m_Initialized)
    {
        return ResultSuccess();
    }

    if ((result = nn::wlan::InitializeSocketManager()).IsFailure())
    {
        return result;
    }

    if ((m_pIfnet = if_alloc(IFT_ETHER)) == nullptr)
    {
        nn::wlan::FinalizeSocketManager();
        return ResultOutOfMemory();
    }

#if ENABLE_SHARED_MEMORY
    NN_STATIC_ASSERT(2 * sizeof(nn::cbuf::Cbuf) < SharedBufferSize);

    result = nn::os::CreateSharedMemory(
                        &m_ServerSharedMemory,
                        SharedBufferSize,
                        nn::os::MemoryPermission_ReadWrite,
                        nn::os::MemoryPermission_ReadWrite);

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    m_SharedBuffer = (uint8_t*)nn::os::MapSharedMemory(
                        &m_ServerSharedMemory,
                        nn::os::MemoryPermission_ReadWrite);

    NN_ABORT_UNLESS(m_SharedBuffer != nullptr);

    if ((result = nn::wlan::RegisterSharedMemory(
                        nn::os::GetSharedMemoryHandle(&m_ServerSharedMemory),
                        SharedBufferSize,
                        m_TxPacketEvent.GetReadableHandle(),
                        m_RxPacketEvent.GetWritableHandle(),
                        m_TxCbufEvent.GetReadableHandle(),
                        m_RxCbufEvent.GetWritableHandle())).IsFailure())
    {
        return result;
    }

    m_pTxCbuf = reinterpret_cast<nn::cbuf::Cbuf*>(m_SharedBuffer);
    m_pRxCbuf = reinterpret_cast<nn::cbuf::Cbuf*>(m_SharedBuffer + SharedBufferSize / 2);
    m_pTxCbuf->Initialize();
    m_pRxCbuf->Initialize();
#endif

    m_Initialized = true;

    nn::os::InitializeEvent(&m_SendEvent,    false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_ReceiveEvent, false, nn::os::EventClearMode_AutoClear);

    m_Running = true;
    m_ReceiveThread.Initialize(std::bind(&Wlan::ReceiveFunction, this),
                               NN_SYSTEM_THREAD_PRIORITY(socket, WlanInterfaceIngress),
                               NN_SYSTEM_THREAD_NAME(socket, WlanInterfaceIngress));
    m_SendThread.Initialize(std::bind(&Wlan::SendFunction, this),
                            NN_SYSTEM_THREAD_PRIORITY(socket, WlanInterfaceEgress),
                            NN_SYSTEM_THREAD_NAME(socket, WlanInterfaceEgress));

    if_initname(m_pIfnet, Name, Instance);

    m_pIfnet->if_softc              = this;
    m_pIfnet->if_flags              = IFF_BROADCAST | IFF_MULTICAST;
    m_pIfnet->if_capabilities       = 0;
    m_pIfnet->if_hwassist           = 0;
    m_pIfnet->if_capenable          = m_pIfnet->if_capabilities;
    m_pIfnet->if_ioctl              = Ioctl;
    m_pIfnet->if_start              = Start;
    m_pIfnet->if_init               = Init;
    m_pIfnet->if_snd.ifq_drv_maxlen = ifqmaxlen;
    m_pIfnet->if_drv_flags          = 0;

    IFQ_SET_MAXLEN(&m_pIfnet->if_snd, ifqmaxlen);
    IFQ_SET_READY(&m_pIfnet->if_snd);

    result = nn::wlan::Socket::CreateRxEntry(
                &m_RxId,
                PacketTypes,
                sizeof(PacketTypes) / sizeof(PacketTypes[0]),
                NumberOfBuffers);

    NN_ABORT_UNLESS(
                result.IsSuccess(),
                "[wlan]: failed to create RX entry - %d:%d\n",
                result.GetModule(),
                result.GetDescription());

#if ENABLE_SHARED_MEMORY
    m_pRxCbuf->SetRxId(m_RxId);

    nn::wlan::EnableSharedMemory();
#endif

    result = ReadMacAddress();
    if (result.IsSuccess())
    {
        ether_ifattach(m_pIfnet, m_pMacAddressData);
        m_AttachedToSocketProcess = true;
    }

    NN_SDK_LOG("[wlan]: %02x:%02x:%02x:%02x:%02x:%02x attached to socket process\n",
               m_pMacAddressData[0],
               m_pMacAddressData[1],
               m_pMacAddressData[2],
               m_pMacAddressData[3],
               m_pMacAddressData[4],
               m_pMacAddressData[5]);

    return result;
} // NOLINT(impl/function_size)

void Wlan::Finalize()
NN_NOEXCEPT
{
    if (!m_Initialized)
    {
        return;
    }

    m_Initialized = false;
    SetLinkDown();

    m_Running = false;

    nn::wlan::Socket::CancelGetFrame(m_RxId);

    SignalEvent(&m_ReceiveEvent);
    m_ReceiveThread.Finalize();

    SignalEvent(&m_SendEvent);
    m_SendThread.Finalize();

    nn::os::FinalizeEvent(&m_ReceiveEvent);
    nn::os::FinalizeEvent(&m_SendEvent);

    nn::wlan::Socket::DeleteRxEntry(m_RxId);

#if ENABLE_SHARED_MEMORY
    nn::wlan::UnregisterSharedMemory();
#endif

    nn::wlan::FinalizeSocketManager();

    if (m_AttachedToSocketProcess)
    {
        ether_ifdetach(m_pIfnet);
    }

    if_free(m_pIfnet);

    m_AttachedToSocketProcess = false;
    m_pMacAddressData = nullptr;
}

void Wlan::ControlThread()
NN_NOEXCEPT
{
    nn::Result result;

    if ((result = Initialize()).IsFailure())
    {
        NN_SDK_LOG("[wlan]: failed to init wlan bsd shim - %d:%d\n",
                   result.GetModule(),
                   result.GetDescription());
    }
    else
    {
        nn::os::MultiWaitType       multiWait;
        nn::os::MultiWaitHolderType terminateEventHolder;
        nn::os::MultiWaitHolderType wlanPmEventHolder;

        nn::psc::PmModule           wlanPmModule;
        nn::psc::PmState            state;
        nn::psc::PmFlagSet          flags;

        result = wlanPmModule.Initialize(
                    nn::psc::PmModuleId_SocketWlan,
                    NULL,
                    0,
                    nn::os::EventClearMode_ManualClear);

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&wlanPmEventHolder, wlanPmModule.GetEventPointer()->GetBase());
        nn::os::InitializeMultiWaitHolder(&terminateEventHolder, &g_TerminateEvent);
        nn::os::LinkMultiWaitHolder(&multiWait, &wlanPmEventHolder);
        nn::os::LinkMultiWaitHolder(&multiWait, &terminateEventHolder);

        while (NN_STATIC_CONDITION(true))
        {
            nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&multiWait);

            if (holder == &terminateEventHolder)
            {
                break;
            }
            else if (holder == &wlanPmEventHolder)
            {
                wlanPmModule.GetEventPointer()->Clear();

                result = wlanPmModule.GetRequest(&state, &flags);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                //We have no work to do, so just acknowledge that we are done proccessing the event.

                result = wlanPmModule.Acknowledge(state, ResultSuccess());
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }

        nn::os::UnlinkAllMultiWaitHolder(&multiWait);
        nn::os::FinalizeMultiWaitHolder(&wlanPmEventHolder);
        nn::os::FinalizeMultiWaitHolder(&terminateEventHolder);
        nn::os::FinalizeMultiWait(&multiWait);

        result = wlanPmModule.Finalize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    Finalize();
}

#if ENABLE_WLAN_INTERFACE

void InitializeWlan()
NN_NOEXCEPT
{
    nn::os::InitializeEvent(&g_TerminateEvent, false, nn::os::EventClearMode_AutoClear);
    // because of the poll for mac address, offload to a thread
    g_InitThread.Initialize(std::bind(&Wlan::ControlThread, &g_Wlan),
                            NN_SYSTEM_THREAD_PRIORITY(socket, WlanInit),
                            NN_SYSTEM_THREAD_NAME(socket, WlanInit));
}

void FinalizeWlan()
NN_NOEXCEPT
{
    nn::os::SignalEvent(&g_TerminateEvent);
    g_InitThread.Finalize();
}

#else

void InitializeWlan()
NN_NOEXCEPT
{
}

void FinalizeWlan()
NN_NOEXCEPT
{
}

#endif

}}}
