﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/tsc/tsc_Result.h>

#include <nn/tsc/tsc_Types.h>
#include <nn/tsc/tsc_ActiveConfigContext.h>

#include "tsc_TcpipStackConfigBase.h"
#include "tsc_Util.h"

namespace nn { namespace tsc {

    // Pointer to the internal base config object defined in tsc_Library.h
    extern class detail::TcpIpStackConfigBase* g_pObjStackConfig;

    ActiveConfigContext::ActiveConfigContext(const char* pInInterfaceName)
    {
        memset(m_InterfaceName, 0x00, g_MaxInterfaceNameLength);

        if(pInInterfaceName != nullptr)
        {
            nn::Result result = this->SetInterfaceName(pInInterfaceName);
            NN_SDK_ASSERT(result.IsSuccess());
            NN_UNUSED(result);
        }
    }

    ActiveConfigContext::~ActiveConfigContext()
    {
    }

    Result ActiveConfigContext::SetInterfaceName(const char* pInInterfaceName) NN_NOEXCEPT
    {
        NN_DETAIL_TSC_RETURN_ON_UNINITIALIZED_STATE();
        NN_SDK_REQUIRES_NOT_NULL(pInInterfaceName);

        uint32_t nameLength = strnlen(pInInterfaceName, g_MaxInterfaceNameLength);
        if (nameLength > g_MaxInterfaceNameLength - 1)
        {
            return tsc::ResultInterfaceNameInvalid();
        }

        strncpy(m_InterfaceName,
                pInInterfaceName,
                NN_DETAIL_TSC_GET_MIN(g_MaxInterfaceNameLength, nameLength));

        return nn::ResultSuccess();
    }

    Result ActiveConfigContext::GetInterfaceName(char* pOutInterfaceName, int nameBufferLen) NN_NOEXCEPT
    {
        NN_DETAIL_TSC_RETURN_ON_UNINITIALIZED_STATE();
        NN_SDK_REQUIRES_NOT_NULL(pOutInterfaceName);

        int nameLength = strnlen(m_InterfaceName, g_MaxInterfaceNameLength);
        if(nameLength > nameBufferLen - 1)
        {
            return tsc::ResultBufferTooShort();
        }

        strncpy(pOutInterfaceName,
                m_InterfaceName,
                NN_DETAIL_TSC_GET_MIN(g_MaxInterfaceNameLength, nameBufferLen));

        return nn::ResultSuccess();
    }

    Result ActiveConfigContext::GetInterfaceAddress(Ipv4AddrStorage* pOutAddress) NN_NOEXCEPT
    {
        detail::TcpIpStackConfigBase* pConfigObj = nullptr;
        NN_DETAIL_TSC_GET_CONFIG_OR_RETURN_ON_UNINITIALIZED_STATE(pConfigObj);
        NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
        NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifyInterfaceName(m_InterfaceName));

        return pConfigObj->GetActiveInterfaceAddress(m_InterfaceName, pOutAddress);
    }

    Result ActiveConfigContext::GetSubnetMask(Ipv4AddrStorage* pOutAddress) NN_NOEXCEPT
    {
        detail::TcpIpStackConfigBase* pConfigObj = nullptr;
        NN_DETAIL_TSC_GET_CONFIG_OR_RETURN_ON_UNINITIALIZED_STATE(pConfigObj);
        NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
        NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifyInterfaceName(m_InterfaceName));

        return pConfigObj->GetActiveSubnetMask(m_InterfaceName, pOutAddress);
    }

    Result ActiveConfigContext::GetDefaultGateway(Ipv4AddrStorage* pOutAddress) NN_NOEXCEPT
    {
        detail::TcpIpStackConfigBase* pConfigObj = nullptr;
        NN_DETAIL_TSC_GET_CONFIG_OR_RETURN_ON_UNINITIALIZED_STATE(pConfigObj);
        NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
        NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifyInterfaceName(m_InterfaceName));

        return pConfigObj->GetActiveDefaultGateway(m_InterfaceName, pOutAddress);
    }

    Result ActiveConfigContext::GetPreferredDns(Ipv4AddrStorage* pOutAddress) NN_NOEXCEPT
    {
        detail::TcpIpStackConfigBase* pConfigObj = nullptr;
        NN_DETAIL_TSC_GET_CONFIG_OR_RETURN_ON_UNINITIALIZED_STATE(pConfigObj);
        NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
        NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifyInterfaceName(m_InterfaceName));

        return pConfigObj->GetActivePreferredDns(m_InterfaceName, pOutAddress);
    }

    Result ActiveConfigContext::GetAlternativeDns(Ipv4AddrStorage* pOutAddress) NN_NOEXCEPT
    {
        detail::TcpIpStackConfigBase* pConfigObj = nullptr;
        NN_DETAIL_TSC_GET_CONFIG_OR_RETURN_ON_UNINITIALIZED_STATE(pConfigObj);
        NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
        NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifyInterfaceName(m_InterfaceName));

        return pConfigObj->GetActiveAlternativeDns(m_InterfaceName, pOutAddress);
    }

    Result ActiveConfigContext::GetMtu(uint32_t* pOutMtuValue) NN_NOEXCEPT
    {
        detail::TcpIpStackConfigBase* pConfigObj = nullptr;
        NN_DETAIL_TSC_GET_CONFIG_OR_RETURN_ON_UNINITIALIZED_STATE(pConfigObj);
        NN_SDK_REQUIRES_NOT_NULL(pOutMtuValue);
        NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifyInterfaceName(m_InterfaceName));

        return pConfigObj->GetActiveMtu(m_InterfaceName, pOutMtuValue);
    }

    Result ActiveConfigContext::GetInterfaceStatistics(InterfaceStatistics* pOutStatsValue) NN_NOEXCEPT
    {
        detail::TcpIpStackConfigBase* pConfigObj = nullptr;
        NN_DETAIL_TSC_GET_CONFIG_OR_RETURN_ON_UNINITIALIZED_STATE(pConfigObj);
        NN_SDK_REQUIRES_NOT_NULL(pOutStatsValue);
        NN_DETAIL_TSC_RETURN_ON_ERROR(detail::VerifyInterfaceName(m_InterfaceName));

        return pConfigObj->GetInterfaceStatistics(m_InterfaceName, pOutStatsValue);
    }


}}
