﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <mutex>
#include <nn/os/os_Mutex.h>
#include <nn/spl/spl_Api.h>
#include <nn/spl/spl_Result.h>
#include <nn/spl/spl_Types.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ExpHeapAllocator.h>

#include "spl_InterfaceHolder.h"

namespace nn { namespace spl {

    namespace
    {
        struct StaticMutex
        {
            nn::os::MutexType mutex;

            void lock() NN_NOEXCEPT
            {
                nn::os::LockMutex(&mutex);
            }

            void unlock() NN_NOEXCEPT
            {
                nn::os::UnlockMutex(&mutex);
            }
        };

        InterfaceHolder g_InterfaceHolder;
        StaticMutex g_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };
        int g_InitializeCount = 0;

        const Bit8 DecryptDeviceUniqueDataKekSource[] =
        {
            0x7f,0x5b,0xb0,0x84,0x7b,0x25,0xaa,0x67,0xfa,0xc8,0x4b,0xe2,0x3d,0x7b,0x69,0x03
        };
        const Bit8 NfpEccBn128KeySource[16] =
        {
            0xec,0xa8,0xac,0x1e,0x36,0x5c,0x67,0x4f,0xe6,0xaf,0x8f,0x07,0xe3,0xba,0xff,0x09
        };
        const Bit8 NfpEccP160KeySource[16] =
        {
            0x00,0xde,0x5b,0xb9,0xd2,0x1e,0x45,0xbc,0xf9,0xe1,0xc5,0x29,0xde,0xda,0x24,0x41
        };
        const Bit8 DrmDeviceCertEccKeySourceForDevelop[16] =
        {
            0x77,0xa0,0x3a,0x30,0x6a,0xb1,0xcb,0x1c,0x18,0x1c,0x57,0xf0,0x64,0x34,0xa5,0x14
        };
        const Bit8 DrmDeviceCertEccKeySourceForEnd[16] =
        {
            0xaf,0x44,0xf3,0x3e,0x82,0x4e,0x83,0x92,0xed,0x38,0xe1,0x2f,0x29,0xcf,0x6f,0x4d
        };
        const Bit8 DecryptAndStoreGcKeyKekSource[] =
        {
            0xd0,0xcc,0xde,0x8d,0x10,0x98,0x67,0x2f,0x64,0x9a,0x1e,0xc4,0xfe,0x85,0x62,0xf0
        };
        const Bit8 GcKeySourceForDevelop[] =
        {
            0x1b,0xa8,0x34,0xd6,0xbc,0x23,0xd1,0xa2,0xb2,0x95,0x1f,0x37,0xaa,0xd0,0x01,0x42
        };
        const Bit8 GcKeySourceForEnd[] =
        {
            0xc3,0xb5,0x86,0x0a,0xe9,0xf7,0x80,0xf5,0xaf,0xa8,0x49,0x2d,0xd4,0x33,0xb5,0xa9
        };
        const Bit8 LoadEsDeviceKeyKekSource[] =
        {
            0x46,0x6e,0x57,0xb7,0x4a,0x44,0x7f,0x02,0xf3,0x21,0xcd,0xe5,0x8f,0x2f,0x55,0x35
        };
        const Bit8 EsDeviceKeySourceForDevelop[] =
        {
            0xbe,0xc0,0xbc,0x8e,0x75,0xa0,0xf6,0x0c,0x4a,0x56,0x64,0x02,0x3e,0xd4,0x9c,0xd5
        };
        const Bit8 EsDeviceKeySourceForEnd[] =
        {
            0xdb,0xa4,0x51,0x12,0x4c,0xa0,0xa9,0x83,0x68,0x14,0xf5,0xed,0x95,0xe3,0x12,0x5b
        };
        const Bit8 DecryptAndStoreSslClientCertKeyKekSource[] =
        {
            0x64,0xB8,0x30,0xDD,0x0F,0x3C,0xB7,0xFB,0x4C,0x16,0x01,0x97,0xEA,0x9D,0x12,0x10
        };
        const Bit8 SslClientCertKeySource[] =
        {
            0x4D,0x92,0x5A,0x69,0x42,0x23,0xBB,0x92,0x59,0x16,0x3E,0x51,0x8C,0x78,0x14,0x0F
        };
        const Bit8 DecryptAndStoreDrmDeviceCertKeyKekSource[] =
        {
            0xD8,0xB6,0x37,0x7D,0xE7,0x4E,0x8B,0x3E,0xF6,0x87,0x91,0x81,0x09,0xB4,0x75,0xB2
        };
        const Bit8 DrmDeviceCertKeySource[] =
        {
            0xF9,0x28,0xB9,0x77,0x77,0x49,0xD9,0x3F,0x77,0x00,0x9D,0x1E,0x64,0xA6,0x5E,0x9D
        };
        const Bit8 ReencryptDeviceUniqueDataKekSourceForDecryption[] =
        {
            0x36,0xF3,0x6E,0x50,0x16,0x91,0x27,0xE3,0x72,0xBA,0x8D,0x14,0x84,0x4F,0x72,0x60
        };
        const Bit8 ReencryptDeviceUniqueDataKeySourceForDecryption[] =
        {
            0xC6,0xA5,0xCA,0xC7,0x68,0xAF,0x13,0x96,0xEC,0xF5,0x30,0x0E,0xB5,0x30,0xF6,0xAF
        };

        const int DecryptDeviceUniqueDataKekOption         = 1 << 5;
        const int DecryptAndStoreGcKeyKekOption            = 2 << 5;
        const int LoadEsDeviceKeyKekOption                 = 3 << 5;
        const int ReencryptDeviceUniqueDataKekOption       = 4 << 5;
        const int DecryptAndStoreSslClientCertKeyKekOption = 5 << 5;
        const int DecryptAndStoreDrmDeviceCertKeyKekOption = 6 << 5;
        const int DeviceUniqueKekOption                    = 1 << 0;

        const size_t DeviceUniqueDataKeyGenerationSize = 4;

        Bit32 LoadWord(const void* p, int offset)
        {
            auto p8 = reinterpret_cast<const Bit8*>(p);
            Bit32 v;
            std::memcpy(&v, &p8[offset], sizeof(v));
            return v;
        }

        void StoreWord(void* p, int offset, Bit32 v)
        {
            auto p8 = reinterpret_cast<Bit8*>(p);
            std::memcpy(&p8[offset], &v, sizeof(v));
        }

        Result ExtractDeviceUniqueData(
            void*                pOutBuffer,
            size_t               outBufferSize,
            const void*          pData,
            size_t               dataSize,
            const Bit8*          pKeySourceForDevelop,
            const Bit8*          pKeySourceForEnd ) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(dataSize > DeviceUniqueDataKeyGenerationSize);
            const auto pKeySource = IsDevelopment()? pKeySourceForDevelop: pKeySourceForEnd;

            auto generation = static_cast<int>(LoadWord(pData, dataSize - DeviceUniqueDataKeyGenerationSize));
            auto kekOption = DecryptDeviceUniqueDataKekOption;
            if (generation > 0)
            {
                kekOption |= DeviceUniqueKekOption;
            }

            AccessKey accessKey;
            auto result = nn::spl::GenerateAesKek(
                &accessKey,
                DecryptDeviceUniqueDataKekSource, sizeof(DecryptDeviceUniqueDataKekSource),
                generation,
                kekOption );
            if( result.IsFailure() )
            {
                return result;
            }

            return g_InterfaceHolder.GetDeviceUniqueDataInterface()->DecryptDeviceUniqueData(
                sf::OutArray<Bit8>(reinterpret_cast<Bit8*>(pOutBuffer), outBufferSize),
                sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pData), dataSize - DeviceUniqueDataKeyGenerationSize),
                accessKey,
                *reinterpret_cast<const detail::KeySource*>(pKeySource) );
        }

        template <typename Function>
        Result  DecryptAndStoreDeviceUniqueKey(
            const void* pData,
            size_t      dataSize,
            const Bit8* pKekSource,
            size_t      kekSourceSize,
            int         kekOption,
            const Bit8* pKeySourceForDevelop,
            const Bit8* pKeySourceForEnd,
            Function    decryptAndStore ) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(dataSize > DeviceUniqueDataKeyGenerationSize);
            NN_SDK_ASSERT(kekSourceSize == 16);
            const auto pKeySource = IsDevelopment()? pKeySourceForDevelop: pKeySourceForEnd;

            auto generation = static_cast<int>(LoadWord(pData, dataSize - DeviceUniqueDataKeyGenerationSize));
            if (generation > 0)
            {
                kekOption |= DeviceUniqueKekOption;
            }

            AccessKey accessKey;
            auto result = nn::spl::GenerateAesKek(
                    &accessKey,
                    pKekSource, kekSourceSize,
                    generation,
                    kekOption );
            if( result.IsFailure() )
            {
                return result;
            }

            return decryptAndStore(
                sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pData), dataSize - DeviceUniqueDataKeyGenerationSize),
                accessKey,
                *reinterpret_cast<const detail::KeySource*>(pKeySource));
        }

        Result ReencryptDeviceUniqueData(
            void*       pOutBuffer,
            size_t      outBufferSize,
            const void* pData,
            size_t      dataSize,
            const Bit8* pKekSourceForEncryption,
            size_t      kekSourceSizeForEncryption,
            const Bit8* pKeySourceForEncryptionForDevelop,
            const Bit8* pKeySourceForEncryptionForEnd,
            int         kekOptionForEncryption,
            int         generationForEncryption,
            Bit32       option ) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(dataSize > DeviceUniqueDataKeyGenerationSize);
            NN_SDK_ASSERT(kekSourceSizeForEncryption == 16);
            const auto pKeySourceForEncryption = IsDevelopment()? pKeySourceForEncryptionForDevelop: pKeySourceForEncryptionForEnd;

            auto generationForDecryption = static_cast<int>(LoadWord(pData, dataSize - DeviceUniqueDataKeyGenerationSize));

            AccessKey accessKeyForEncryption;
            auto result = nn::spl::GenerateAesKek(
                &accessKeyForEncryption,
                pKekSourceForEncryption, kekSourceSizeForEncryption,
                generationForEncryption,
                kekOptionForEncryption | DeviceUniqueKekOption );
            if( result.IsFailure() )
            {
                return result;
            }

            AccessKey accessKeyForDecryption;
            result = nn::spl::GenerateAesKek(
                &accessKeyForDecryption,
                ReencryptDeviceUniqueDataKekSourceForDecryption, sizeof(ReencryptDeviceUniqueDataKekSourceForDecryption),
                generationForDecryption,
                ReencryptDeviceUniqueDataKekOption );
            if( result.IsFailure() )
            {
                return result;
            }

            result = g_InterfaceHolder.GetManuInterface()->ReencryptDeviceUniqueData(
                sf::OutArray<Bit8>(reinterpret_cast<Bit8*>(pOutBuffer), outBufferSize),
                sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pData), dataSize - DeviceUniqueDataKeyGenerationSize),
                accessKeyForDecryption,
                *reinterpret_cast<const detail::KeySource*>(ReencryptDeviceUniqueDataKeySourceForDecryption),
                accessKeyForEncryption,
                *reinterpret_cast<const detail::KeySource*>(pKeySourceForEncryption),
                option );
            if( result.IsFailure() )
            {
                return result;
            }

            StoreWord(pOutBuffer, dataSize - DeviceUniqueDataKeyGenerationSize, generationForEncryption);

            return ResultSuccess();
        }

        void GetAesKeySlotAvailableEvent(nn::os::SystemEvent* pEvent) NN_NOEXCEPT
        {
            nn::sf::NativeHandle handle;
            auto result = g_InterfaceHolder.GetCryptoInterface()->GetAesKeySlotAvailableEvent(&handle);
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );

            pEvent->AttachReadableHandle(
                handle.GetOsHandle(),
                handle.IsManaged(),
                nn::os::EventClearMode_ManualClear);
            handle.Detach();
        }

        template <typename Function>
        Result WaitAvailableKeySlotAndExecute(Function function) NN_NOEXCEPT
        {
            nn::os::SystemEvent event;
            auto isEventInitialized = false;

            while (true)
            {
                auto result = function();

                if (result <= ResultNoAvailableKeySlot())
                {
                    if (!isEventInitialized)
                    {
                        GetAesKeySlotAvailableEvent(&event);
                        isEventInitialized = true;
                    }
                    event.Wait();
                }
                else
                {
                    return result;
                }
            }
        }

        template <typename Function>
        void Initialize(Function function) NN_NOEXCEPT
        {
            std::lock_guard<StaticMutex> lock(g_Mutex);
            NN_SDK_ASSERT( g_InitializeCount >= 0 );

            if( g_InitializeCount == 0)
            {
                function();
            }

            ++g_InitializeCount;
        }
    }
    // anonymouse namespace

    void Initialize() NN_NOEXCEPT
    {
        Initialize(
            [&]() NN_NOEXCEPT
            {
                g_InterfaceHolder.AcquireGeneralInterface();
            });
    }

    void InitializeForCrypto() NN_NOEXCEPT
    {
        Initialize(
            [&]() NN_NOEXCEPT
            {
                g_InterfaceHolder.AcquireCryptoInterface();
            });
    }

    void InitializeForFs() NN_NOEXCEPT
    {
        Initialize(
            [&]() NN_NOEXCEPT
            {
                g_InterfaceHolder.AcquireFsInterface();
            });
    }

    void InitializeForSsl() NN_NOEXCEPT
    {
        Initialize(
            [&]() NN_NOEXCEPT
            {
                g_InterfaceHolder.AcquireSslInterface();
            });
    }

    void InitializeForEs() NN_NOEXCEPT
    {
        Initialize(
            [&]() NN_NOEXCEPT
            {
                g_InterfaceHolder.AcquireEsInterface();
            });
    }

    void InitializeForManu() NN_NOEXCEPT
    {
        Initialize(
            [&]() NN_NOEXCEPT
            {
                g_InterfaceHolder.AcquireManuInterface();
            });
    }

    void Finalize() NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);
        NN_SDK_ASSERT( g_InitializeCount > 0 );

        --g_InitializeCount;

        if( g_InitializeCount == 0)
        {
            // HIPC プロキシの解放
            g_InterfaceHolder.Release();
        }
    }

    Result GetConfig(
        Bit64*      pOut,
        ConfigItem  key ) NN_NOEXCEPT
    {
        return g_InterfaceHolder.GetGeneralInterface()->GetConfig(pOut, key);
    }

    SocType GetSocType() NN_NOEXCEPT
    {
        switch (GetHardwareType())
        {
        case HardwareType_Icosa:
            return SocType_Erista;
        case HardwareType_Iowa:
        case HardwareType_Hoag:
            return SocType_Mariko;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    Result ModularExponentiate(
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pBase,
        size_t      baseSize,
        const void* pExponent,
        size_t      exponentSize,
        const void* pModulus,
        size_t      modulusSize ) NN_NOEXCEPT
    {
        return g_InterfaceHolder.GetGeneralInterface()->ModularExponentiate(
            sf::OutArray<Bit8>(reinterpret_cast<Bit8*>(pResultBuffer), resultBufferSize),
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pBase), baseSize),
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pExponent), exponentSize),
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pModulus), modulusSize)
        );
    }

    Result AllocateAesKeySlot(
        int* pSlotIndex ) NN_NOEXCEPT
    {
        return WaitAvailableKeySlotAndExecute(
            [&]() NN_NOEXCEPT
            {
                return g_InterfaceHolder.GetCryptoInterface()->AllocateAesKeySlot(pSlotIndex);
            });
    }

    Result DeallocateAesKeySlot(
        int slotIndex ) NN_NOEXCEPT
    {
        return g_InterfaceHolder.GetCryptoInterface()->DeallocateAesKeySlot( slotIndex );
    }

    Result GenerateAesKek(
        AccessKey*          pAccessKey,
        const void*         pKeySource,
        size_t              keySourceSize,
        int                 generation,
        int                 option ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES( keySourceSize == KeySourceSize );

        return g_InterfaceHolder.GetCryptoInterface()->GenerateAesKek(
            sf::Out<AccessKey>(reinterpret_cast<AccessKey*>(pAccessKey)),
            *reinterpret_cast<const detail::KeySource*>(pKeySource),
            generation,
            option );
    }

    Result LoadAesKey(
        int                 slotIndex,
        const AccessKey&    accessKey,
        const void*         pKeySource,
        size_t              keySourceSize ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES( keySourceSize == KeySourceSize );

        return g_InterfaceHolder.GetCryptoInterface()->LoadAesKey(
            slotIndex,
            accessKey,
            *reinterpret_cast<const detail::KeySource*>(pKeySource) );
    }

    Result GenerateAesKey(
        void*               pOutBuffer,
        size_t              outBufferSize,
        const AccessKey&    accessKey,
        const void*         pKeySource,
        size_t              keySourceSize ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES( outBufferSize >= AesKeySize );
        NN_SDK_REQUIRES( keySourceSize == KeySourceSize );

        return WaitAvailableKeySlotAndExecute(
            [&]() NN_NOEXCEPT
            {
                return g_InterfaceHolder.GetCryptoInterface()->GenerateAesKey(
                    sf::Out<detail::AesKey>(reinterpret_cast<detail::AesKey*>(pOutBuffer)),
                    accessKey,
                    *reinterpret_cast<const detail::KeySource*>(pKeySource));
            });
    }

    Result GenerateSpecificAesKey(
        void*       pOutBuffer,
        size_t      outBufferSize,
        const void* pSource,
        size_t      sourceSize,
        int         generation,
        Bit32       purpose  ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES( outBufferSize >= AesKeySize );
        NN_SDK_REQUIRES( sourceSize == KeySourceSize );

        return g_InterfaceHolder.GetFsInterface()->GenerateSpecificAesKey(
            sf::Out<detail::AesKey>(reinterpret_cast<detail::AesKey*>(pOutBuffer)),
            *reinterpret_cast<const detail::KeySource*>(pSource),
            generation,
            purpose );
    }

    Result ComputeCtr(
        void*       pOutBuffer,
        size_t      outBufferSize,
        int         slotIndex,
        const void* pInBuffer,
        size_t      inBufferSize,
        const void* pInitialCounter,
        size_t      initialCounterSize ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES( initialCounterSize >= sizeof(detail::IvIc) );
        NN_SDK_REQUIRES( outBufferSize >= inBufferSize );

        return g_InterfaceHolder.GetCryptoInterface()->ComputeCtr(
            sf::OutBuffer(reinterpret_cast<char*>(pOutBuffer), outBufferSize),
            slotIndex,
            sf::InBuffer(reinterpret_cast<const char*>(pInBuffer), inBufferSize),
            *reinterpret_cast<const detail::IvIc*>(pInitialCounter) );
    }

    Result ComputeCmac(
        void*       pOutBuffer,
        size_t      outBufferSize,
        int         slotIndex,
        const void* pInBuffer,
        size_t      inBufferSize ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES( outBufferSize >= AesBlockSize );

        return g_InterfaceHolder.GetCryptoInterface()->ComputeCmac(
            sf::Out<detail::Cmac>(reinterpret_cast<detail::Cmac*>(pOutBuffer)),
            slotIndex,
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pInBuffer), inBufferSize) );
    }

    Result ExtractDrmDeviceCertEccKey(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize ) NN_NOEXCEPT
    {
        return ExtractDeviceUniqueData(
            pOutBuffer, outBufferSize,
            pData, dataSize,
            DrmDeviceCertEccKeySourceForDevelop,
            DrmDeviceCertEccKeySourceForEnd);
    }

    Result ExtractSslClientCertKey(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize ) NN_NOEXCEPT
    {
        const Bit8 keySourceForDevelop[16] =
        {
            0xd5,0xd2,0xfc,0x00,0xfd,0x49,0xdd,0xf8,0xee,0x7b,0xc4,0x4b,0xe1,0x4c,0xaa,0x99
        };
        const Bit8 keySourceForEnd[16] =
        {
            0x9a,0x38,0x3b,0xf4,0x31,0xd0,0xbd,0x81,0x32,0x53,0x4b,0xa9,0x64,0x39,0x7d,0xe3
        };

        return ExtractDeviceUniqueData(
            pOutBuffer, outBufferSize,
            pData, dataSize,
            keySourceForDevelop,
            keySourceForEnd);
    }

    Result ExtractNfpEccBn128Key(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize ) NN_NOEXCEPT
    {
        return ExtractDeviceUniqueData(
            pOutBuffer, outBufferSize,
            pData, dataSize,
            NfpEccBn128KeySource,
            NfpEccBn128KeySource);
    }

    Result ExtractNfpEccP160Key(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize ) NN_NOEXCEPT
    {
        return ExtractDeviceUniqueData(
            pOutBuffer, outBufferSize,
            pData, dataSize,
            NfpEccP160KeySource,
            NfpEccP160KeySource);
    }

    Result DecryptAesKey(
        void*       pOutBuffer,
        size_t      outBufferSize,
        const void* pSource,
        size_t      sourceSize,
        int         generation,
        Bit32       option) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES( outBufferSize >= AesKeySize );
        NN_SDK_REQUIRES( sourceSize == KeySourceSize );

        return WaitAvailableKeySlotAndExecute(
            [&]() NN_NOEXCEPT
            {
                return g_InterfaceHolder.GetCryptoInterface()->DecryptAesKey(
                    sf::Out<detail::AesKey>(reinterpret_cast<detail::AesKey*>(pOutBuffer)),
                    *reinterpret_cast<const detail::KeySource*>(pSource),
                    generation,
                    option);
            });
    }

    Result SetConfig(
        ConfigItem  key,
        Bit64       value ) NN_NOEXCEPT
    {
        return g_InterfaceHolder.GetGeneralInterface()->SetConfig(key, value);
    }

    bool IsDevelopment() NN_NOEXCEPT
    {
        bool isDevelopment;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            g_InterfaceHolder.GetGeneralInterface()->IsDevelopment(&isDevelopment) );
        return isDevelopment;
    }

    Result  GenerateRandomBytes(
        void*   pOutBuffer,
        size_t  bufferSize ) NN_NOEXCEPT
    {
        return g_InterfaceHolder.GetGeneralInterface()->GenerateRandomBytes(
            sf::OutArray<Bit8>(reinterpret_cast<Bit8*>(pOutBuffer), bufferSize)
        );
    }

    Result  DecryptAndStoreGcKey(
        const void* pData,
        size_t      dataSize ) NN_NOEXCEPT
    {
        return DecryptAndStoreDeviceUniqueKey(
            pData, dataSize,
            DecryptAndStoreGcKeyKekSource, sizeof(DecryptAndStoreGcKeyKekSource),
            DecryptAndStoreGcKeyKekOption,
            GcKeySourceForDevelop,
            GcKeySourceForEnd,
            [&](const sf::InArray<Bit8>& data, const AccessKey& accessKey, const detail::KeySource& keySource) NN_NOEXCEPT
            {
                return g_InterfaceHolder.GetFsInterface()->DecryptAndStoreGcKey(
                    data, accessKey, keySource);
            }
        );
    }

    Result  DecryptGcMessage(
        size_t*     pOutResultSize,
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pCipher,
        size_t      cipherSize,
        const void* pModulus,
        size_t      modulusSize,
        const void* pLabelDigest,
        size_t      labelDigestSize) NN_NOEXCEPT
    {
        int resultSize;
        auto result = g_InterfaceHolder.GetFsInterface()->DecryptGcMessage(
            sf::Out<int>(&resultSize),
            sf::OutArray<Bit8>(reinterpret_cast<Bit8*>(pResultBuffer), resultBufferSize),
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pCipher), cipherSize),
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pModulus), modulusSize),
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pLabelDigest), labelDigestSize) );

        *pOutResultSize = resultSize;
        return result;
    }

    Result DecryptAndStoreSslClientCertKey(
        const void* pData,
        size_t      dataSize ) NN_NOEXCEPT
    {
        return DecryptAndStoreDeviceUniqueKey(
            pData, dataSize,
            DecryptAndStoreSslClientCertKeyKekSource, sizeof(DecryptAndStoreSslClientCertKeyKekSource),
            DecryptAndStoreSslClientCertKeyKekOption,
            SslClientCertKeySource,
            SslClientCertKeySource,
            [&](const sf::InArray<Bit8>& data, const AccessKey& accessKey, const detail::KeySource& keySource) NN_NOEXCEPT
            {
                return g_InterfaceHolder.GetSslInterface()->DecryptAndStoreSslClientCertKey(
                    data, accessKey, keySource);
            }
        );
    }

    Result ModularExponentiateWithSslClientCertKey(
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pCipher,
        size_t      cipherSize,
        const void* pModulus,
        size_t      modulusSize) NN_NOEXCEPT
    {
        return g_InterfaceHolder.GetSslInterface()->ModularExponentiateWithSslClientCertKey(
            sf::OutArray<Bit8>(reinterpret_cast<Bit8*>(pResultBuffer), resultBufferSize),
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pCipher), cipherSize),
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pModulus), modulusSize));
    }

    Result DecryptAndStoreDrmDeviceCertKey(
        const void* pData,
        size_t      dataSize ) NN_NOEXCEPT
    {
        return DecryptAndStoreDeviceUniqueKey(
            pData, dataSize,
            DecryptAndStoreDrmDeviceCertKeyKekSource, sizeof(DecryptAndStoreDrmDeviceCertKeyKekSource),
            DecryptAndStoreDrmDeviceCertKeyKekOption,
            DrmDeviceCertKeySource,
            DrmDeviceCertKeySource,
            [&](const sf::InArray<Bit8>& data, const AccessKey& accessKey, const detail::KeySource& keySource) NN_NOEXCEPT
            {
                return g_InterfaceHolder.GetEsInterface()->DecryptAndStoreDrmDeviceCertKey(
                    data, accessKey, keySource);
            }
        );
    }

    Result ModularExponentiateWithDrmDeviceCertKey(
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pCipher,
        size_t      cipherSize,
        const void* pModulus,
        size_t      modulusSize) NN_NOEXCEPT
    {
        return g_InterfaceHolder.GetEsInterface()->ModularExponentiateWithDrmDeviceCertKey(
            sf::OutArray<Bit8>(reinterpret_cast<Bit8*>(pResultBuffer), resultBufferSize),
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pCipher), cipherSize),
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pModulus), modulusSize));
    }

    Result LoadEsDeviceKey(
        const void*      pData,
        size_t           dataSize ) NN_NOEXCEPT
    {
        return DecryptAndStoreDeviceUniqueKey(
            pData, dataSize,
            LoadEsDeviceKeyKekSource, sizeof(LoadEsDeviceKeyKekSource),
            LoadEsDeviceKeyKekOption,
            EsDeviceKeySourceForDevelop,
            EsDeviceKeySourceForEnd,
            [&](const sf::InArray<Bit8>& data, const AccessKey& accessKey, const detail::KeySource& keySource) NN_NOEXCEPT
            {
                return g_InterfaceHolder.GetEsInterface()->LoadEsDeviceKey(
                    data, accessKey, keySource);
            }
        );
    }

    Result PrepareEsTitleKey(
        AccessKey*       pAccessKey,
        const void*      pCipher,
        size_t           cipherSize,
        const void*      pModulus,
        size_t           modulusSize,
        const void*      pLabelDigest,
        size_t           labelDigestSize,
        int              generation) NN_NOEXCEPT
    {
        return g_InterfaceHolder.GetEsInterface()->PrepareEsTitleKey(
            sf::Out<AccessKey>(reinterpret_cast<AccessKey*>(pAccessKey)),
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pCipher), cipherSize),
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pModulus), modulusSize),
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pLabelDigest), labelDigestSize),
            generation
        );
    }

    Result PrepareEsArchiveKey(
        AccessKey*       pAccessKey,
        const void*      pCipher,
        size_t           cipherSize,
        const void*      pModulus,
        size_t           modulusSize,
        const void*      pLabelDigest,
        size_t           labelDigestSize,
        int              generation) NN_NOEXCEPT
    {
        return g_InterfaceHolder.GetEsInterface()->PrepareEsArchiveKey(
            sf::Out<AccessKey>(reinterpret_cast<AccessKey*>(pAccessKey)),
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pCipher), cipherSize),
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pModulus), modulusSize),
            sf::InArray<Bit8>(reinterpret_cast<const Bit8*>(pLabelDigest), labelDigestSize),
            generation
        );
    }

    Result PrepareCommonEsTitleKey(
        AccessKey*       pAccessKey,
        const void*      pKeySource,
        size_t           keySourceSize,
        int              generation) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES( keySourceSize == KeySourceSize );

        return g_InterfaceHolder.GetEsInterface()->PrepareCommonEsTitleKey(
            sf::Out<AccessKey>(reinterpret_cast<AccessKey*>(pAccessKey)),
            *reinterpret_cast<const detail::KeySource*>(pKeySource),
            generation
        );
    }

    Result LoadPreparedAesKey(
        int              slotIndex,
        const AccessKey& accessKey) NN_NOEXCEPT
    {
        if (g_InterfaceHolder.IsFsInterfaceAvailable())
        {
            return g_InterfaceHolder.GetFsInterface()->LoadPreparedAesKey(
                slotIndex,
                accessKey);
        }
        else
        {
            return g_InterfaceHolder.GetEsInterface()->LoadPreparedAesKey(
                slotIndex,
                accessKey);
        }
    }

    Result SetBootReason(BootReasonValue bootReason) NN_NOEXCEPT
    {
        return g_InterfaceHolder.GetGeneralInterface()->SetBootReason(bootReason);
    }

    Result GetBootReason(BootReasonValue* pOut) NN_NOEXCEPT
    {
        return g_InterfaceHolder.GetGeneralInterface()->GetBootReason(pOut);
    }

    Result GetPackage2Hash(
        void*       pOutBuffer,
        size_t      bufferSize) NN_NOEXCEPT
    {
        return g_InterfaceHolder.GetFsInterface()->GetPackage2Hash(
            sf::OutArray<Bit8>(reinterpret_cast<Bit8*>(pOutBuffer), bufferSize));
    }

    Result ReencryptNfpEccBn128Key(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize,
        int                  generationForEncryption ) NN_NOEXCEPT
    {
        return ReencryptDeviceUniqueData(
            pOutBuffer, outBufferSize,
            pData, dataSize,
            DecryptDeviceUniqueDataKekSource, sizeof(DecryptDeviceUniqueDataKekSource),
            NfpEccBn128KeySource,
            NfpEccBn128KeySource,
            DecryptDeviceUniqueDataKekOption,
            generationForEncryption,
            0 );
    }

    Result ReencryptNfpEccP160Key(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize,
        int                  generationForEncryption ) NN_NOEXCEPT
    {
        return ReencryptDeviceUniqueData(
            pOutBuffer, outBufferSize,
            pData, dataSize,
            DecryptDeviceUniqueDataKekSource, sizeof(DecryptDeviceUniqueDataKekSource),
            NfpEccP160KeySource,
            NfpEccP160KeySource,
            DecryptDeviceUniqueDataKekOption,
            generationForEncryption,
            0 );
    }

    Result ReencryptDrmDeviceCertEccKey(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize,
        int                  generationForEncryption ) NN_NOEXCEPT
    {
        return ReencryptDeviceUniqueData(
            pOutBuffer, outBufferSize,
            pData, dataSize,
            DecryptDeviceUniqueDataKekSource, sizeof(DecryptDeviceUniqueDataKekSource),
            DrmDeviceCertEccKeySourceForDevelop,
            DrmDeviceCertEccKeySourceForEnd,
            DecryptDeviceUniqueDataKekOption,
            generationForEncryption,
            0 );
    }

    Result ReencryptGcKey(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize,
        int                  generationForEncryption ) NN_NOEXCEPT
    {
        return ReencryptDeviceUniqueData(
            pOutBuffer, outBufferSize,
            pData, dataSize,
            DecryptAndStoreGcKeyKekSource, sizeof(DecryptAndStoreGcKeyKekSource),
            GcKeySourceForDevelop,
            GcKeySourceForEnd,
            DecryptAndStoreGcKeyKekOption,
            generationForEncryption,
            1 );
    }

    Result ReencryptEsDeviceKey(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize,
        int                  generationForEncryption ) NN_NOEXCEPT
    {
        return ReencryptDeviceUniqueData(
            pOutBuffer, outBufferSize,
            pData, dataSize,
            LoadEsDeviceKeyKekSource, sizeof(LoadEsDeviceKeyKekSource),
            EsDeviceKeySourceForDevelop,
            EsDeviceKeySourceForEnd,
            LoadEsDeviceKeyKekOption,
            generationForEncryption,
            2 );
    }

    Result ReencryptSslClientCertKey(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize,
        int                  generationForEncryption ) NN_NOEXCEPT
    {
        return ReencryptDeviceUniqueData(
            pOutBuffer, outBufferSize,
            pData, dataSize,
            DecryptAndStoreSslClientCertKeyKekSource, sizeof(DecryptAndStoreSslClientCertKeyKekSource),
            SslClientCertKeySource,
            SslClientCertKeySource,
            DecryptAndStoreSslClientCertKeyKekOption,
            generationForEncryption,
            3 );
    }

    Result ReencryptDrmDeviceCertKey(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize,
        int                  generationForEncryption ) NN_NOEXCEPT
    {
        return ReencryptDeviceUniqueData(
            pOutBuffer, outBufferSize,
            pData, dataSize,
            DecryptAndStoreDrmDeviceCertKeyKekSource, sizeof(DecryptAndStoreDrmDeviceCertKeyKekSource),
            DrmDeviceCertKeySource,
            DrmDeviceCertKeySource,
            DecryptAndStoreDrmDeviceCertKeyKekOption,
            generationForEncryption,
            4 );
    }

}}  // namespace nn::spl
