﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/crypto/crypto_RsaOaepSha256Decoder.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/os.h>
#include <nn/dd.h>
#include <nn/crypto.h>
#include <nn/util/util_BitPack.h>

#include <nn/spl/spl_Result.h>
#include <nn/spl/spl_PrivateResult.h>
#include <nn/spl/impl/spl_ApiImpl.h>
#include <nn/spl/smc/spl_SecureMonitorApi.h>

#include "spl_DeviceAddressMapper.h"
#include "spl_CtrDrbg.h"

#include <algorithm>
#include <mutex>


namespace nn { namespace spl { namespace impl {

    namespace
    {
        typedef CtrDrbg<crypto::AesEncryptor128, 16, false> Drbg;

        enum DeviceUniqueDataDecryptionMode
        {
            DeviceUniqueDataDecryptionMode_GetDeviceUniqueData,
            DeviceUniqueDataDecryptionMode_LoadGcKey,
            DeviceUniqueDataDecryptionMode_LoadEsDeviceKey,
            DeviceUniqueDataDecryptionMode_LoadSslClientCertKey,
            DeviceUniqueDataDecryptionMode_LoadDrmDeviceCert,
        };

        enum ModularExponentiateWithStorageKeyMode
        {
            ModularExponentiateWithStorageKeyMode_Gc,
            ModularExponentiateWithStorageKeyMode_SslClientCert,
            ModularExponentiateWithStorageKeyMode_DrmDeviceCert,
        };

        struct ModularExponentiateParams
        {
            Bit8    base[256];
            Bit8    exponent[256];
            Bit8    modulus[256];
        };

        struct DecryptAndStoreKeyParams
        {
            Bit8    data[576];
        };

        struct ModularExponentiateWithStorageKeyParams
        {
            Bit8    cipher[256];
            Bit8    modulus[256];
        };

        struct DecryptDeviceUniqueDataParams
        {
            Bit8    data[304];
        };

        struct ReencryptDeviceUniqueDataParams
        {
            Bit8           data[576];
            smc::AccessKey accessKeyForDecryption;
            Bit64          keySourceForDecryption[2];
            smc::AccessKey accessKeyForEncryption;
            Bit64          keySourceForEncryption[2];
        };

        struct LoadEsDeviceKeyParams
        {
            Bit8    data[576];
        };

        struct PrepareEsDeviceUniqueKeyParams
        {
            Bit8    cipher[256];
            Bit8    modulus[256];
        };

        struct LinkedListEntry
        {
            Bit32       zero;
            uint32_t    address;
            uint32_t    size;
        };

        struct InOutEntry
        {
            LinkedListEntry in;
            LinkedListEntry out;
        };

        struct DecryptAesParams
        {
            InOutEntry  inout;
            NN_PADDING8;
            Bit8    inBuffer[16];
            Bit8    outBuffer[16];
        };

        class ScopedAesKeySlot
        {
        public:
            ScopedAesKeySlot()
                : m_IsAllocated(false)
            {
            }

            ~ScopedAesKeySlot()
            {
                if( m_IsAllocated )
                {
                    DeallocateAesKeySlot(m_SlotIndex);
                }
            }

            Result Allocate()
            {
                auto result = AllocateAesKeySlot(&m_SlotIndex);
                if( result.IsFailure() )
                {
                    return result;
                }

                m_IsAllocated = true;
                return ResultSuccess();
            }

            int GetIndex()
            {
                return m_SlotIndex;
            }

        private:
            int m_SlotIndex;
            bool m_IsAllocated;
        };

        const size_t DeviceAddressSpaceBlockSize    = (1 << 22);
        const Bit64  DeviceAddressSpaceBlockMask    = DeviceAddressSpaceBlockSize - 1;
        const uint32_t WorkBufferBase       = 0x80000000;
        const size_t   ComputeAesSizeMax    = 0x30000000;
        const uint32_t ComputeAesInMapBase  = 0x90000000;
        const uint32_t ComputeAesOutMapBase = 0xC0000000;

        const size_t DeviceUniqueDataIvSize            = 16;
        const size_t DeviceUniqueDataPaddingSize       = 8;
        const size_t DeviceUniqueDataDeviceIdSize      = 8;
        const size_t DeviceUniqueDataGmacSize          = 16;
        const size_t DeviceUniqueDataPlainMetaDataSize =
            DeviceUniqueDataIvSize + DeviceUniqueDataGmacSize;
        const size_t DeviceUniqueDataMetaDataSize      =
            DeviceUniqueDataPlainMetaDataSize + DeviceUniqueDataPaddingSize + DeviceUniqueDataDeviceIdSize;

        const Bit8 KeyGenerationSource[16] =
        {
            0x89,0x61,0x5e,0xe0,0x5c,0x31,0xb6,0x80,0x5f,0xe5,0x8f,0x3d,0xa2,0x4f,0x7a,0xa8,
        };

        const Bit8 AesKeyDecriptionSource[16] =
        {
            0x11,0x70,0x24,0x2b,0x48,0x69,0x11,0xf1,0x11,0xb0,0x0c,0x47,0x7c,0xc3,0xef,0x7e,
        };


        NN_ALIGNAS(4096) Bit8       g_WorkBuffer[2048];
        os::InterruptName           g_InterruptName;
        os::InterruptEventType      g_Interrupt;
        os::Mutex                   g_OperationLock(false);
        dd::DeviceAddressSpaceType  g_DeviceAddressSpace;
        uint32_t                    g_WorkBufferMappedAddress;
        Drbg                        g_Drbg;
        bool                        g_IsAesKeySlotAllocated[AesKeySlotCount];
        os::SystemEvent             g_AesKeySlotAvailableEvent(os::EventClearMode_ManualClear, true);
        BootReasonValue             g_BootReason;
        bool                        g_IsBootReasonInitialized;


        void InitializeAsyncOperation()
        {
            Bit64 interruptNumber;

            GetConfig(&interruptNumber, ConfigItem_InterruptNumber);
            g_InterruptName = static_cast<os::InterruptName>(interruptNumber);

            os::InitializeInterruptEvent(&g_Interrupt, g_InterruptName, os::EventClearMode_AutoClear);
        }

        void InitializeDeviceAddressSpace()
        {
            // DAS を作成して SE にアタッチする
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                dd::CreateDeviceAddressSpace(&g_DeviceAddressSpace, 0, (1ull << 32)) );
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                dd::AttachDeviceAddressSpace(&g_DeviceAddressSpace, dd::DeviceName_Se) );
#endif

            // g_WorkBuffer は常時 SE にマップしておく
            const uintptr_t workBufferAddress = reinterpret_cast<uintptr_t>(g_WorkBuffer);
            g_WorkBufferMappedAddress = WorkBufferBase + (workBufferAddress & DeviceAddressSpaceBlockMask);

            NN_ABORT_UNLESS_RESULT_SUCCESS(
                dd::MapDeviceAddressSpaceAligned(
                    &g_DeviceAddressSpace,
                    dd::GetCurrentProcessHandle(),
                    workBufferAddress,
                    dd::DeviceAddressSpaceMemoryRegionAlignment,
                    g_WorkBufferMappedAddress,
                    dd::MemoryPermission_ReadWrite ));
        }

        void InitializeDrbg()
        {
            Bit8 seed[Drbg::SeedSize];

            auto result = smc::GenerateRandomBytes(seed, sizeof(seed));
            NN_ABORT_UNLESS( result == smc::SmcResult_Success );

            g_Drbg.Initialize(seed, sizeof(seed), NULL, 0, NULL, 0);
        }

        void InitializeKeySlots()
        {
            for (int i = 0; i < AesKeySlotCount; ++i)
            {
                g_IsAesKeySlotAllocated[i] = false;
            }
            g_AesKeySlotAvailableEvent.Signal();
        }

        void WaitOperation()
        {
            os::WaitInterruptEvent(&g_Interrupt);
        }

        Result MakeSplResult(int sr)
        {
            if( sr == smc::SmcResult_Success )
            {
                return ResultSuccess();
            }
            if( (ResultRemoteError::DescriptionBegin <= sr)
             && (sr < ResultRemoteError::DescriptionEnd) )
            {
                return
                    result::detail::ConstructResult(
                        result::detail::ResultTraits::MakeInnerValue(
                            ResultRemoteError::ErrorRange::Module,
                            sr ));
            }

            return ResultUnexpectedRemoteResult();
        }

        smc::SmcResult WaitAndGetResult(smc::OperationKey operationKey)
        {
            WaitOperation();

            smc::SmcResult asyncResult;
            smc::SmcResult result = smc::GetResult(&asyncResult, operationKey);

            if( result != smc::SmcResult_Success )
            {
                return result;
            }

            return asyncResult;
        }

        smc::SmcResult WaitAndGetResultData(void* pBuffer, size_t size, smc::OperationKey operationKey)
        {
            WaitOperation();

            smc::SmcResult asyncResult;
            smc::SmcResult result = smc::GetResultData(&asyncResult, pBuffer, size, operationKey);

            if( result != smc::SmcResult_Success )
            {
                return result;
            }

            return asyncResult;
        }

        Bit32 MakeComputeAesMode(smc::CipherMode cipherMode, int slotIndex)
        {
            return ((slotIndex & 0x7) | (cipherMode << 4));
        }

        smc::SmcResult DecryptAes(void* pOutput, int slotIndex, const void* pInput)
        {
            // SE 用の入出力エントリ作成
            auto& param = *reinterpret_cast<DecryptAesParams*>(g_WorkBuffer);

            param.inout.in.zero       = 0;
            param.inout.in.address    = g_WorkBufferMappedAddress + offsetof(DecryptAesParams, inBuffer);
            param.inout.in.size       = sizeof(param.inBuffer);
            param.inout.out.zero      = 0;
            param.inout.out.address   = g_WorkBufferMappedAddress + offsetof(DecryptAesParams, outBuffer);
            param.inout.out.size      = sizeof(param.outBuffer);

            std::memcpy(param.inBuffer, pInput, sizeof(param.inBuffer));

            // SE が読み書きするのでキャッシュフラッシュ
            os::FlushDataCache(&param, sizeof(param));

            {
                std::lock_guard<os::Mutex> lock(g_OperationLock);
                smc::OperationKey operationKey;

                Bit32 mode = MakeComputeAesMode(smc::CipherMode_CbcDecryption, slotIndex);
                Bit64 ivic[2];

                std::memset(ivic, 0, sizeof(ivic));

                smc::SmcResult sr;
                sr = smc::ComputeAes(
                    &operationKey,
                    g_WorkBufferMappedAddress + offsetof(DecryptAesParams, inout.out),
                    mode,
                    ivic,
                    g_WorkBufferMappedAddress + offsetof(DecryptAesParams, inout.in),
                    sizeof(param.outBuffer) );
                if( sr != smc::SmcResult_Success )
                {
                    return sr;
                }

                sr = WaitAndGetResult(operationKey);
                if( sr != smc::SmcResult_Success )
                {
                    return sr;
                }
            }

            // SE が書いたのでキャッシュフラッシュ
            os::FlushDataCache(&param.outBuffer, sizeof(param.outBuffer));

            std::memcpy(pOutput, param.outBuffer, sizeof(param.outBuffer));

            return smc::SmcResult_Success;
        }

        Result GenerateRandomBytesImpl(
            void*   pOutBuffer,
            size_t  bufferSize ) NN_NOEXCEPT
        {
            NN_SDK_ASSERT( bufferSize <= Drbg::RequestSizeMax );
            bool isSuccess = g_Drbg.Generate(pOutBuffer, bufferSize, NULL, 0);

            if( ! isSuccess )
            {
                Bit8 seed[Drbg::SeedSize];

                auto result = smc::GenerateRandomBytes(seed, sizeof(seed));
                if( result != smc::SmcResult_Success )
                {
                    return MakeSplResult(result);
                }

                g_Drbg.Reseed(seed, sizeof(seed), NULL, 0);
                g_Drbg.Generate(pOutBuffer, bufferSize, NULL, 0);
            }

            return ResultSuccess();
        }

        Result DecryptAndStoreDeviceUniqueKey(
            const void*                    pData,
            size_t                         dataSize,
            const AccessKey&               accessKey,
            const void*                    pKeySource,
            size_t                         keySourceSize,
            DeviceUniqueDataDecryptionMode mode) NN_NOEXCEPT
        {
            auto& params = *reinterpret_cast<DecryptAndStoreKeyParams*>(g_WorkBuffer);
            smc::AccessKey key;
            Bit64 source[2];

            if( ! (dataSize <= sizeof(params.data)) )
            {
                return ResultInvalidBufferSize();
            }
            if( keySourceSize != sizeof(source) )
            {
                return ResultInvalidBufferSize();
            }

            std::memcpy(&key, &accessKey, sizeof(key));
            std::memcpy(source, pKeySource, sizeof(source));
            std::memcpy(params.data, pData, dataSize);

            auto sr = smc::DecryptDeviceUniqueData(params.data, dataSize, key, source, mode);

            return MakeSplResult(sr);
        }

        Result ModularExponentiateWithStorageKey(
            void*       pResultBuffer,
            size_t      resultBufferSize,
            const void* pCipher,
            size_t      cipherSize,
            const void* pModulus,
            size_t      modulusSize,
            Bit32       option) NN_NOEXCEPT
        {
            auto& params = *reinterpret_cast<ModularExponentiateWithStorageKeyParams*>(g_WorkBuffer);

            if( ! (cipherSize <= sizeof(params.cipher)) )
            {
                return ResultInvalidBufferSize();
            }
            if( ! (modulusSize <= sizeof(params.modulus)) )
            {
                return ResultInvalidBufferSize();
            }
            if( ! (resultBufferSize <= sizeof(g_WorkBuffer)) )
            {
                return ResultInvalidBufferSize();
            }

            size_t cipherSpaceSize      = sizeof(params.cipher)     - cipherSize;
            size_t modulusSpaceSize     = sizeof(params.modulus)    - modulusSize;

            std::memset(params.cipher,  0, sizeof(params.cipher));
            std::memset(params.modulus, 0, sizeof(params.modulus));

            std::memcpy(params.cipher   + cipherSpaceSize,      pCipher,    cipherSize);
            std::memcpy(params.modulus  + modulusSpaceSize,     pModulus,   modulusSize);

            {
                std::lock_guard<os::Mutex> lock(g_OperationLock);
                smc::OperationKey operationKey;

                smc::SmcResult sr;
                sr = smc::ModularExponentiateWithStorageKey(
                    &operationKey,
                    params.cipher,
                    params.modulus,
                    option );
                if( sr != smc::SmcResult_Success )
                {
                    return MakeSplResult(sr);
                }

                sr = WaitAndGetResultData(g_WorkBuffer, resultBufferSize, operationKey);
                if( sr != smc::SmcResult_Success )
                {
                    return MakeSplResult(sr);
                }
            }

            if (pResultBuffer != g_WorkBuffer)
            {
                std::memcpy(pResultBuffer, g_WorkBuffer, resultBufferSize);
            }

            return ResultSuccess();
        }

        Bit32 MakePrepareEsDeviceUniqueKeyOption(int generation, smc::EsDeviceUniqueKeyType type)
        {
            return ((generation & 0x3F) | (type << 6));
        }

        Result PrepareEsDeviceUniqueKey(
            AccessKey*                 pAccessKey,
            const void*                pCipher,
            size_t                     cipherSize,
            const void*                pModulus,
            size_t                     modulusSize,
            const void*                pLabelDigest,
            size_t                     labelDigestSize,
            smc::EsDeviceUniqueKeyType type,
            int                        generation) NN_NOEXCEPT
        {
            auto& params = *reinterpret_cast<PrepareEsDeviceUniqueKeyParams*>(g_WorkBuffer);

            if( ! (cipherSize <= sizeof(params.cipher)) )
            {
                return ResultInvalidBufferSize();
            }
            if( ! (modulusSize <= sizeof(params.modulus)) )
            {
                return ResultInvalidBufferSize();
            }
            if( ! (labelDigestSize <= 32) )
            {
                return ResultInvalidBufferSize();
            }

            size_t cipherSpaceSize      = sizeof(params.cipher)     - cipherSize;
            size_t modulusSpaceSize     = sizeof(params.modulus)    - modulusSize;

            std::memset(params.cipher,  0, sizeof(params.cipher));
            std::memset(params.modulus, 0, sizeof(params.modulus));

            std::memcpy(params.cipher   + cipherSpaceSize,      pCipher,    cipherSize);
            std::memcpy(params.modulus  + modulusSpaceSize,     pModulus,   modulusSize);

            {
                std::lock_guard<os::Mutex> lock(g_OperationLock);
                smc::OperationKey operationKey;

                smc::SmcResult sr;
                auto option = MakePrepareEsDeviceUniqueKeyOption(generation, type);
                sr = smc::PrepareEsDeviceUniqueKey(
                    &operationKey,
                    params.cipher,
                    params.modulus,
                    pLabelDigest,
                    labelDigestSize,
                    option );
                if( sr != smc::SmcResult_Success )
                {
                    return MakeSplResult(sr);
                }

                sr = WaitAndGetResultData(g_WorkBuffer, sizeof(AccessKey), operationKey);
                if( sr != smc::SmcResult_Success )
                {
                    return MakeSplResult(sr);
                }
            }

            std::memcpy(pAccessKey, g_WorkBuffer, sizeof(AccessKey));
            return ResultSuccess();
        }
    }
    // anonymouse namespace

    void Initialize() NN_NOEXCEPT
    {
        InitializeAsyncOperation();
        InitializeDeviceAddressSpace();
        InitializeDrbg();
        InitializeKeySlots();
    }

    Result ModularExponentiate(
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pBase,
        size_t      baseSize,
        const void* pExponent,
        size_t      exponentSize,
        const void* pModulus,
        size_t      modulusSize ) NN_NOEXCEPT
    {
        auto& params = *reinterpret_cast<ModularExponentiateParams*>(g_WorkBuffer);

        if( ! (baseSize <= sizeof(params.base)) )
        {
            return ResultInvalidBufferSize();
        }
        if( ! (exponentSize <= sizeof(params.exponent)) )
        {
            return ResultInvalidBufferSize();
        }
        if( ! (modulusSize <= sizeof(params.modulus)) )
        {
            return ResultInvalidBufferSize();
        }
        if( ! (resultBufferSize <= sizeof(g_WorkBuffer)) )
        {
            return ResultInvalidBufferSize();
        }

        size_t baseSpaceSize        = sizeof(params.base)       - baseSize;
        size_t modulusSpaceSize     = sizeof(params.modulus)    - modulusSize;

        std::memset(params.base,    0, sizeof(params.base));
        std::memset(params.modulus, 0, sizeof(params.modulus));

        std::memcpy(params.base     + baseSpaceSize,        pBase,      baseSize);
        std::memcpy(params.modulus  + modulusSpaceSize,     pModulus,   modulusSize);
        std::memcpy(params.exponent, pExponent,  exponentSize);

        {
            std::lock_guard<os::Mutex> lock(g_OperationLock);
            smc::OperationKey operationKey;

            smc::SmcResult sr;
            sr = smc::ModularExponentiate(
                &operationKey,
                params.base,
                params.exponent,
                exponentSize,
                params.modulus );
            if( sr != smc::SmcResult_Success )
            {
                return MakeSplResult(sr);
            }

            sr = WaitAndGetResultData(g_WorkBuffer, resultBufferSize, operationKey);
            if( sr != smc::SmcResult_Success )
            {
                return MakeSplResult(sr);
            }
        }

        std::memcpy(pResultBuffer, g_WorkBuffer, resultBufferSize);

        return ResultSuccess();
    }

    Result GenerateAesKek(
        AccessKey*  pAccessKey,
        const void* pSource,
        size_t      sourceSize,
        int         generation,
        int         option ) NN_NOEXCEPT
    {
        smc::AccessKey accessKey;
        Bit64 source[2];

        if( sourceSize != sizeof(source) )
        {
            return ResultInvalidBufferSize();
        }

        std::memcpy(source, pSource, sizeof(source));

        auto sr = smc::GenerateAesKek(&accessKey, source, generation, option);

        std::memcpy(pAccessKey, &accessKey, sizeof(accessKey));
        return MakeSplResult(sr);
    }

    Result LoadAesKey(
        int                 slotIndex,
        const AccessKey&    accessKey,
        const void*         pSource,
        size_t              sourceSize ) NN_NOEXCEPT
    {
        smc::AccessKey key;
        Bit64 source[2];

        if( sourceSize != sizeof(source) )
        {
            return ResultInvalidBufferSize();
        }

        std::memcpy(&key, &accessKey, sizeof(key));
        std::memcpy(source, pSource, sizeof(source));

        return
            MakeSplResult(
                smc::LoadAesKey(slotIndex, key, source) );
    }

    Result GenerateAesKey(
        void*               pOutBuffer,
        size_t              outBufferSize,
        const AccessKey&    accessKey,
        const void*         pKeySource,
        size_t              keySourceSize ) NN_NOEXCEPT
    {
        smc::AccessKey key;
        Bit64 source[2];

        if( keySourceSize != sizeof(source) )
        {
            return ResultInvalidBufferSize();
        }
        if( outBufferSize != sizeof(source) )
        {
            return ResultInvalidBufferSize();
        }

        ScopedAesKeySlot slot;
        auto result = slot.Allocate();
        if( result.IsFailure() )
        {
            return result;
        }
        const int slotIndex = slot.GetIndex();

        std::memcpy(&key, &accessKey, sizeof(key));
        std::memcpy(source, KeyGenerationSource, sizeof(source));

        auto sr = smc::LoadAesKey(slotIndex, key, source);
        if( sr == smc::SmcResult_Success )
        {
            std::memcpy(source, pKeySource, sizeof(source));

            sr = DecryptAes(source, slotIndex, source);

            std::memcpy(pOutBuffer, source, sizeof(source));
        }

        return MakeSplResult(sr);
    }

    Result DecryptDeviceUniqueData(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize,
        const AccessKey&     accessKey,
        const void*          pKeySource,
        size_t               keySourceSize ) NN_NOEXCEPT
    {
        auto& params = *reinterpret_cast<DecryptDeviceUniqueDataParams*>(g_WorkBuffer);
        smc::AccessKey key;
        Bit64 source[2];

        if( ! (DeviceUniqueDataMetaDataSize < dataSize && dataSize <= sizeof(params.data)) )
        {
            return ResultInvalidBufferSize();
        }
        if( keySourceSize != sizeof(source) )
        {
            return ResultInvalidBufferSize();
        }

        std::memcpy(&key, &accessKey, sizeof(key));
        std::memcpy(source, pKeySource, sizeof(source));
        std::memcpy(params.data, pData, dataSize);

        auto sr = smc::DecryptDeviceUniqueData(params.data, dataSize, key, source, DeviceUniqueDataDecryptionMode_GetDeviceUniqueData);
        if( sr != smc::SmcResult_Success )
        {
            return MakeSplResult(sr);
        }

        std::memcpy(pOutBuffer, params.data, std::min(outBufferSize, dataSize - DeviceUniqueDataMetaDataSize));

        return MakeSplResult(sr);
    }

    Result ReencryptDeviceUniqueData(
        void*                pOutBuffer,
        size_t               outBufferSize,
        const void*          pData,
        size_t               dataSize,
        const AccessKey&     accessKeyForDecryption,
        const void*          pKeySourceForDecryption,
        size_t               keySourceSizeForDecryption,
        const AccessKey&     accessKeyForEncryption,
        const void*          pKeySourceForEncryption,
        size_t               keySourceSizeForEncryption,
        Bit32                option ) NN_NOEXCEPT
    {
        auto& params = *reinterpret_cast<ReencryptDeviceUniqueDataParams*>(g_WorkBuffer);

        if( ! (DeviceUniqueDataMetaDataSize < dataSize && dataSize <= sizeof(params.data)) )
        {
            return ResultInvalidBufferSize();
        }
        if( keySourceSizeForDecryption != sizeof(params.keySourceForDecryption) )
        {
            return ResultInvalidBufferSize();
        }
        if( keySourceSizeForEncryption != sizeof(params.keySourceForEncryption) )
        {
            return ResultInvalidBufferSize();
        }

        std::memcpy(params.data, pData, dataSize);
        std::memcpy(&params.accessKeyForDecryption, &accessKeyForDecryption, sizeof(params.accessKeyForDecryption));
        std::memcpy(&params.accessKeyForEncryption, &accessKeyForEncryption, sizeof(params.accessKeyForEncryption));
        std::memcpy(params.keySourceForDecryption, pKeySourceForDecryption, sizeof(params.keySourceForDecryption));
        std::memcpy(params.keySourceForEncryption, pKeySourceForEncryption, sizeof(params.keySourceForEncryption));

        auto sr = smc::ReencryptDeviceUniqueData(
            params.data, dataSize,
            params.accessKeyForDecryption,
            params.keySourceForDecryption,
            params.accessKeyForEncryption,
            params.keySourceForEncryption,
            option);
        if( sr != smc::SmcResult_Success )
        {
            return MakeSplResult(sr);
        }

        std::memcpy(pOutBuffer, params.data, std::min(outBufferSize, dataSize));

        return MakeSplResult(sr);
    }

    Result GetConfig(
        Bit64*      pOut,
        ConfigItem  key ) NN_NOEXCEPT
    {
        if (key == ConfigItem_Package2Hash)
        {
            return ResultInvalidArgument();
        }

        auto sr = smc::GetConfig(pOut, 1, key);

        if (key == ConfigItem_HardwareType && sr == smc::SmcResult_InvalidArgument)
        {
            *pOut = HardwareType_Icosa;
            sr = smc::SmcResult_Success;
        }
        if (key == ConfigItem_HardwareState && sr == smc::SmcResult_InvalidArgument)
        {
            *pOut = HardwareState_Development;
            sr = smc::SmcResult_Success;
        }

        return MakeSplResult(sr);
    }

    Result SetConfig(
        ConfigItem  key,
        Bit64       value ) NN_NOEXCEPT
    {
        smc::OperationKey operationKey;

        return
            MakeSplResult(
                smc::SetConfig(&operationKey, key, &value, 1, NULL) );
    }

    Result GetPackage2Hash(
        void*       pOutBuffer,
        size_t      bufferSize) NN_NOEXCEPT
    {
        Bit64 hash[4];

        if( ! (bufferSize >= sizeof(hash)) )
        {
            return ResultInvalidBufferSize();
        }

        auto sr = smc::GetConfig(hash, 4, ConfigItem_Package2Hash);
        if( sr != smc::SmcResult_Success )
        {
            return MakeSplResult(sr);
        }

        std::memcpy(pOutBuffer, hash, sizeof(hash));
        return ResultSuccess();
    }

    Result  GenerateRandomBytes(
        void*   pOutBuffer,
        size_t  bufferSize ) NN_NOEXCEPT
    {
        for( size_t offset = 0; offset < bufferSize; offset += Drbg::RequestSizeMax )
        {
            size_t oneSize = std::min(bufferSize - offset, Drbg::RequestSizeMax);

            Result result = GenerateRandomBytesImpl(static_cast<Bit8*>(pOutBuffer) + offset, oneSize);
            if( result.IsFailure() )
            {
                return result;
            }
        }

        return ResultSuccess();
    }

    Result  DecryptAndStoreGcKey(
        const void*         pData,
        size_t              dataSize,
        const AccessKey&    accessKey,
        const void*         pKeySource,
        size_t              keySourceSize ) NN_NOEXCEPT
    {
        return DecryptAndStoreDeviceUniqueKey(
            pData, dataSize,
            accessKey,
            pKeySource, keySourceSize,
            DeviceUniqueDataDecryptionMode_LoadGcKey);
    }

    Result  DecryptGcMessage(
        int*        pOutResultSize,
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pCipher,
        size_t      cipherSize,
        const void* pModulus,
        size_t      modulusSize,
        const void* pLabelDigest,
        size_t      labelDigestSize) NN_NOEXCEPT
    {
        if( ! (resultBufferSize <= sizeof(g_WorkBuffer)) )
        {
            return ResultInvalidBufferSize();
        }
        // OAEP パディングによる制限
        if( ! (labelDigestSize == 32) )
        {
            return ResultInvalidBufferSize();
        }

        ModularExponentiateWithStorageKey(
            g_WorkBuffer, 256,
            pCipher, cipherSize,
            pModulus, modulusSize,
            ModularExponentiateWithStorageKeyMode_Gc);

        auto resultSize = crypto::DecodeRsa2048OaepSha256(
            pResultBuffer, resultBufferSize,
            pLabelDigest, labelDigestSize,
            g_WorkBuffer, 256 );

        *pOutResultSize = static_cast<int>(resultSize);
        if (resultSize <= 0)
        {
            return ResultDecryptionFailed();
        }

        return ResultSuccess();
    }

    Result  DecryptAndStoreSslClientCertKey(
        const void*         pData,
        size_t              dataSize,
        const AccessKey&    accessKey,
        const void*         pKeySource,
        size_t              keySourceSize ) NN_NOEXCEPT
    {
        return DecryptAndStoreDeviceUniqueKey(
            pData, dataSize,
            accessKey,
            pKeySource, keySourceSize,
            DeviceUniqueDataDecryptionMode_LoadSslClientCertKey);
    }

    Result ModularExponentiateWithSslClientCertKey(
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pCipher,
        size_t      cipherSize,
        const void* pModulus,
        size_t      modulusSize) NN_NOEXCEPT
    {
        return ModularExponentiateWithStorageKey(
            pResultBuffer, resultBufferSize,
            pCipher, cipherSize,
            pModulus, modulusSize,
            ModularExponentiateWithStorageKeyMode_SslClientCert);
    }

    Result  DecryptAndStoreDrmDeviceCertKey(
        const void*         pData,
        size_t              dataSize,
        const AccessKey&    accessKey,
        const void*         pKeySource,
        size_t              keySourceSize ) NN_NOEXCEPT
    {
        return DecryptAndStoreDeviceUniqueKey(
            pData, dataSize,
            accessKey,
            pKeySource, keySourceSize,
            DeviceUniqueDataDecryptionMode_LoadDrmDeviceCert);
    }

    Result ModularExponentiateWithDrmDeviceCertKey(
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pCipher,
        size_t      cipherSize,
        const void* pModulus,
        size_t      modulusSize) NN_NOEXCEPT
    {
        return ModularExponentiateWithStorageKey(
            pResultBuffer, resultBufferSize,
            pCipher, cipherSize,
            pModulus, modulusSize,
            ModularExponentiateWithStorageKeyMode_DrmDeviceCert);
    }

    Result IsDevelopment(bool* pIsDevelopment) NN_NOEXCEPT
    {
        Bit64 hardwareState;
        auto result = GetConfig(&hardwareState, ConfigItem_HardwareState);
        if (result.IsFailure())
        {
            return result;
        }

        if (hardwareState == HardwareState_Development)
        {
            *pIsDevelopment = true;
        }
        else
        {
            *pIsDevelopment = false;
        }

        return ResultSuccess();
    }

    Result GenerateSpecificAesKey(
        void*       pOutBuffer,
        size_t      outBufferSize,
        const void* pSource,
        size_t      sourceSize,
        int         generation,
        Bit32       purpose  ) NN_NOEXCEPT
    {
        Bit64 source[2];
        Bit64 key[2];

        if( sourceSize != sizeof(source) )
        {
            return ResultInvalidBufferSize();
        }
        if( outBufferSize != sizeof(key) )
        {
            return ResultInvalidBufferSize();
        }

        std::memcpy(source, pSource, sizeof(source));

        auto sr = smc::GenerateSpecificAesKey(key, source, generation, purpose);

        std::memcpy(pOutBuffer, &key, sizeof(key));
        return MakeSplResult(sr);
    }

    Result DecryptAesKey(
        void*       pOutBuffer,
        size_t      outBufferSize,
        const void* pSource,
        size_t      sourceSize,
        int         generation,
        Bit32       option) NN_NOEXCEPT
    {
        Bit64 source[2];
        Bit64 key[2];

        if( sourceSize != sizeof(source) )
        {
            return ResultInvalidBufferSize();
        }
        if( outBufferSize != sizeof(key) )
        {
            return ResultInvalidBufferSize();
        }

        std::memcpy(source, pSource, sizeof(source));

        AccessKey accessKey;
        auto result = GenerateAesKek(
            &accessKey,
            AesKeyDecriptionSource, sizeof(AesKeyDecriptionSource),
            generation,
            option);
        if( result.IsFailure() )
        {
            return result;
        }

        result = GenerateAesKey(
            key, sizeof(key),
            accessKey,
            source, sizeof(source));
        if( result.IsFailure() )
        {
            return result;
        }

        std::memcpy(pOutBuffer, &key, sizeof(key));
        return ResultSuccess();
    }

    Result ComputeCtr(
        void*       pOutBuffer,
        size_t      outBufferSize,
        int         slotIndex,
        const void* pInBuffer,
        size_t      inBufferSize,
        const void* pInitialCounter,
        size_t      initialCounterSize ) NN_NOEXCEPT
    {
        if( inBufferSize == 0 )
        {
            return ResultSuccess();
        }

        // マップするためのアドレス計算
        const uintptr_t inBufferAddress  = reinterpret_cast<uintptr_t>(pInBuffer);
        const uintptr_t outBufferAddress = reinterpret_cast<uintptr_t>(pOutBuffer);

        // 前後に広げて 4 K アライメントする
        const uintptr_t inBufferAddressAligned  = util::align_down(inBufferAddress,  dd::DeviceAddressSpaceMemoryRegionAlignment);
        const uintptr_t outBufferAddressAligned = util::align_down(outBufferAddress, dd::DeviceAddressSpaceMemoryRegionAlignment);
        const size_t    inBufferSizeAligned     = util::align_up(inBufferAddress  + inBufferSize,  dd::DeviceAddressSpaceMemoryRegionAlignment) - inBufferAddressAligned;
        const size_t    outBufferSizeAligned    = util::align_up(outBufferAddress + outBufferSize, dd::DeviceAddressSpaceMemoryRegionAlignment) - outBufferAddressAligned;

        // SE にマップするアドレス
        const uint32_t seInMapAddress   = ComputeAesInMapBase  + (inBufferAddressAligned    & DeviceAddressSpaceBlockMask);
        const uint32_t seOutMapAddress  = ComputeAesOutMapBase + (outBufferAddressAligned   & DeviceAddressSpaceBlockMask);

        // SE が読み書きするアドレス
        const uint32_t seInAddress      = ComputeAesInMapBase  + (inBufferAddress           & DeviceAddressSpaceBlockMask);
        const uint32_t seOutAddress     = ComputeAesOutMapBase + (outBufferAddress          & DeviceAddressSpaceBlockMask);


        // 出力バッファは入力データより大きくないといけない
        if( ! (inBufferSize <= outBufferSize) )
        {
            return ResultInvalidBufferSize();
        }
        // 処理対象のデータサイズは AesBlockSize の倍数でないといけない
        if( (inBufferSize % AesBlockSize) != 0 )
        {
            return ResultInvalidBufferSize();
        }

        // マップ対象のバッファのサイズは ComputeAesSizeMax を超えてはいけない
        if( ComputeAesSizeMax <= inBufferSizeAligned )
        {
            return ResultInvalidBufferSize();
        }
        if( ComputeAesSizeMax <= outBufferSizeAligned )
        {
            return ResultInvalidBufferSize();
        }

        // SE の空間にマッピング
        DeviceAddressMapper inMap(
            &g_DeviceAddressSpace,
            inBufferAddressAligned,
            inBufferSizeAligned,
            seInMapAddress,
            dd::MemoryPermission_ReadOnly
        );
        DeviceAddressMapper outMap(
            &g_DeviceAddressSpace,
            outBufferAddressAligned,
            outBufferSizeAligned,
            seOutMapAddress,
            dd::MemoryPermission_WriteOnly
        );

        // SE 用の入出力エントリ作成
        auto& inout = *reinterpret_cast<InOutEntry*>(g_WorkBuffer);

        inout.in.zero       = 0;
        inout.in.address    = seInAddress;
        inout.in.size       = inBufferSize;
        inout.out.zero      = 0;
        inout.out.address   = seOutAddress;
        inout.out.size      = outBufferSize;

        // SE が読み書きするのでキャッシュフラッシュ
        os::FlushDataCache(&inout, sizeof(inout));
        os::FlushDataCache(pInBuffer,  inBufferSize);
        os::FlushDataCache(pOutBuffer, outBufferSize);

        {
            std::lock_guard<os::Mutex> lock(g_OperationLock);
            smc::OperationKey operationKey;

            Bit32 mode = MakeComputeAesMode(smc::CipherMode_Ctr, slotIndex);
            Bit64 ivic[2];

            std::memcpy(ivic, pInitialCounter, sizeof(ivic));

            smc::SmcResult sr;
            sr = smc::ComputeAes(
                &operationKey,
                g_WorkBufferMappedAddress + offsetof(InOutEntry, out),
                mode,
                ivic,
                g_WorkBufferMappedAddress + offsetof(InOutEntry, in),
                inBufferSize );
            if( sr != smc::SmcResult_Success )
            {
                return MakeSplResult(sr);
            }

            sr = WaitAndGetResult(operationKey);
            if( sr != smc::SmcResult_Success )
            {
                return MakeSplResult(sr);
            }
        }

        // SE が書いたのでキャッシュフラッシュ
        os::FlushDataCache(pOutBuffer, outBufferSize);

        return ResultSuccess();
    }

    Result ComputeCmac(
        void*       pOutBuffer,
        size_t      outBufferSize,
        int         slotIndex,
        const void* pInBuffer,
        size_t      inBufferSize ) NN_NOEXCEPT
    {
        Bit64 mac[2];

        if( ! (outBufferSize >= AesBlockSize) )
        {
            return ResultInvalidBufferSize();
        }
        if( ! (inBufferSize <= sizeof(g_WorkBuffer)) )
        {
            return ResultInvalidBufferSize();
        }

        std::memcpy(g_WorkBuffer, pInBuffer, inBufferSize);

        auto sr = smc::ComputeCmac(mac, slotIndex, g_WorkBuffer, inBufferSize);

        std::memcpy(pOutBuffer, mac, sizeof(mac));
        return MakeSplResult(sr);
    }

    Result  LoadEsDeviceKey(
        const void*      pData,
        size_t           dataSize,
        const AccessKey& accessKey,
        const void*      pKeySource,
        size_t           keySourceSize) NN_NOEXCEPT
    {
        auto& params = *reinterpret_cast<LoadEsDeviceKeyParams*>(g_WorkBuffer);
        smc::AccessKey key;
        Bit64 source[2];

        if( ! (dataSize <= sizeof(params.data)) )
        {
            return ResultInvalidBufferSize();
        }
        if( keySourceSize != sizeof(source) )
        {
            return ResultInvalidBufferSize();
        }

        std::memcpy(&key, &accessKey, sizeof(key));
        std::memcpy(source, pKeySource, sizeof(source));
        std::memcpy(params.data, pData, dataSize);

        auto sr = smc::DecryptDeviceUniqueData(params.data, dataSize, key, source, DeviceUniqueDataDecryptionMode_LoadEsDeviceKey);

        return MakeSplResult(sr);
    }

    Result  PrepareEsTitleKey(
        AccessKey*       pAccessKey,
        const void*      pCipher,
        size_t           cipherSize,
        const void*      pModulus,
        size_t           modulusSize,
        const void*      pLabelDigest,
        size_t           labelDigestSize,
        int              generation) NN_NOEXCEPT
    {
        return PrepareEsDeviceUniqueKey(
            pAccessKey,
            pCipher, cipherSize,
            pModulus, modulusSize,
            pLabelDigest, labelDigestSize,
            smc::EsDeviceUniqueKeyType_TitleKey,
            generation);
    }

    Result  PrepareEsArchiveKey(
        AccessKey*       pAccessKey,
        const void*      pCipher,
        size_t           cipherSize,
        const void*      pModulus,
        size_t           modulusSize,
        const void*      pLabelDigest,
        size_t           labelDigestSize,
        int              generation) NN_NOEXCEPT
    {
        return PrepareEsDeviceUniqueKey(
            pAccessKey,
            pCipher, cipherSize,
            pModulus, modulusSize,
            pLabelDigest, labelDigestSize,
            smc::EsDeviceUniqueKeyType_ArchiveKey,
            generation);
    }

    Result  PrepareCommonEsTitleKey(
        AccessKey*       pAccessKey,
        const void*      pKeySource,
        size_t           keySourceSize,
        int              generation) NN_NOEXCEPT
    {
        Bit64 source[2];
        smc::AccessKey accessKey;

        if( keySourceSize != sizeof(source) )
        {
            return ResultInvalidBufferSize();
        }

        std::memcpy(source, pKeySource, sizeof(source));

        auto sr = smc::PrepareCommonEsTitleKey(&accessKey, source, generation);

        std::memcpy(pAccessKey, &accessKey, sizeof(accessKey));
        return MakeSplResult(sr);
    }

    Result  LoadPreparedAesKey(
        int              slotIndex,
        const AccessKey& accessKey) NN_NOEXCEPT
    {
        smc::AccessKey key;
        std::memcpy(&key, &accessKey, sizeof(key));

        return
            MakeSplResult(
                smc::LoadPreparedAesKey(slotIndex, key) );
    }

    Result AllocateAesKeySlot(
        int*        pSlotIndex) NN_NOEXCEPT
    {
        for( int i = 0; i < AesKeySlotCount; ++i )
        {
            if( ! g_IsAesKeySlotAllocated[i] )
            {
                g_IsAesKeySlotAllocated[i] = true;
                *pSlotIndex = i;
                return ResultSuccess();
            }
        }

        g_AesKeySlotAvailableEvent.Clear();
        return ResultNoAvailableKeySlot();
    }

    Result DeallocateAesKeySlot(
        int         slotIndex) NN_NOEXCEPT
    {
        if( ! (0 <= slotIndex && slotIndex < AesKeySlotCount) )
        {
            return ResultInvalidKeySlot();
        }
        if( ! g_IsAesKeySlotAllocated[slotIndex] )
        {
            return ResultInvalidKeySlot();
        }

        smc::AccessKey key = {};
        Bit64 source[2] = {};
        smc::LoadAesKey(slotIndex, key, source);

        g_IsAesKeySlotAllocated[slotIndex] = false;
        g_AesKeySlotAvailableEvent.Signal();
        return ResultSuccess();
    }

    nn::os::SystemEvent* GetAesKeySlotAvailableEvent() NN_NOEXCEPT
    {
        return &g_AesKeySlotAvailableEvent;
    }

    Result SetBootReason(BootReasonValue bootReason) NN_NOEXCEPT
    {
        if (g_IsBootReasonInitialized)
        {
            return ResultBootReasonAlreadyInitialized();
        }
        g_BootReason = bootReason;
        g_IsBootReasonInitialized = true;
        return ResultSuccess();
    }

    Result GetBootReason(BootReasonValue* pOut) NN_NOEXCEPT
    {
        if (!g_IsBootReasonInitialized)
        {
            return ResultBootReasonNotInitialized();
        }
        *pOut = g_BootReason;
        return ResultSuccess();
    }

}}}  // namespace nn::spl::impl

