﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/socket/socket_Api.h>

namespace nn { namespace socket { namespace resolver {

/**
 * @brief This function zeroes-out the DNS server settings.
 *
 * This function is called on library initialize and finalize.
 */
void ZeroDnsServerSettings() NN_NOEXCEPT;

/**
 * @brief Set an array of ip addresses stored in uint32_t values to be
 * the be DNS servers.
 *
 * @detail
 * The maximum array size is 2. This function uses the default DNS
 * port number of 53.
 *
 * @see SetSockaddrDnsAddresses
 *
 * @param[in] pIpv4AddressArray This parameter is a pointer to an array
 * of uint32_t values that contain IP addresses. This function does
 * not validate that the addresses are correctly formatted.
 *
 * @param[in] count the number of elements in the array
 */
Result SetInaddrDnsAddresses(const uint32_t* pIpv4AddressArray,
                             size_t count) NN_NOEXCEPT;

/**
 * @brief Set an array of of sockaddrs that contain both the address
 * and port of the DNS server.
 *
 * @detail
 * The maximum size of the array is 2
 *
 * @param[in] pAddresses This parameter is a pointer to an array
 * of uint32_t values that contain IP addresses. This function does
 * not validate that the addresses are correctly formatted.
 *
 * @param[in] length This parameter is the the number of elements
 * present in the array.
 */
Result SetSockaddrDnsAddresses(const nn::socket::SockAddrIn* pAddresses,
                               size_t length) NN_NOEXCEPT;

/**
 * @brief This function will return a single sockaddr from  the
 * DNS server array
 *
 * @param[in] nIndex This parameter is the index in the array you
 * are trying to fetch
 *
 * @param pSinAddrOut This parameter is an out parameter that points
 * at a sockaddr_in structure and on success contains the structure
 * in the DNS Server Array at 'nIndex'.
 */
Result GetDnsAddress(unsigned nIndex, nn::socket::SockAddrIn* pSinAddrOut) NN_NOEXCEPT;

/**
 * @brief This function updates the bionic thread-local structures
 * with the latest dns servers. It is called prior to any entry into
 * bionic resolver functions such as @ref bionic_gethostbyname
 */
void UpdateBionicServers() NN_NOEXCEPT;

}}};
