﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "serializer.h"
#include "serializer_Specializations.h"
#include "../../detail/socket_Allocator.h"

//#define LOG_LEVEL LOG_LEVEL_MAJOR
#define LOG_MODULE_NAME "cstring serializer" // NOLINT(preprocessor/const)
#include <nn/socket/resolver/private/resolver_DebugLog.h>

namespace nn { namespace socket { namespace resolver { namespace serializer {

template <>
size_t DNSSerializer::SizeOf(const char* pString)
{
    if ( pString == NULL )
    {
        return sizeof(char);
    };
    return strlen(pString) + 1;
}

template <>
size_t DNSSerializer::SizeOf(const char** ppString, uint32_t &countOut)
{
    size_t rc = 0;
    countOut = 0;
    rc += sizeof(uint32_t); // leading length parameter
    if (NULL == ppString)
    {
        goto bail;
    };

    for (const char** pointer = ppString; *pointer != NULL; pointer++)
    {
        ++countOut;
        rc += SizeOf(*pointer);
    };

bail:
    return rc;
}

// serialize a c-string to a buffer
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t * const pBufferOut,
                                size_t     sizeOfBuffer,
                                char    *  pString)
{
    ssize_t rc = -1;
    uint8_t * pBufferCurrent = pBufferOut;

    if ( pString == NULL && sizeOfBuffer == 0)
    {
        goto bail;
    }
    else if ( pString == NULL )
    {
        *pBufferCurrent = '\0';
        rc = 1;
        goto bail;
    }
    else if ( 0 == (rc = SizeOf(((const char*)pString))))
    {
        *pBufferCurrent = '\0';
        rc = 1;
        goto bail;
    }
    else if ( -1 == CheckToBufferArguments(pBufferCurrent,
                                           sizeOfBuffer,
                                           rc + 1,
                                           __LINE__))
    {
        rc = -1;
        goto bail;
    };

    ::memmove(pBufferCurrent, pString, rc);

bail:
    return rc;
}

// serialize a c-string from a buffer
template <>
ssize_t DNSSerializer::FromBuffer(char          *& pOut,
                                  const uint8_t *  pBufferIn,
                                  size_t           bufferSize)
{
    size_t len = 0;
    ssize_t rc = -1;

    if ( pBufferIn == NULL )
    {
        LogDebug("pBufferIn is NULL\n")
        rc = 0;
        goto bail;
    }
    else if ( bufferSize == 0 )
    {
        LogDebug("bufferSize is zero\n");
        rc = 0;
        goto bail;
    }
    else if ( bufferSize < (len = SizeOf((const char*)pBufferIn)))
    {
        rc = 1;
        goto bail;
    }
    else if (pBufferIn[0] == '\0')
    {
        rc = 1;
        goto bail;
    }

    pOut = (char*)socket::detail::Alloc(len);
    if ( NULL == pOut )
    {
        LogDebug("Unable to allocate c-string of length: %d\n", len);
        rc = -1;
        goto bail;
    };
    memmove(pOut, pBufferIn, len);
    rc = len;

bail:
    return rc;
}

// serialize a null-terminated array of strings to buffer
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t * const pBufferOut,
                                size_t     sizeOfBuffer,
                                char    ** pStringArray)
{
    ssize_t rc = -1;
    ssize_t step = -1;
    size_t totalSize = 0;
    uint8_t * pBufferCurrent = pBufferOut;
    uint32_t numberOfStrings = 0;

    if (sizeOfBuffer == 0)
    {
        LogDebug("sizeOfBuffer is 0\n");
        goto bail;
    }

    totalSize = SizeOf(const_cast<const char**>(pStringArray), numberOfStrings);

    if ( -1 == (rc = CheckToBufferArguments(pBufferCurrent,
                                            sizeOfBuffer,
                                            sizeof(uint32_t), // length
                                            __LINE__)))
    {
        LogDebug("C-String Array CheckToBufferArguments failed\n");
        goto bail;
    }
    else if ( -1 == (rc = ToBuffer(pBufferCurrent, sizeOfBuffer, numberOfStrings)))
    {
        LogDebug("C-String Array ToBuffer failed\n");
        goto bail;
    };
    pBufferCurrent += rc;

    if ( NULL != pStringArray )
    {
        // add each string
        for (char** pString = pStringArray; *pString != NULL; pString++)
        {
            if ( -1 == (step = ToBuffer(pBufferCurrent, sizeOfBuffer, *pString)))
            {
                goto bail;
            };

            pBufferCurrent += step;
            sizeOfBuffer -= step;
            rc += step;
        };
    };

    rc = pBufferCurrent - pBufferOut;

bail:
    return rc;
}


// serialize a null-terminated array of strings from a buffer
template <>
ssize_t DNSSerializer::FromBuffer(char          **& pOut,
                                  const uint8_t *   pBufferIn,
                                  size_t            bufferSize)
{
    ssize_t rc = -1;
    const uint8_t* pBufferCurrent = pBufferIn;
    pOut = NULL;
    uint32_t numberOfStrings = 0;
    unsigned int idx = 0;

    if ( pBufferIn == NULL )
    {
        LogDebug("pBufferIn is NULL\n");
        rc = 0;
        goto bail;
    }
    else if ( bufferSize == 0 )
    {
        LogDebug("bufferSize is zero\n");
        rc = 0;
        goto bail;
    }
    else if ( -1 == (rc = FromBuffer(numberOfStrings, pBufferCurrent, bufferSize)))
    {
        goto bail;
    };
    pBufferCurrent += rc;

    pOut = (char**)socket::detail::Alloc((numberOfStrings + 1) * sizeof(char*));
    if ( NULL == pOut )
    {
        LogDebug("Unable to allocate c-string array with\n", numberOfStrings);
        rc = -1;
        goto bail;
    };

    memset(pOut, '\0', ((numberOfStrings + 1) * sizeof(char*)));

    for (idx=0 ; idx < numberOfStrings; ++idx)
    {
        size_t len = strlen((char*)pBufferCurrent);
        pOut[idx] = (char*)socket::detail::Alloc( len + 1 );
        if ( NULL == pOut[idx] )
        {
            LogDebug("Unable to allocate c-string inside of array %d/%d\n",
                           idx, numberOfStrings);
            rc = -1;
            goto bail;
        };

        memmove(pOut[idx], (char*) pBufferCurrent, len + 1);
        pBufferCurrent += len + 1;
    };

    pOut[idx] = NULL;
    rc = pBufferCurrent - pBufferIn;

bail:
    // free allocated resources on error
    if ( rc < 0 &&  NULL != pOut )
    {
        idx=0;
        for (char* pString = *pOut; pString != NULL; pString = pOut[idx])
        {
            socket::detail::Free(pString);
            ++idx;
        }
        socket::detail::Free(pOut);
        pOut = NULL;
    }
    return rc;
}

}}}}// namespace nn::socket::resolver::serializer
