﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "resolver_ThreadLocalStorage.h"
#include <nn/os/os_Thread.h>
#include <nn/os/os_ThreadLocalStorage.h>
#include <nn/os/os_MutexApi.h>
#include <nn/os.h>
#include <nn/socket/resolver/private/resolver_PrivateApi.h>
#include <cstring> // for free
#include "serializer.h"
// for FreeHostentContents
#include "serializer/serializer_Specializations.h"
#include <nn/os/os_Result.public.h>
#include "../../detail/socket_Allocator.h"

namespace nn { namespace socket { namespace resolver { namespace tls { namespace Client {

//#define SOCKET_TRACE_ALLOCATIONS

#ifdef SOCKET_TRACE_ALLOCATIONS
#define NN_SOCKET_ALLOCATION_LOG NN_SDK_LOG
#else
#define NN_SOCKET_ALLOCATION_LOG(...)
#endif

/**
 * @brief a structure that contains the client-side thread local storage for the bionic resolver
 */
class ClientThreadLocalStorage
{
private:
    /** @brief currrent thread-localized host errno */
    int m_CurrentHostErrno;

    /**
     * @brief The current thread-localized struct hostent returned by gethostbyname
     */
    nn::socket::HostEnt* m_CurrentHostent;
    struct hostent* m_CurrentHostentOld;

    /** @brief current herror string */
    char* m_CurrentHerrorString;

    /** @brief the current gaistrerror string */
    char* m_CurrentGaiErrorString;

public:
    void* operator new(size_t size) NN_NOEXCEPT
    {
        return nn::socket::detail::Alloc(size);
    };

    void operator delete(void* pointer, size_t size) NN_NOEXCEPT
    {
        return nn::socket::detail::Free(pointer);
    };

    /**
     * @brief client thread local storage constructor
     */
    ClientThreadLocalStorage() NN_NOEXCEPT :
        m_CurrentHostErrno(0),
        m_CurrentHostent(NULL),
        m_CurrentHostentOld(NULL),
        m_CurrentHerrorString(NULL),
        m_CurrentGaiErrorString(NULL)
    {
        NN_SOCKET_ALLOCATION_LOG("ClientThreadLocalStorage is alive!\n");
    };

    /**
     * @brief client thread local storage destructor
     */
    ~ClientThreadLocalStorage() NN_NOEXCEPT
    {
        NN_SOCKET_ALLOCATION_LOG("ClientThreadLocalStorage is dead!\n");

        if ( NULL != m_CurrentHostent)
        {
            nn::socket::
                resolver::serializer::
                  FreeHostentContents(*m_CurrentHostent);

            socket::detail::Free(m_CurrentHostent);
            m_CurrentHostent = NULL;
        };

        if ( NULL != m_CurrentHostentOld)
        {
            nn::socket::
                resolver::serializer::
                  FreeHostentContents(*m_CurrentHostentOld);

            socket::detail::Free(m_CurrentHostentOld);
            m_CurrentHostentOld = NULL;
        };

        if ( NULL != m_CurrentHerrorString)
        {
            socket::detail::Free(m_CurrentHerrorString);
            m_CurrentHerrorString = NULL;
        }

        if ( NULL != m_CurrentGaiErrorString)
        {
            socket::detail::Free(m_CurrentGaiErrorString);
            m_CurrentGaiErrorString = NULL;
        }
    };

    /**
     * @brief initialize the client TLS
     * @return success or error result code
     */
    Result Initialize() NN_NOEXCEPT;

    /**
     * @brief finalize the client TLS
     * @return success or error result code
     */
    Result Finalize() NN_NOEXCEPT;

    /**
     * @brief get host errno (gethostbyname / gethostbyaddr)
     * @return reference to host errno
     */
    int& InternalHostErrno() NN_NOEXCEPT
    {
        return m_CurrentHostErrno;
    };

    /**
     * @brief get hostent structure currently in tls
     * @return see above
     */
    const nn::socket::HostEnt* GetCurrentHostent() NN_NOEXCEPT
    {
        return m_CurrentHostent;
    };

    /**
     * @brief get hostent structure currently in tls
     * @return see above
     */
    const struct hostent* GetCurrentHostentOld() NN_NOEXCEPT
    {
        return m_CurrentHostentOld;
    };

    /**
     * @brief set hostent structure currently in tls
     * @param pHostentIn the new hostent structure to set
     * @note frees previous structure
     * @return see above
     */
    bool SetCurrentHostent(nn::socket::HostEnt* pHostentIn) NN_NOEXCEPT
    {
        if ( nullptr != m_CurrentHostent )
        {
            nn::socket::
                resolver::serializer::
                  FreeHostentContents(*m_CurrentHostent);
            socket::detail::Free(m_CurrentHostent);
            m_CurrentHostent = nullptr;
        };

        return nullptr != (m_CurrentHostent = pHostentIn);
    };

    /**
     * @brief set hostent structure currently in tls
     * @param pHostentIn the new hostent structure to set
     * @note frees previous structure
     * @return see above
     */
    bool SetCurrentHostentOld(struct hostent* pHostentIn) NN_NOEXCEPT
    {
        if ( NULL != m_CurrentHostentOld )
        {
            nn::socket::
                resolver::serializer::
                  FreeHostentContents(*m_CurrentHostentOld);
            socket::detail::Free(m_CurrentHostentOld);
            m_CurrentHostentOld = NULL;
        };

        return NULL != (m_CurrentHostentOld = pHostentIn);
    };

    /**
     * @brief get current host error string in tls
     * @return see above
     */
    const char* GetCurrentHErrorString() NN_NOEXCEPT
    {
        return m_CurrentHerrorString;
    };

    /**
     * @brief set current host error string in tls
     * @param pStringIn the new string to set
     * @note frees previous string
     * @return false on allocation error, true otherwise
     */
    const char* SetCurrentHErrorString(const char* pStringIn) NN_NOEXCEPT
    {
        size_t len = 0;

        if ( NULL != m_CurrentHerrorString )
        {
            socket::detail::Free(m_CurrentHerrorString);
            m_CurrentHerrorString = NULL;
        };

        if ( NULL == pStringIn || 0 == (len = strlen(pStringIn)) )
        {
            goto bail;
        };

        if (NULL !=  (m_CurrentHerrorString =
                      (char*)socket::detail::Alloc(len + 1)))
        {
            memmove(m_CurrentHerrorString, pStringIn, len + 1);
        };

    bail:
        return m_CurrentHerrorString;
    };

    /**
     * @brief get current host error string in tls
     * @return see above
     */
    const char* GetCurrentGaiErrorString()
    {
        return m_CurrentGaiErrorString;
    };

    /**
     * @brief set current gai error string in tls
     * @param pStringIn the new string to set
     * @note frees previous striung
     * @return null on allocation error, string otherwise
     */
    const char* SetCurrentGaiErrorString(const char* pStringIn)
    {
        size_t len = 0;

        if ( NULL != m_CurrentGaiErrorString )
        {
            socket::detail::Free(m_CurrentGaiErrorString);
            m_CurrentGaiErrorString = NULL;
        };

        if ( NULL == pStringIn || 0 == (len = strlen(pStringIn)) )
        {
            goto bail;
        };

        if (NULL != (m_CurrentGaiErrorString =
                     (char*)socket::detail::Alloc(len + 1)))
        {
            memmove(m_CurrentGaiErrorString, pStringIn, len + 1);
        };

    bail:
        return m_CurrentGaiErrorString;
    }
};

/**
 * @brief destroys client-side tls, called by ResolverThreadLocalImplementation pair
 * @param pVoid the TLS structure to delete
 */
void ClientThreadLocalStorage_dtor(void* pVoid)
{
    ClientThreadLocalStorage* pTls = reinterpret_cast<ClientThreadLocalStorage*>(pVoid);

    if ( pTls == NULL )
    {
        NN_SDK_LOG("ClientThreadLocalStorage_dtor should not be reached with NULL pointer.");
        NN_SDK_ASSERT(false);
    }
    else
    {
        delete pTls;
        pTls = NULL;
    };

    return;
};

/**
 * @brief public function to initialize the client TLS
 * @return result code indicating failure or succes
 */
Result Initialize()
{
    return nn::socket::resolver::tls::Initialize();
}

/**
 * @brief get the client-side TLS; if it does not exist then attempt to create it
 * @param pTlsOut the tls structure on create
 * @return result code indicating failure or success
 */
Result GetCreateClientTLS(void* &pTlsOut)
{
    Result result = ResultSuccess();

    pTlsOut = NULL;

    result = nn::socket::resolver::tls::GetClientTLS(pTlsOut);

    if (result.IsSuccess() && NULL != pTlsOut ) // success
    {
        result = ResultSuccess();
        goto bail;
    }
    else if ((result = nn::socket::resolver::tls::Initialize()).IsFailure() )
    {
        goto bail;
    }

    if ( NULL == ( pTlsOut = new ClientThreadLocalStorage() ) )
    {
        result = ResultOutOfMemory();
        goto bail;
    }
    else if ((result = nn::socket::resolver::tls::SetClientTLS(ClientThreadLocalStorage_dtor,
                                                                pTlsOut )).IsFailure())
    {
        NN_SDK_LOG("Unable to set the client tls structure.\n");
        delete static_cast<ClientThreadLocalStorage*>(pTlsOut);
        pTlsOut = NULL;
        result = ResultInternalError();
        goto bail;
    };

    result = ResultSuccess();

bail:
    return result;
};

// finalize the client TLS
Result Finalize()
{
    // Use the generic finalizer
    return nn::socket::resolver::tls::Finalize();
}

int& InternalHostErrno()
{
    static int dontcrash = -1;
    void* pTls = NULL;

    if (GetCreateClientTLS(pTls).IsFailure() || pTls == NULL )
    {
        NN_SDK_LOG("%s error %d\n", __FUNCTION__, __LINE__);
        return dontcrash;
    };

    return reinterpret_cast<ClientThreadLocalStorage*>(pTls)->InternalHostErrno();
};

const nn::socket::HostEnt* GetCurrentHostent()
{
    void* pTls = NULL;

    if (GetCreateClientTLS(pTls).IsFailure() || pTls == NULL )
    {
        NN_SDK_LOG("%s error %d\n", __FUNCTION__, __LINE__);
        return NULL;
    };

    return reinterpret_cast<ClientThreadLocalStorage*>(pTls)->GetCurrentHostent();
};

const struct hostent* GetCurrentHostentOld()
{
    void* pTls = NULL;

    if (GetCreateClientTLS(pTls).IsFailure() || pTls == NULL )
    {
        NN_SDK_LOG("%s error %d\n", __FUNCTION__, __LINE__);
        return NULL;
    };

    return reinterpret_cast<ClientThreadLocalStorage*>(pTls)->GetCurrentHostentOld();
};

// takes ownership of the hostent
bool SetCurrentHostent(nn::socket::HostEnt* pHostentIn)
{
    void* pTls = NULL;

    if (GetCreateClientTLS(pTls).IsFailure() || pTls == NULL )
    {

        NN_SDK_LOG("%s error %d\n", __FUNCTION__, __LINE__);
        return false;
    };

    return reinterpret_cast<ClientThreadLocalStorage*>(pTls)->SetCurrentHostent(pHostentIn);
};

// takes ownership of the hostent
bool SetCurrentHostentOld(struct hostent* pHostentIn)
{
    void* pTls = NULL;

    if (GetCreateClientTLS(pTls).IsFailure() || pTls == NULL )
    {

        NN_SDK_LOG("%s error %d\n", __FUNCTION__, __LINE__);
        return false;
    };

    return reinterpret_cast<ClientThreadLocalStorage*>(pTls)->SetCurrentHostentOld(pHostentIn);
};

const char* GetCurrentHErrorString()
{
    void* pTls = NULL;

    if (GetCreateClientTLS(pTls).IsFailure() || pTls == NULL )
    {
        NN_SDK_LOG("%s error %d\n", __FUNCTION__, __LINE__);
        return NULL;
    };

    return reinterpret_cast<ClientThreadLocalStorage*>(pTls)->GetCurrentHErrorString();
};

const char* SetCurrentHErrorString(const char* pStringIn)
{
    void* pTls = NULL;

    if (GetCreateClientTLS(pTls).IsFailure() || pTls == NULL )
    {
        NN_SDK_LOG("%s error %d\n", __FUNCTION__, __LINE__);
        return NULL;
    };

    return reinterpret_cast<ClientThreadLocalStorage*>(pTls)->SetCurrentHErrorString(pStringIn);
};

const char* GetCurrentGaiErrorString()
{
    void* pTls = NULL;

    if (GetCreateClientTLS(pTls).IsFailure() || pTls == NULL )
    {
        NN_SDK_LOG("%s error %d\n", __FUNCTION__, __LINE__);
        return NULL;
    };

    return reinterpret_cast<ClientThreadLocalStorage*>(pTls)->GetCurrentGaiErrorString();
};

const char* SetCurrentGaiErrorString(const char* pStringIn)
{
    void* pTls = NULL;

    if (GetCreateClientTLS(pTls).IsFailure() || pTls == NULL )
    {
        NN_SDK_LOG("%s error %d\n", __FUNCTION__, __LINE__);
        return NULL;
    };

    return reinterpret_cast<ClientThreadLocalStorage*>(pTls)->SetCurrentGaiErrorString(pStringIn);
};

}}}}} //nn::socket::resolver::tls::Client
