﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/sf/sf_Types.h>

#include <nn/os/os_Result.h>
#include <nn/os/os_Result.public.h>

#include <nn/socket/socket_TypesPrivate.h>
#include <nn/socket/socket_ResolverOptionsPrivate.h>
#include <nn/socket/resolver/sfdl/resolver.sfdl.h>
#include <nn/socket/resolver/private/resolver_PrivateApi.h>
#include <nn/socket/private/socket_PlatformTypesTranslation.h>
#include <nn/socket/resolver/resolver_Client.h>

#include <nn/util/util_FormatString.h>

#include "resolver_CreateClient.h"
#include "resolver_ThreadLocalStorage.h"
#include "serializer.h"
#include "serializer_Specializations.h"
#include "resolver_ClientServerShared.h"
#include "resolver_LibraryReferenceCountHelper.h"
#include "../../detail/socket_Allocator.h"

#include <random>

//#define LOG_LEVEL LOG_LEVEL_HEX
#define LOG_MODULE_NAME "client shim" // NOLINT(preprocessor/const)
#include <nn/socket/resolver/private/resolver_DebugLog.h>

namespace nn { namespace socket { namespace resolver {

namespace {

bool g_resolverClientInitialized = false;

os::Mutex g_LocalResolverOptionsLock(true);
static size_t g_LocalResolverOptionsTable[] = { 512,   //0: ResolverOptionKey::LocalGetHostByNameBufferSizeUnsigned64
                                                512,   //1: ResolverOptionKey::LocalGetHostByAddrBufferSizeUnsigned64
                                                128,   //2: ResolverOptionKey::LocalGetHostErrorStringBufferSizeUnsigned64
                                                128,   //3: ResolverOptionKey::LocalGaiErrorStringBufferSizeUnsigned64
                                                4096,  //4: ResolverOptionKey::LocalGetAddrInfoBufferSizeUnsigned64
                                                1024   //5: ResolverOptionKey::LocalGetAddrInfoHintsBufferSizeUnsigned64
};

static const char *InfoFailureString = "Error string is not available";
static const char *InfoRetryString = "System busy, try again";

const unsigned g_ResolverClientOptionsVersion = 1;

nn::socket::HostEnt* GetHostEntByNamePrivate(const char* pHostname,
                                             const ResolverOption* pOptions,
                                             size_t optionsCount) NN_NOEXCEPT
{
    nn::socket::HostEnt* pHostentry = nullptr;
    Result result = ResultInternalError();
    nn::sf::SharedPointer<IResolver> resolverClient = nullptr;
    uint8_t* pOptionsBuffer = nullptr;
    uint8_t* pOutBuffer = nullptr;
    size_t resultHostentBufferSize = 0;
    ResolverOption temp;
    uint32_t bufferLength = 0;

    if(!IncrementSocketIfNeeded())
    {
        LogDebug("\n");
        socket::SetLastError(nn::socket::Errno::EInval);
        return pHostentry;
    }
    else if (-1 == ResolverGetOption(&temp, static_cast<ResolverOptionKey>(ResolverOptionLocalKey::GetHostByNameBufferSizeUnsigned64)))
    {
        LogMajor("\n");
        tls::Client::InternalHostErrno() = NETDB_INTERNAL;
        nn::socket::SetLastError(nn::socket::Errno::ENoSpc);
        goto bail;
    }
    else if ( 0 == (resultHostentBufferSize = temp.data.unsigned64Value))
    {
        LogMajor("\n");
        tls::Client::InternalHostErrno() = NETDB_INTERNAL;
        nn::socket::SetLastError(nn::socket::Errno::ENoSpc);
        goto bail;
    }
    if ( nullptr == pHostname )
    {
        LogDebug("\n");
        tls::Client::InternalHostErrno() = HOST_NOT_FOUND;
        nn::socket::SetLastError(nn::socket::Errno::EInval);
        goto bail;
    }
    else if ( (result = CreateResolverByHipc(resolverClient)).IsFailure() )
    {
        LogMajor("\n");
        tls::Client::InternalHostErrno() = NETDB_INTERNAL;
        socket::SetLastError(nn::socket::Errno::EAgain);
        goto bail;
    }
    else
    {
        serializer::DNSSerializer serializer;

        const size_t hostLength = strlen(pHostname) + 1;
        nn::sf::InBuffer hostnameInBuffer(pHostname, hostLength);

        const size_t optionsSize = serializer.SizeOf(pOptions, optionsCount);
        pOptionsBuffer = static_cast<uint8_t*>(detail::Calloc(1, optionsSize));
        if ( nullptr == pOptionsBuffer )
        {
            LogMajor("\n");
            *nn::socket::GetHErrno() = NETDB_INTERNAL;
            socket::SetLastError(nn::socket::Errno::ENoMem);
            goto bail;
        };
        serializer.ToBuffer(pOptionsBuffer, optionsSize, pOptions, optionsCount);
        nn::sf::InBuffer optionsInBuffer(reinterpret_cast<const char*>(pOptionsBuffer), optionsSize);

        pOutBuffer = static_cast<unsigned char*>(detail::Calloc(1, resultHostentBufferSize));
        if ( nullptr == pOutBuffer )
        {
            LogMajor("\n");
            *nn::socket::GetHErrno() = NETDB_INTERNAL;
            socket::SetLastError(nn::socket::Errno::ENoMem);
            goto bail;
        };

        // hostent return value
        nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(pOutBuffer), resultHostentBufferSize);

        // this frees up previous structures
        tls::Client::SetCurrentHostent( nullptr );

        result = resolverClient->
            GetHostByNameRequestWithOptions(0,
                                            hostnameInBuffer,
                                            outBuffer,
                                            &bufferLength,
                                            g_ResolverClientOptionsVersion,
                                            optionsInBuffer,
                                            optionsCount,
                                            nn::socket::GetHErrno(),
                                            &errno);

        if ( result.IsFailure() )
        {
            NN_SDK_ASSERT(false, "GetHostByNameRequest failure");
            goto bail;
        }
        else if (0 != bufferLength)
        {
            if (nullptr == (pHostentry = reinterpret_cast<nn::socket::HostEnt*>
                                        (detail::Calloc(1, sizeof(*pHostentry)))))
            {
                LogDebug("\n");
                tls::Client::InternalHostErrno() = TRY_AGAIN;
                socket::SetLastError(nn::socket::Errno::ENoMem);
                goto bail;
            }
            else if ( -1 == serializer.FromBuffer(*pHostentry,
                                                  reinterpret_cast<uint8_t*>(pOutBuffer),
                                                  bufferLength))
            {
                LogDebug("\n");
                if ( nn::socket::Errno::ENoMem == nn::socket::GetLastError() )
                {
                    tls::Client::InternalHostErrno() = TRY_AGAIN;
                }
                else
                {
                    tls::Client::InternalHostErrno() = NETDB_INTERNAL;
                };

                if ( nullptr != pHostentry )
                {
                    serializer::FreeHostentContents(*pHostentry);
                    pHostentry = nullptr;
                };
            };
            // set global hostent to null on error
            tls::Client::SetCurrentHostent(pHostentry);
        };
    };

bail:
    if (pOutBuffer != nullptr)
    {
        socket::detail::Free(pOutBuffer);
        pOutBuffer = nullptr;
    };

    if (pOptionsBuffer != nullptr)
    {
        socket::detail::Free(pOptionsBuffer);
        pOptionsBuffer = nullptr;
    };

    resolverClient = nullptr;

    DecrementSocketAndSignalIfNeeded();

    return pHostentry;
}; //NOLINT(impl/function_size)

struct hostent* GetHostByNamePrivate(const char* pHostname,
                                     const ResolverOption* pOptions,
                                     size_t optionsCount) NN_NOEXCEPT
{
    struct hostent* pHostentry = nullptr;
    Result result = ResultInternalError();
    nn::sf::SharedPointer<IResolver> resolverClient = nullptr;
    uint8_t* pOptionsBuffer = nullptr;
    uint8_t* pOutBuffer = nullptr;
    size_t resultHostentBufferSize = 0;
    ResolverOption temp;
    uint32_t bufferLength = 0;

    if(!IncrementSocketIfNeeded())
    {
        LogDebug("\n");
        socket::SetLastError(socket::Errno::EInval);
        return pHostentry;
    }
    else if (-1 == ResolverGetOption(&temp, static_cast<ResolverOptionKey>(ResolverOptionLocalKey::GetHostByNameBufferSizeUnsigned64)))
    {
        LogMajor("\n");
        tls::Client::InternalHostErrno() = NETDB_INTERNAL;
        nn::socket::SetLastError(socket::Errno::ENoSpc);
        goto bail;
    }
    else if ( 0 == (resultHostentBufferSize = temp.data.unsigned64Value))
    {
        LogMajor("\n");
        tls::Client::InternalHostErrno() = NETDB_INTERNAL;
        nn::socket::SetLastError(socket::Errno::ENoSpc);
        goto bail;
    }
    if ( nullptr == pHostname )
    {
        LogDebug("\n");
        tls::Client::InternalHostErrno() = HOST_NOT_FOUND;
        nn::socket::SetLastError(socket::Errno::EInval);
        goto bail;
    }
    else if ( (result = CreateResolverByHipc(resolverClient)).IsFailure() )
    {
        LogMajor("\n");
        tls::Client::InternalHostErrno() = NETDB_INTERNAL;
        socket::SetLastError(socket::Errno::EAgain);
        goto bail;
    }
    else
    {
        serializer::DNSSerializer serializer;

        const size_t hostLength = strlen(pHostname) + 1;
        nn::sf::InBuffer hostnameInBuffer(pHostname, hostLength);

        const size_t optionsSize = serializer.SizeOf(pOptions, optionsCount);
        pOptionsBuffer = static_cast<uint8_t*>(detail::Calloc(1, optionsSize));
        if ( nullptr == pOptionsBuffer )
        {
            LogMajor("\n");
            *nn::socket::GetHErrno() = NETDB_INTERNAL;
            socket::SetLastError(socket::Errno::ENoMem);
            goto bail;
        };
        serializer.ToBuffer(pOptionsBuffer, optionsSize, pOptions, optionsCount);
        nn::sf::InBuffer optionsInBuffer(reinterpret_cast<const char*>(pOptionsBuffer), optionsSize);

        pOutBuffer = static_cast<unsigned char*>(detail::Calloc(1, resultHostentBufferSize));
        if ( nullptr == pOutBuffer )
        {
            LogMajor("\n");
            *nn::socket::GetHErrno() = NETDB_INTERNAL;
            socket::SetLastError(socket::Errno::ENoMem);
            goto bail;
        };

        // hostent return value
        nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(pOutBuffer), resultHostentBufferSize);

        // this frees up previous structures
        tls::Client::SetCurrentHostentOld( nullptr );

        result = resolverClient->
            GetHostByNameRequestWithOptions(0,
                                            hostnameInBuffer,
                                            outBuffer,
                                            &bufferLength,
                                            g_ResolverClientOptionsVersion,
                                            optionsInBuffer,
                                            optionsCount,
                                            nn::socket::GetHErrno(),
                                            &errno);

        if ( result.IsFailure() )
        {
            NN_SDK_ASSERT(false, "GetHostByNameRequest failure");
            goto bail;
        }
        else if (0 != bufferLength)
        {
            if (nullptr == (pHostentry =
                            reinterpret_cast<struct hostent*>(
                                detail::Calloc(1, sizeof(struct hostent)))))
            {
                LogDebug("\n");
                tls::Client::InternalHostErrno() = TRY_AGAIN;
                socket::SetLastError(socket::Errno::ENoMem);
                goto bail;
            }
            else if ( -1 == serializer.FromBuffer(*pHostentry,
                                                  reinterpret_cast<uint8_t*>(pOutBuffer),
                                                  bufferLength))
            {
                LogDebug("\n");
                if ( nn::socket::Errno::ENoMem == nn::socket::GetLastError() )
                {
                    tls::Client::InternalHostErrno() = TRY_AGAIN;
                }
                else
                {
                    tls::Client::InternalHostErrno() = NETDB_INTERNAL;
                };

                if ( nullptr != pHostentry )
                {
                    serializer::FreeHostentContents(*pHostentry);
                    pHostentry = nullptr;
                };
            };
            // set global hostent to null on error
            tls::Client::SetCurrentHostentOld(pHostentry);
        };
    };

bail:
    if (pOutBuffer != nullptr)
    {
        socket::detail::Free(pOutBuffer);
        pOutBuffer = nullptr;
    };

    if (pOptionsBuffer != nullptr)
    {
        socket::detail::Free(pOptionsBuffer);
        pOptionsBuffer = nullptr;
    };

    resolverClient = nullptr;

    DecrementSocketAndSignalIfNeeded();

    return pHostentry;
}; //NOLINT(impl/function_size)

nn::socket::AiErrno GetAddrInfoPrivate(const char* node,
                                       const char* service,
                                       const nn::socket::AddrInfo* hints,
                                       nn::socket::AddrInfo** ppAddrinfoOut,
                                       const ResolverOption* pOptions,
                                       size_t optionsCount) NN_NOEXCEPT
{
    int32_t rc = -1;
    nn::socket::AiErrno returnValue = nn::socket::AiErrno::EAi_Fail;
    Result result = ResultInternalError();
    nn::sf::SharedPointer<IResolver> resolverClient = nullptr;
    size_t resultAddrinfoBufferSize = 0;
    size_t hintsAddrinfoBufferSize = 0;
    uint8_t* pOutBuffer = nullptr;
    ResolverOption temp;
    uint8_t* pOptionsBuffer = nullptr;
    uint8_t* pHintsBuffer = nullptr;

    if(!IncrementSocketIfNeeded())
    {
        LogDebug("\n");
        return nn::socket::AiErrno::EAi_System;
    }
    else if ( nullptr == node && nullptr == service )
    {
        LogDebug("\n");
        returnValue = nn::socket::AiErrno::EAi_NoName;
        goto bail;
    }
    else if ( ppAddrinfoOut == nullptr )
    {
        LogDebug("\n");
        returnValue  = nn::socket::AiErrno::EAi_System;
        socket::SetLastError(nn::socket::Errno::EInval);
        goto bail;
    }
    else if ( (result = CreateResolverByHipc(resolverClient)).IsFailure() )
    {
        LogDebug("\n");
        returnValue = nn::socket::AiErrno::EAi_Again;
        socket::SetLastError(nn::socket::Errno::EAgain);
        goto bail;
    }
    else if (-1 == ResolverGetOption(&temp, static_cast<ResolverOptionKey>(ResolverOptionLocalKey::GetAddrInfoBufferSizeUnsigned64)))
    {
        LogMajor("\n");
        tls::Client::InternalHostErrno() = NETDB_INTERNAL;
        nn::socket::SetLastError(nn::socket::Errno::ENoSpc);
        goto bail;
    }
    else if (0 == (resultAddrinfoBufferSize = temp.data.unsigned64Value))
    {
        LogMajor("\n");
        tls::Client::InternalHostErrno() = NETDB_INTERNAL;
        nn::socket::SetLastError(nn::socket::Errno::ENoSpc);
        goto bail;
    }
    // todo: use the stack instead
    else if (-1 == ResolverGetOption(&temp, static_cast<ResolverOptionKey>(ResolverOptionLocalKey::GetAddrInfoHintsBufferSizeUnsigned64)))
    {
        LogMajor("\n");
        tls::Client::InternalHostErrno() = NETDB_INTERNAL;
        nn::socket::SetLastError(nn::socket::Errno::ENoSpc);
        goto bail;
    }
    else if (0 == ( hintsAddrinfoBufferSize = temp.data.unsigned64Value))
    {
        LogMajor("\n");
        tls::Client::InternalHostErrno() = NETDB_INTERNAL;
        nn::socket::SetLastError(nn::socket::Errno::ENoSpc);
        goto bail;
    }
    else
    {
        *ppAddrinfoOut = nullptr;

        serializer::DNSSerializer serializer;

        size_t nodeLength = 0;
        if ( nullptr != node )
        {
            nodeLength = strlen(node) + 1;
        };
        nn::sf::InBuffer nodenameInBuffer(node, nodeLength);

        size_t serviceLength = 0;
        if ( nullptr != service )
        {
            serviceLength = strlen(service) + 1;
        };
        nn::sf::InBuffer servicenameInBuffer(service, serviceLength);

        const size_t optionsBufferSize = serializer.SizeOf(pOptions, optionsCount);
        pOptionsBuffer = reinterpret_cast<uint8_t*>(detail::Calloc(1, optionsBufferSize));
        if (nullptr == pOptionsBuffer)
        {
            LogMajor("\n");
            goto bail;
        };
        serializer.ToBuffer(pOptionsBuffer, optionsBufferSize, pOptions, optionsCount);
        nn::sf::InBuffer optionsInBuffer(reinterpret_cast<const char*>(pOptionsBuffer), optionsBufferSize);

        pHintsBuffer = reinterpret_cast<uint8_t*>(detail::Calloc(1, hintsAddrinfoBufferSize));
        if (nullptr == pHintsBuffer)
        {
            LogMajor("\n");
            goto bail;
        };
        nn::sf::InBuffer hintsInBuffer(reinterpret_cast<char*>(pHintsBuffer), hintsAddrinfoBufferSize);
        nn::sf::InBuffer hintsNullBuffer(nullptr, 0);

        pOutBuffer = static_cast<unsigned char*>(detail::Calloc(1, resultAddrinfoBufferSize));
        if ( pOutBuffer == nullptr )
        {
            *nn::socket::GetHErrno() = NETDB_INTERNAL;
            nn::socket::SetLastError(nn::socket::Errno::ENoMem);
            returnValue = nn::socket::AiErrno::EAi_Memory;
            goto bail;
        };
        nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(pOutBuffer), resultAddrinfoBufferSize);

        uint32_t bufferLength = 0;

        if (nullptr != hints && -1 == (rc = serializer.ToBuffer(pHintsBuffer,
                                                                hintsAddrinfoBufferSize,
                                                                *hints)))
        {
            LogDebug("\n");
            returnValue = nn::socket::AiErrno::EAi_Overflow;
            goto bail;
        }
        else if (nullptr == ( *ppAddrinfoOut =
                              static_cast<nn::socket::AddrInfo*>(
                                  detail::Calloc(1, sizeof(nn::socket::AddrInfo)))))
        {
            LogDebug("\n");
            *nn::socket::GetHErrno() = NETDB_INTERNAL;
            nn::socket::SetLastError(nn::socket::Errno::ENoMem);
            returnValue = nn::socket::AiErrno::EAi_Memory;
            goto bail;
        };

        result = resolverClient->
            GetAddrInfoRequestWithOptions(0,
                                          nodenameInBuffer,
                                          servicenameInBuffer,
                                          (hints == nullptr) ? hintsNullBuffer : hintsInBuffer,
                                          outBuffer,
                                          &bufferLength,
                                          reinterpret_cast<int32_t*>(&returnValue),
                                          g_ResolverClientOptionsVersion,
                                          optionsInBuffer,
                                          optionsCount,
                                          nn::socket::GetHErrno(),
                                          &errno);

        if ( result.IsFailure() )
        {
            LogDebug("\n");
            NN_SDK_ASSERT(false, "GetAddrInfoRequest failure\n");
            socket::FreeAddrInfo(*ppAddrinfoOut);
            *ppAddrinfoOut = nullptr;
            // returnValue already set (EAI_FAIL)
            goto bail;
        }
        else if (nn::socket::AiErrno::EAi_Success != returnValue)
        {
            LogDebug("\n");
            // Return values are not fatal errors.
            socket::FreeAddrInfo(*ppAddrinfoOut);
            *ppAddrinfoOut = nullptr;
            goto bail;
        }
        else if (0 == bufferLength)
        {
            LogDebug("\n");
            socket::FreeAddrInfo(*ppAddrinfoOut);
            *ppAddrinfoOut = nullptr;
            goto bail;
        }
        else if ( -1 == (rc = serializer.FromBuffer(**ppAddrinfoOut,
                                                    reinterpret_cast<uint8_t*>(pOutBuffer),
                                                    bufferLength)))
        {
            LogDebug("\n");
            tls::Client::InternalHostErrno() = NETDB_INTERNAL;
            if ( nn::socket::Errno::ENoMem == nn::socket::GetLastError() )
            {
                returnValue = nn::socket::AiErrno::EAi_Memory;
            };

            if ( nullptr != ppAddrinfoOut && nullptr != *ppAddrinfoOut)
            {
                socket::FreeAddrInfo(*ppAddrinfoOut);
                *ppAddrinfoOut = nullptr;
            };

            goto bail;
        };
    };

bail:

    if ( pOutBuffer != nullptr )
    {
        socket::detail::Free(pOutBuffer);
        pOutBuffer = nullptr;
    };

    if (nullptr != pHintsBuffer)
    {
        socket::detail::Free(pHintsBuffer);
        pHintsBuffer = nullptr;
    };

    if (nullptr != pOptionsBuffer)
    {
        socket::detail::Free(pOptionsBuffer);
        pOptionsBuffer = nullptr;
    };

    resolverClient = nullptr;

    DecrementSocketAndSignalIfNeeded();

    return returnValue;
}  // NOLINT(impl/function_size)

int GetAddrInfoPrivate(const char* node,
                       const char* service,
                       const struct addrinfo* hints,
                       struct addrinfo** ppAddrinfoOut,
                       const ResolverOption* pOptions,
                       size_t optionsCount) NN_NOEXCEPT
{
    int32_t rc = -1;
    int32_t returnValue = EAI_FAIL;
    Result result = ResultInternalError();
    nn::sf::SharedPointer<IResolver> resolverClient = nullptr;
    size_t resultAddrinfoBufferSize = 0;
    size_t hintsAddrinfoBufferSize = 0;
    uint8_t* pOutBuffer = nullptr;
    ResolverOption temp;
    uint8_t* pOptionsBuffer = nullptr;
    uint8_t* pHintsBuffer = nullptr;

    if(!IncrementSocketIfNeeded())
    {
        LogDebug("\n");
        return EAI_SYSTEM;
    }
    else if ( nullptr == node && nullptr == service )
    {
        LogDebug("\n");
        returnValue = EAI_NONAME;
        goto bail;
    }
    else if ( ppAddrinfoOut == nullptr )
    {
        LogDebug("\n");
        returnValue  = EAI_SYSTEM;
        socket::SetLastError(socket::Errno::EInval);
        goto bail;
    }
    else if ( (result = CreateResolverByHipc(resolverClient)).IsFailure() )
    {
        LogDebug("\n");
        returnValue = EAI_AGAIN;
        socket::SetLastError(socket::Errno::EAgain);
        goto bail;
    }
    else if (-1 == ResolverGetOption(&temp, static_cast<ResolverOptionKey>(ResolverOptionLocalKey::GetAddrInfoBufferSizeUnsigned64)))
    {
        LogMajor("\n");
        tls::Client::InternalHostErrno() = NETDB_INTERNAL;
        nn::socket::SetLastError(socket::Errno::ENoSpc);
        goto bail;
    }
    else if (0 == (resultAddrinfoBufferSize = temp.data.unsigned64Value))
    {
        LogMajor("\n");
        tls::Client::InternalHostErrno() = NETDB_INTERNAL;
        nn::socket::SetLastError(socket::Errno::ENoSpc);
        goto bail;
    }
    // todo: use the stack instead
    else if (-1 == ResolverGetOption(&temp, static_cast<ResolverOptionKey>(ResolverOptionLocalKey::GetAddrInfoHintsBufferSizeUnsigned64)))
    {
        LogMajor("\n");
        tls::Client::InternalHostErrno() = NETDB_INTERNAL;
        nn::socket::SetLastError(socket::Errno::ENoSpc);
        goto bail;
    }
    else if (0 == ( hintsAddrinfoBufferSize = temp.data.unsigned64Value))
    {
        LogMajor("\n");
        tls::Client::InternalHostErrno() = NETDB_INTERNAL;
        nn::socket::SetLastError(socket::Errno::ENoSpc);
        goto bail;
    }
    else
    {
        *ppAddrinfoOut = nullptr;

        serializer::DNSSerializer serializer;

        size_t nodeLength = 0;
        if ( nullptr != node )
        {
            nodeLength = strlen(node) + 1;
        };
        nn::sf::InBuffer nodenameInBuffer(node, nodeLength);

        size_t serviceLength = 0;
        if ( nullptr != service )
        {
            serviceLength = strlen(service) + 1;
        };
        nn::sf::InBuffer servicenameInBuffer(service, serviceLength);

        const size_t optionsBufferSize = serializer.SizeOf(pOptions, optionsCount);
        pOptionsBuffer = reinterpret_cast<uint8_t*>(detail::Calloc(1, optionsBufferSize));
        if (nullptr == pOptionsBuffer)
        {
            LogMajor("\n");
            goto bail;
        };
        serializer.ToBuffer(pOptionsBuffer, optionsBufferSize, pOptions, optionsCount);
        nn::sf::InBuffer optionsInBuffer(reinterpret_cast<const char*>(pOptionsBuffer), optionsBufferSize);

        pHintsBuffer = reinterpret_cast<uint8_t*>(detail::Calloc(1, hintsAddrinfoBufferSize));
        if (nullptr == pHintsBuffer)
        {
            LogMajor("\n");
            goto bail;
        };
        nn::sf::InBuffer hintsInBuffer(reinterpret_cast<char*>(pHintsBuffer), hintsAddrinfoBufferSize);
        nn::sf::InBuffer hintsNullBuffer(nullptr, 0);

        pOutBuffer = static_cast<unsigned char*>(detail::Calloc(1, resultAddrinfoBufferSize));
        if ( pOutBuffer == nullptr )
        {
            *nn::socket::GetHErrno() = NETDB_INTERNAL;
            nn::socket::SetLastError(socket::Errno::ENoMem);
            returnValue = EAI_MEMORY;
            goto bail;
        };
        nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(pOutBuffer), resultAddrinfoBufferSize);

        uint32_t bufferLength = 0;

        if (nullptr != hints && -1 == (rc = serializer.ToBuffer(pHintsBuffer,
                                                                hintsAddrinfoBufferSize,
                                                                *hints)))
        {
            LogDebug("\n");
            returnValue = EAI_OVERFLOW;
            goto bail;
        }
        else if (nullptr == ( *ppAddrinfoOut =
                              static_cast<struct addrinfo*>(
                                  detail::Calloc(1, sizeof(struct addrinfo)))))
        {
            LogDebug("\n");
            *nn::socket::GetHErrno() = NETDB_INTERNAL;
            nn::socket::SetLastError(socket::Errno::ENoMem);
            returnValue = EAI_MEMORY;
            goto bail;
        };

        result = resolverClient->
            GetAddrInfoRequestWithOptions(0,
                                          nodenameInBuffer,
                                          servicenameInBuffer,
                                          (hints == nullptr) ? hintsNullBuffer : hintsInBuffer,
                                          outBuffer,
                                          &bufferLength,
                                          &returnValue,
                                          g_ResolverClientOptionsVersion,
                                          optionsInBuffer,
                                          optionsCount,
                                          nn::socket::GetHErrno(),
                                          &errno);

        if ( result.IsFailure() )
        {
            LogDebug("\n");
            NN_SDK_ASSERT(false, "GetAddrInfoRequest failure\n");
            socket::FreeAddrInfo(*ppAddrinfoOut);
            *ppAddrinfoOut = nullptr;
            // returnValue already set (EAI_FAIL)
            goto bail;
        }
        else if (0 != returnValue)
        {
            LogDebug("\n");
            // Return values are not fatal errors.
            socket::FreeAddrInfo(*ppAddrinfoOut);
            *ppAddrinfoOut = nullptr;
            goto bail;
        }
        else if (0 == bufferLength)
        {
            LogDebug("\n");
            socket::FreeAddrInfo(*ppAddrinfoOut);
            *ppAddrinfoOut = nullptr;
            goto bail;
        }
        else if ( -1 == (rc = serializer.FromBuffer(**ppAddrinfoOut,
                                                    reinterpret_cast<uint8_t*>(pOutBuffer),
                                                    bufferLength)))
        {
            LogDebug("\n");
            tls::Client::InternalHostErrno() = NETDB_INTERNAL;
            if ( nn::socket::Errno::ENoMem == nn::socket::GetLastError() )
            {
                returnValue = EAI_MEMORY;
            };

            if ( nullptr != ppAddrinfoOut && nullptr != *ppAddrinfoOut)
            {
                socket::FreeAddrInfo(*ppAddrinfoOut);
                *ppAddrinfoOut = nullptr;
            };

            goto bail;
        };
    };

bail:

    if ( pOutBuffer != nullptr )
    {
        socket::detail::Free(pOutBuffer);
        pOutBuffer = nullptr;
    };

    if (nullptr != pHintsBuffer)
    {
        socket::detail::Free(pHintsBuffer);
        pHintsBuffer = nullptr;
    };

    if (nullptr != pOptionsBuffer)
    {
        socket::detail::Free(pOptionsBuffer);
        pOptionsBuffer = nullptr;
    };

    resolverClient = nullptr;

    DecrementSocketAndSignalIfNeeded();

    return returnValue;
}  // NOLINT(impl/function_size)

nn::socket::HostEnt* GetHostEntByAddrPrivate(const void* pInaddr,
                                             uint32_t inAddrLength,
                                             nn::socket::Family addressFamilyIn,
                                             const ResolverOption* pOptions,
                                             size_t optionsCount) NN_NOEXCEPT
{
    nn::socket::HostEnt* pHostentry = nullptr;
    Result result = ResultInternalError();
    nn::sf::SharedPointer<IResolver> resolverClient = nullptr;
    uint8_t* pOutBuffer = nullptr;
    size_t resultHostentBufferSize = 0;
    ResolverOption temp;
    uint8_t* pOptionsBuffer = nullptr;

    if(!IncrementSocketIfNeeded())
    {
        LogDebug("\n");
        return pHostentry;
    }
    else if ( -1 == (ResolverGetOption(&temp, static_cast<ResolverOptionKey>(ResolverOptionLocalKey::GetHostByAddrBufferSizeUnsigned64))))
    {
        LogMajor("\n");
        nn::socket::SetLastError(nn::socket::Errno::ENoSpc);
        goto bail;
    }
    else if (0 == (resultHostentBufferSize = temp.data.unsigned64Value))
    {
        LogMajor("\n");
        nn::socket::SetLastError(nn::socket::Errno::ENoSpc);
        goto bail;
    }
    else if ( nullptr == pInaddr )
    {
        LogDebug("\n");
        nn::socket::SetLastError(nn::socket::Errno::EInval);
        goto bail;
    }
    else if ( 0 == inAddrLength )
    {
        LogDebug("\n");
        nn::socket::SetLastError(nn::socket::Errno::EInval);
        goto bail;
    }
    else if ( nn::socket::Family::Af_Inet != addressFamilyIn)
    {
        LogDebug("\n");
        nn::socket::SetLastError(nn::socket::Errno::ENotSup);
        goto bail;
    }
    else if ( (result = CreateResolverByHipc(resolverClient)).IsFailure() )
    {
        LogMajor("\n");
        socket::SetLastError(nn::socket::Errno::EAgain);
        goto bail;
    }
    else
    {
        serializer::DNSSerializer serializer;

        nn::sf::InBuffer inAddrInBuffer(reinterpret_cast<const char*>(pInaddr), inAddrLength);
        uint32_t bufferLength = 0;

        const size_t optionsSize = serializer.SizeOf(pOptions, optionsCount);
        pOptionsBuffer = reinterpret_cast<uint8_t*>(detail::Calloc(1, optionsSize));
        if (nullptr == pOptionsBuffer)
        {
            LogMajor("\n");
            socket::SetLastError(nn::socket::Errno::EAgain);
            goto bail;
        };

        serializer.ToBuffer(pOptionsBuffer, optionsSize, pOptions, optionsCount);
        nn::sf::InBuffer optionsInBuffer(reinterpret_cast<const char*>(pOptionsBuffer), optionsSize);

        pOutBuffer = static_cast<unsigned char*>(detail::Calloc(1, resultHostentBufferSize));
        if ( pOutBuffer == nullptr )
        {
            LogDebug("\n");
            *nn::socket::GetHErrno() = NETDB_INTERNAL;
            nn::socket::SetLastError(nn::socket::Errno::ENoMem);
            goto bail;
        };

        nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(pOutBuffer), resultHostentBufferSize);

        result = resolverClient->
            GetHostByAddrRequestWithOptions(0,
                                            inAddrInBuffer,
                                            inAddrLength,
                                            static_cast<int32_t>(addressFamilyIn),
                                            outBuffer,
                                            &bufferLength,
                                            g_ResolverClientOptionsVersion,
                                            optionsInBuffer,
                                            optionsCount,
                                            nn::socket::GetHErrno(),
                                            &errno);

        if ( result.IsFailure() )
        {
            NN_SDK_ASSERT(false, "GetHostByAddrRequest failure");
            goto bail;
        }
        else if (   0 != bufferLength )
        {
            if ( nullptr == ( pHostentry =
                              static_cast<nn::socket::HostEnt*>(
                                  detail::Calloc(1, sizeof(*pHostentry)))))
            {
                LogDebug("\n");
                *nn::socket::GetHErrno() = NETDB_INTERNAL;
                nn::socket::SetLastError(nn::socket::Errno::ENoMem);
                goto bail;
            }
            else if ( -1 == serializer.FromBuffer(*pHostentry,
                                                  reinterpret_cast<uint8_t*>(pOutBuffer),
                                                  bufferLength))
            {
                LogDebug("\n");

                tls::Client::InternalHostErrno() = NETDB_INTERNAL;
                if ( nullptr != pHostentry )
                {
                    serializer::FreeHostentContents(*pHostentry);
                    pHostentry = nullptr;
                };
            };
            // set thread-localized hostent structure to null on error
            tls::Client::SetCurrentHostent(pHostentry);
        };
    };

bail:
    if ( pOutBuffer != nullptr )
    {
        socket::detail::Free(pOutBuffer);
        pOutBuffer = nullptr;
    };

    if ( pOptionsBuffer != nullptr )
    {
        socket::detail::Free(pOptionsBuffer);
        pOptionsBuffer = nullptr;
    };

    resolverClient = nullptr;

    DecrementSocketAndSignalIfNeeded();

    return pHostentry;
}; //NOLINT(impl/function_size)

struct hostent* GetHostByAddrPrivate(const void* pInaddr,
                                     uint32_t inAddrLength,
                                     uint32_t addressFamilyIn,
                                     const ResolverOption* pOptions,
                                     size_t optionsCount) NN_NOEXCEPT
{
    struct hostent* pHostentry = nullptr;
    Result result = ResultInternalError();
    nn::sf::SharedPointer<IResolver> resolverClient = nullptr;
    uint8_t* pOutBuffer = nullptr;
    size_t resultHostentBufferSize = 0;
    ResolverOption temp;
    uint8_t* pOptionsBuffer = nullptr;

    if(!IncrementSocketIfNeeded())
    {
        LogDebug("\n");
        return pHostentry;
    }
    else if ( -1 == (ResolverGetOption(&temp, static_cast<ResolverOptionKey>(ResolverOptionLocalKey::GetHostByAddrBufferSizeUnsigned64))))
    {
        LogMajor("\n");
        nn::socket::SetLastError(socket::Errno::ENoSpc);
        goto bail;
    }
    else if (0 == (resultHostentBufferSize = temp.data.unsigned64Value))
    {
        LogMajor("\n");
        nn::socket::SetLastError(socket::Errno::ENoSpc);
        goto bail;
    }
    else if ( nullptr == pInaddr )
    {
        LogDebug("\n");
        nn::socket::SetLastError(socket::Errno::EInval);
        goto bail;
    }
    else if ( 0 == inAddrLength )
    {
        LogDebug("\n");
        nn::socket::SetLastError(socket::Errno::EInval);
        goto bail;
    }
    else if ( AF_INET != addressFamilyIn)
    {
        LogDebug("\n");
        nn::socket::SetLastError(socket::Errno::ENotSup);
        goto bail;
    }
    else if ( (result = CreateResolverByHipc(resolverClient)).IsFailure() )
    {
        LogMajor("\n");
        socket::SetLastError(socket::Errno::EAgain);
        goto bail;
    }
    else
    {
        serializer::DNSSerializer serializer;

        nn::sf::InBuffer inAddrInBuffer(reinterpret_cast<const char*>(pInaddr), inAddrLength);
        uint32_t bufferLength = 0;

        const size_t optionsSize = serializer.SizeOf(pOptions, optionsCount);
        pOptionsBuffer = reinterpret_cast<uint8_t*>(detail::Calloc(1, optionsSize));
        if (nullptr == pOptionsBuffer)
        {
            LogMajor("\n");
            socket::SetLastError(socket::Errno::EAgain);
            goto bail;
        };

        serializer.ToBuffer(pOptionsBuffer, optionsSize, pOptions, optionsCount);
        nn::sf::InBuffer optionsInBuffer(reinterpret_cast<const char*>(pOptionsBuffer), optionsSize);

        pOutBuffer = static_cast<unsigned char*>(detail::Calloc(1, resultHostentBufferSize));
        if ( pOutBuffer == nullptr )
        {
            LogDebug("\n");
            *nn::socket::GetHErrno() = NETDB_INTERNAL;
            nn::socket::SetLastError(nn::socket::Errno::ENoMem);
            goto bail;
        };

        nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(pOutBuffer), resultHostentBufferSize);

        result = resolverClient->
            GetHostByAddrRequestWithOptions(0,
                                            inAddrInBuffer,
                                            inAddrLength,
                                            addressFamilyIn,
                                            outBuffer,
                                            &bufferLength,
                                            g_ResolverClientOptionsVersion,
                                            optionsInBuffer,
                                            optionsCount,
                                            nn::socket::GetHErrno(),
                                            &errno);

        if ( result.IsFailure() )
        {
            NN_SDK_ASSERT(false, "GetHostByAddrRequest failure");
            goto bail;
        }
        else if (   0 != bufferLength )
        {
            if ( nullptr == ( pHostentry =
                              static_cast<struct hostent*>(
                                  detail::Calloc(1, sizeof(struct hostent)))))
            {
                LogDebug("\n");
                *nn::socket::GetHErrno() = NETDB_INTERNAL;
                nn::socket::SetLastError(socket::Errno::ENoMem);
                goto bail;
            }
            else if ( -1 == serializer.FromBuffer(*pHostentry,
                                                  reinterpret_cast<uint8_t*>(pOutBuffer),
                                                  bufferLength))
            {
                LogDebug("\n");

                tls::Client::InternalHostErrno() = NETDB_INTERNAL;
                if ( nullptr != pHostentry )
                {
                    serializer::FreeHostentContents(*pHostentry);
                    pHostentry = nullptr;
                };
            };
            // set thread-localized hostent structure to null on error
            tls::Client::SetCurrentHostentOld(pHostentry);
        };
    };

bail:
    if ( pOutBuffer != nullptr )
    {
        socket::detail::Free(pOutBuffer);
        pOutBuffer = nullptr;
    };

    if ( pOptionsBuffer != nullptr )
    {
        socket::detail::Free(pOptionsBuffer);
        pOptionsBuffer = nullptr;
    };

    resolverClient = nullptr;

    DecrementSocketAndSignalIfNeeded();

    return pHostentry;
}; //NOLINT(impl/function_size)

nn::socket::AiErrno GetNameInfoPrivate(const nn::socket::SockAddr* sa, nn::socket::SockLenT salen,
                                       char* host, size_t hostlen,
                                       char* serv, size_t servlen,
                                       nn::socket::NameInfoFlag flags,
                                       const ResolverOption* pOptions,
                                       size_t optionsCount) NN_NOEXCEPT
{
    Result result = ResultInternalError();
    nn::sf::SharedPointer<IResolver> resolverClient = nullptr;
    nn::socket::AiErrno returnValue = nn::socket::AiErrno::EAi_Success;
    uint8_t* pOptionsBuffer = nullptr;

    if(!IncrementSocketIfNeeded())
    {
        LogDebug("\n");
        return nn::socket::AiErrno::EAi_System;
    }
    else if ( (result = CreateResolverByHipc(resolverClient)).IsFailure() )
    {
        LogDebug("\n");
        returnValue = nn::socket::AiErrno::EAi_Again;
        socket::SetLastError(nn::socket::Errno::EAgain);
        goto bail;
    }
    else if ( sa == nullptr && salen != 0 )
    {
        returnValue = nn::socket::AiErrno::EAi_Fail;
        LogDebug("nullptr struct sockaddr but nonzero length provided");
        goto bail;
    }
    else if (host == nullptr && hostlen != 0)
    {
        returnValue = nn::socket::AiErrno::EAi_Fail;
        LogDebug("nullptr host buffer but nonzero length provided");
        goto bail;
    }
    else if (serv == nullptr && servlen != 0)
    {
        returnValue = nn::socket::AiErrno::EAi_Fail;
        LogDebug("nullptr service buffer but nonzero length provided");
        goto bail;
    }
    else
    {
        serializer::DNSSerializer serializer;

        nn::sf::InBuffer serializedSocketAddressIn(reinterpret_cast<const char*>(sa),salen);

        nn::sf::OutBuffer hostBufferOut(host, hostlen);

        nn::sf::OutBuffer serviceBufferOut(serv, servlen);

        const size_t optionsSize = serializer.SizeOf(pOptions, optionsCount);
        pOptionsBuffer = reinterpret_cast<uint8_t*>(detail::Calloc(1, optionsSize));
        if (nullptr == pOptionsBuffer)
        {
            goto bail;
        }
        serializer.ToBuffer(pOptionsBuffer, optionsSize, pOptions, optionsCount);
        nn::sf::InBuffer optionsInBuffer(reinterpret_cast<const char*>(pOptionsBuffer), optionsSize);

        result = resolverClient->
            GetNameInfoRequestWithOptions(0,
                                          serializedSocketAddressIn,
                                          hostBufferOut,
                                          serviceBufferOut,
                                          static_cast<uint32_t>(flags),
                                          reinterpret_cast<int32_t*>(&returnValue),
                                          g_ResolverClientOptionsVersion,
                                          optionsInBuffer,
                                          optionsCount,
                                          nn::socket::GetHErrno(),
                                          &errno);
        if ( result.IsFailure() )
        {
            NN_SDK_ASSERT(false, "GetNameInfoRequest failure");
            goto bail;
        };
    };

bail:
    if (nullptr != pOptionsBuffer)
    {
        socket::detail::Free(pOptionsBuffer);
        pOptionsBuffer = nullptr;
    };

    resolverClient = nullptr;

    DecrementSocketAndSignalIfNeeded();

    return returnValue;
}

int GetNameInfoPrivate(const struct sockaddr* sa, socklen_t salen,
                       char* host, size_t hostlen,
                       char* serv, size_t servlen,
                       int flags,
                       const ResolverOption* pOptions,
                       size_t optionsCount) NN_NOEXCEPT
{
    Result result = ResultInternalError();
    nn::sf::SharedPointer<IResolver> resolverClient = nullptr;
    int32_t returnValue = 0;
    uint8_t* pOptionsBuffer = nullptr;

    if(!IncrementSocketIfNeeded())
    {
        LogDebug("\n");
        return EAI_SYSTEM;
    }
    else if ( (result = CreateResolverByHipc(resolverClient)).IsFailure() )
    {
        LogDebug("\n");
        returnValue = EAI_AGAIN;
        socket::SetLastError(socket::Errno::EAgain);
        goto bail;
    }
    else if ( sa == nullptr && salen != 0 )
    {
        returnValue = EAI_FAIL;
        LogDebug("nullptr struct sockaddr but nonzero length provided");
        goto bail;
    }
    else if (host == nullptr && hostlen != 0)
    {
        returnValue = EAI_FAIL;
        LogDebug("nullptr host buffer but nonzero length provided");
        goto bail;
    }
    else if (serv == nullptr && servlen != 0)
    {
        returnValue = EAI_FAIL;
        LogDebug("nullptr service buffer but nonzero length provided");
        goto bail;
    }
    else
    {
        serializer::DNSSerializer serializer;

        nn::sf::InBuffer serializedSocketAddressIn(reinterpret_cast<const char*>(sa),salen);

        nn::sf::OutBuffer hostBufferOut(host, hostlen);

        nn::sf::OutBuffer serviceBufferOut(serv, servlen);

        const size_t optionsSize = serializer.SizeOf(pOptions, optionsCount);
        pOptionsBuffer = reinterpret_cast<uint8_t*>(detail::Calloc(1, optionsSize));
        if (nullptr == pOptionsBuffer)
        {
            goto bail;
        }
        serializer.ToBuffer(pOptionsBuffer, optionsSize, pOptions, optionsCount);
        nn::sf::InBuffer optionsInBuffer(reinterpret_cast<const char*>(pOptionsBuffer), optionsSize);

        result = resolverClient->
            GetNameInfoRequestWithOptions(0,
                                          serializedSocketAddressIn,
                                          hostBufferOut,
                                          serviceBufferOut,
                                          flags,
                                          &returnValue,
                                          g_ResolverClientOptionsVersion,
                                          optionsInBuffer,
                                          optionsCount,
                                          nn::socket::GetHErrno(),
                                          &errno);
        if ( result.IsFailure() )
        {
            NN_SDK_ASSERT(false, "GetNameInfoRequest failure");
            goto bail;
        };
    };

bail:
    if (nullptr != pOptionsBuffer)
    {
        socket::detail::Free(pOptionsBuffer);
        pOptionsBuffer = nullptr;
    };

    resolverClient = nullptr;

    DecrementSocketAndSignalIfNeeded();

    return returnValue;
};
}  // anonymous namespace

Result InitializeClient() NN_NOEXCEPT
{
    Result result = ResultInternalError();

    RefCountHelperInitialize();

    if ( true == g_resolverClientInitialized )
    {
        result = ResultSuccess();
        goto bail;
    }
    else if ( (result = tls::Client::Initialize() ).IsFailure() )
    {
        NN_SDK_ASSERT(false, "Unable to initialize resolver client thread local storage\n");
        goto bail;
    };

    result = ResultSuccess();
    g_resolverClientInitialized = true;

bail:
    return result;
};

Result FinalizeClient() NN_NOEXCEPT
{
    Result result = ResultSuccess(); // TODO: make this a reasonable failure code

    RefCountHelperFinalize();

    if ( true == g_resolverClientInitialized )
    {
        return result;
    }
    else if ( (result = tls::Client::Finalize() ).IsFailure() )
    {
        NN_SDK_ASSERT(false, "Unable to finalize resolver client thread local storage\n");
        return result;
    };

    return ResultSuccess();
};

const char* GetHostStringError(uint32_t errorIn) NN_NOEXCEPT
{
    Result result = ResultInternalError();
    nn::sf::SharedPointer<IResolver> resolverClient = nullptr;
    const char* errorString = InfoFailureString;
    size_t bufferSize = 0;
    ResolverOption temp;
    uint8_t* pOutBuffer = nullptr;

    // This must come before atomic inc/dec to prevent false signals
    if(!IncrementSocketIfNeeded())
    {
        LogMajor("\n");
        return errorString;
    }
    else if ( -1 == ResolverGetOption(&temp, static_cast<ResolverOptionKey>(ResolverOptionLocalKey::GetHostErrorStringBufferSizeUnsigned64)))
    {
        LogMajor("\n");
        nn::socket::SetLastError(nn::socket::Errno::ENoSpc);
        goto bail;
    }
    else if ( 0 == (bufferSize = temp.data.unsigned64Value))
    {
        LogMajor("\n");
        nn::socket::SetLastError(nn::socket::Errno::ENoSpc);
        goto bail;
    }
    else if ( (result = CreateResolverByHipc(resolverClient)).IsFailure() )
    {
        LogMajor("\n");
        errorString = InfoRetryString;
        socket::SetLastError(nn::socket::Errno::EAgain);
        goto bail;
    }
    else
    {
        result = ResultInternalError();

        pOutBuffer = reinterpret_cast<uint8_t*>(detail::Calloc(1, bufferSize));
        if (nullptr == pOutBuffer)
        {
            LogDebug("\n");
            goto bail;
        }
        nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(pOutBuffer), bufferSize);

        result = resolverClient->GetHostStringErrorRequest(errorIn, outBuffer );
        if ( result.IsFailure() )
        {
            LogMajor("\n");
            NN_SDK_ASSERT(false, "GetHostStringErrorRequest failure");
            goto bail;
        };

        errorString = tls::Client::SetCurrentHErrorString(reinterpret_cast<const char*>(pOutBuffer));
        if (nullptr == errorString)
        {
            LogMajor("\n");
            errorString = InfoFailureString;
            socket::SetLastError(nn::socket::Errno::ENoMem);
        };
    };

bail:
    if (nullptr != pOutBuffer)
    {
        detail::Free(pOutBuffer);
        pOutBuffer = nullptr;
    };

    resolverClient = nullptr;

    DecrementSocketAndSignalIfNeeded();

    return errorString;
}

const char* GetGaiStringError(uint32_t errorIn) NN_NOEXCEPT
{
    Result result = ResultInternalError();
    nn::sf::SharedPointer<IResolver> resolverClient = nullptr;
    const char* errorString = InfoFailureString;
    size_t bufferSize = 0;
    ResolverOption temp;
    uint8_t* pOutBuffer = nullptr;

    // This must come before atomic inc/dec to prevent false signals
    if(!IncrementSocketIfNeeded())
    {
        LogDebug("\n");
        return errorString;
    }
    else if ( -1 == ResolverGetOption(&temp, static_cast<ResolverOptionKey>(ResolverOptionLocalKey::GaiErrorStringBufferSizeUnsigned64)))
    {
        LogMajor("\n");
        nn::socket::SetLastError(socket::Errno::ENoSpc);
        goto bail;
    }
    else if ( 0 == (bufferSize = temp.data.unsigned64Value))
    {
        LogMajor("\n");
        nn::socket::SetLastError(socket::Errno::ENoSpc);
        goto bail;
    }
    else if ( (result = CreateResolverByHipc(resolverClient)).IsFailure() )
    {
        LogMajor("\n");
        errorString = InfoRetryString;
        socket::SetLastError(socket::Errno::EAgain);
        goto bail;
    }
    else if (nullptr == (pOutBuffer = reinterpret_cast<uint8_t*>(detail::Calloc(1, bufferSize))))
    {
        LogMajor("\n");
        errorString = InfoRetryString;
        socket::SetLastError(socket::Errno::EAgain);
        goto bail;
    }
    else
    {
        nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(pOutBuffer), bufferSize);

        result = resolverClient->GetGaiStringErrorRequest(errorIn, outBuffer );
        if ( result.IsFailure() )
        {
            LogDebug("\n");
            NN_SDK_ASSERT(false, "GetGaiStringErrorRequest failure");
            goto bail;
        };

        errorString = tls::Client::SetCurrentGaiErrorString(reinterpret_cast<const char*>(pOutBuffer));
        if (nullptr == errorString)
        {
            LogDebug("\n");
            errorString = InfoFailureString;
            socket::SetLastError(socket::Errno::ENoMem);
        };
    };

bail:
    if (nullptr != pOutBuffer)
    {
        detail::Free(pOutBuffer);
        pOutBuffer = nullptr;
    };

    resolverClient = nullptr;

    DecrementSocketAndSignalIfNeeded();

    return errorString;
};

int ResolverGetOption(ResolverOption* pOption, ResolverOptionKey keyAsResolverOptionKey) NN_NOEXCEPT
{
    uint32_t key = static_cast<uint32_t>(keyAsResolverOptionKey);

    if (nullptr == pOption)
    {
        return -1;
    };

    ResolverOption & option = *pOption;

    if (key >= static_cast<uint32_t>(ResolverOptionLocalKey::MinimumValue) &&
        key <= static_cast<uint32_t>(ResolverOptionLocalKey::MaximumValue))
    {
        if (key == static_cast<uint32_t>(ResolverOptionLocalKey::MinimumValue) ||
            key == static_cast<uint32_t>(ResolverOptionLocalKey::MaximumValue))
        {
            errno = static_cast<int>(socket::Errno::EInval);
            return -1;
        };

        memset(&option, 0, sizeof(option));
        unsigned index = (key - static_cast<uint32_t>(ResolverOptionLocalKey::MinimumValue) - 1);
        std::lock_guard<os::Mutex> lock(g_LocalResolverOptionsLock);
        option.key = static_cast<ResolverOptionKey>(key);
        option.type = ResolverOptionType::Unsigned64;
        option.size = sizeof(uint64_t);
        option.data.unsigned64Value = g_LocalResolverOptionsTable[index];
        LogDebug("key: %s, type: %s, size: %zu, u64Value: %zu\n",
                 resolver::GetResolverOptionKeyString(option.key),
                 resolver::GetResolverOptionTypeString(option.type),
                 option.size,
                 option.data.unsigned64Value);
        return 0;
    }
    else if (static_cast<uint32_t>(ResolverOptionKey::GetCancelHandleInteger) == key)
    {
        CreateRandomId(option.data.integerValue);;
        option.size = sizeof(int);
        return 0;
    };

    static const size_t PointerOutMaxData = 255;
    static const size_t size = sizeof(option) + PointerOutMaxData;
    Result result = ResultInternalError();

    int rc = -1;
    nn::sf::SharedPointer<IResolver> resolverClient = nullptr;
    serializer::DNSSerializer serializer;
    uint8_t buffer[size];
    uint32_t bufferUsed = 0;
    size_t oldSize = 0;
    const char* oldPointer = nullptr;

    size_t inBufferSize = serializer.SizeOf(option);
    uint8_t* pInBytes = reinterpret_cast<uint8_t*>(detail::Calloc(1, inBufferSize));
    if (nullptr == pInBytes)
    {
        LogMajor("\n");
        socket::SetLastError(socket::Errno::EAgain);
        goto bail;
    }
    else
    {
        serializer.ToBuffer(pInBytes, inBufferSize, option);
        nn::sf::InBuffer inBuffer(reinterpret_cast<char*>(pInBytes), inBufferSize);
        nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(buffer), sizeof(buffer));

        if (!IncrementSocketIfNeeded())
        {
            LogDebug("\n");
            socket::SetLastError(socket::Errno::EInval);
            goto bail;
        }
        else if ((result = CreateResolverByHipc(resolverClient)).IsFailure() )
        {
            LogDebug("\n");
            socket::SetLastError(socket::Errno::EAgain);
        }
        else if ((result = resolverClient->ResolverGetOptionRequest(0, // pid
                                                                    1, // count
                                                                    static_cast<uint32_t>(key),
                                                                    inBuffer,
                                                                    outBuffer,
                                                                    &bufferUsed,
                                                                    &rc,
                                                                    &errno)).IsFailure())
        {
            LogMajor("\n");
            goto bail;
        }
        else if ( 0 == bufferUsed )
        {
            LogDebug("\n");
            goto bail;
        };

        // fixup and copy the pointer value from the buffer
        if (ResolverOptionType::Pointer == option.type)
        {
            oldPointer = option.data.pointerValue;
            oldSize = option.size;
        };

        if ( -1 == serializer.FromBuffer(option, buffer, bufferUsed))
        {
            LogDebug("\n");
            goto bail;
        };

        // continue fixup
        if ( nullptr != oldPointer )
        {
            size_t fixupsize = oldSize <= option.size ? oldSize : option.size;
            memcpy(const_cast<char*>(oldPointer),
                   const_cast<char*>(option.data.pointerValue),
                   fixupsize);
            option.size = fixupsize;
            option.data.pointerValue = oldPointer;
        };
    };

bail:
    if (nullptr != pInBytes)
    {
        detail::Free(pInBytes);
        pInBytes = nullptr;
    };

    resolverClient = nullptr;

    DecrementSocketAndSignalIfNeeded();

    return rc;
}; //NOLINT(impl/function_size)

int ResolverSetOption(const ResolverOption& option) NN_NOEXCEPT
{
    if (static_cast<uint32_t>(option.key) > static_cast<uint32_t>(ResolverOptionLocalKey::MinimumValue) &&
        static_cast<uint32_t>(option.key) < static_cast<uint32_t>(ResolverOptionLocalKey::MaximumValue) &&
        option.type == ResolverOptionType::Unsigned64 &&
        option.size == sizeof(uint64_t))
    {
        unsigned index = (static_cast<uint32_t>(option.key) -
                          static_cast<uint32_t>(ResolverOptionLocalKey::MinimumValue) - 1);
        std::lock_guard<os::Mutex> lock(g_LocalResolverOptionsLock);
        LogDebug("key: %s, type: %s, size: %zu, u64Value: %zu\n",
                 resolver::GetResolverOptionKeyString(option.key),
                 resolver::GetResolverOptionTypeString(option.type),
                 option.size,
                 option.data.unsigned64Value);
        g_LocalResolverOptionsTable[index] = option.data.unsigned64Value;
        return 0;
    };

    Result result = ResultInternalError();
    int rc = -1;
    nn::sf::SharedPointer<IResolver> resolverClient = nullptr;
    serializer::DNSSerializer serializer;
    size_t bufferSize = serializer.SizeOf(option);
    uint8_t* pBuffer = nullptr;

    if (nullptr == (pBuffer = reinterpret_cast<uint8_t*>(detail::Calloc(1, bufferSize))))
    {
        LogMajor("\n");
        socket::SetLastError(socket::Errno::ENoMem);
        goto bail;
    }
    else if (!IncrementSocketIfNeeded())
    {
        LogDebug("\n");
        socket::SetLastError(socket::Errno::EInval);
        goto bail;
    }
    else if (-1 == (rc = serializer.ToBuffer(pBuffer, bufferSize, option)))
    {
        LogMajor("\n");
        goto bail;
    }
    else if ((result = CreateResolverByHipc(resolverClient)).IsFailure())
    {
        LogMajor("\n");
        socket::SetLastError(socket::Errno::EAgain);
    }
    else
    {
        nn::sf::InBuffer inBuffer(reinterpret_cast<char*>(pBuffer), bufferSize);

        if ((result = resolverClient->ResolverSetOptionRequest(0, // pid
                                                               1, // count
                                                               inBuffer,
                                                               &rc,
                                                               &errno)).IsFailure())
        {
            LogMajor("\n");
            goto bail;
        };
    };

    rc = 0;

bail:
    if (nullptr != pBuffer)
    {
        detail::Free(pBuffer);
        pBuffer = nullptr;
    };

    resolverClient = nullptr;
    DecrementSocketAndSignalIfNeeded();
    return rc;
};

//--------------------------------------------------------------------
//  Entry Points
// -------------------------------------------------------------------

nn::socket::HostEnt* GetHostEntByName(const char* pHostname) NN_NOEXCEPT
{
    return GetHostEntByNamePrivate(pHostname, nullptr, 0);
}

struct hostent* GetHostByName(const char* pHostname) NN_NOEXCEPT
{
    return GetHostByNamePrivate(pHostname, nullptr, 0);
};

nn::socket::HostEnt* GetHostEntByName(const char* pHostname,
                                   const ResolverOption* options,
                                   size_t optionsCount) NN_NOEXCEPT
{
    return GetHostEntByNamePrivate(pHostname, options, optionsCount);
}

struct hostent* GetHostByName(const char* pHostname,
                              const ResolverOption* options,
                              size_t optionsCount) NN_NOEXCEPT
{
    return GetHostByNamePrivate(pHostname, options, optionsCount);
};

nn::socket::HostEnt* GetHostEntByAddr(const void* pInaddr,
                                      uint32_t socketLengthIn,
                                      nn::socket::Family addressFamilyIn) NN_NOEXCEPT
{
    return GetHostEntByAddrPrivate(pInaddr, socketLengthIn, addressFamilyIn, nullptr, 0);
}

struct hostent* GetHostByAddr(const void* pInaddr,
                              uint32_t socketLengthIn,
                              uint32_t addressFamilyIn) NN_NOEXCEPT
{
    return GetHostByAddrPrivate(pInaddr, socketLengthIn, addressFamilyIn, nullptr, 0);
};

nn::socket::HostEnt* GetHostEntByAddr(const void* pInaddr,
                                      uint32_t socketLengthIn,
                                      nn::socket::Family addressFamilyIn,
                                      const ResolverOption* options,
                                      size_t optionsCount) NN_NOEXCEPT
{
    return GetHostEntByAddrPrivate(pInaddr, socketLengthIn, addressFamilyIn, options, optionsCount);
}

struct hostent* GetHostByAddr(const void* pInaddr,
                              uint32_t socketLengthIn,
                              uint32_t addressFamilyIn,
                              const ResolverOption* options,
                              size_t optionsCount) NN_NOEXCEPT
{
    return GetHostByAddrPrivate(pInaddr, socketLengthIn, addressFamilyIn, options, optionsCount);
};

nn::socket::AiErrno GetAddrInfo(const char* node,
                                const char* service,
                                const nn::socket::AddrInfo* hints,
                                nn::socket::AddrInfo** ppAddrinfoOut) NN_NOEXCEPT
{

    return GetAddrInfoPrivate(node, service, hints, ppAddrinfoOut, NULL, 0);
}

int GetAddrInfo(const char* node,
                const char* service,
                const struct addrinfo* hints,
                struct addrinfo** ppAddrinfoOut) NN_NOEXCEPT
{

    return GetAddrInfoPrivate(node, service, hints, ppAddrinfoOut, nullptr, 0);
};

nn::socket::AiErrno GetAddrInfo(const char* node,
                                const char* service,
                                const nn::socket::AddrInfo* hints,
                                nn::socket::AddrInfo** ppAddrinfoOut,
                                const ResolverOption* options,
                                size_t optionsCount) NN_NOEXCEPT
{
    return GetAddrInfoPrivate(node, service, hints, ppAddrinfoOut, options, optionsCount);
}

int GetAddrInfo(const char* node,
                const char* service,
                const struct addrinfo* hints,
                struct addrinfo** ppAddrinfoOut,
                const ResolverOption* options,
                size_t optionsCount) NN_NOEXCEPT
{
    return GetAddrInfoPrivate(node, service, hints, ppAddrinfoOut, options, optionsCount);
};

nn::socket::AiErrno GetNameInfo(const nn::socket::SockAddr* sa, nn::socket::SockLenT salen,
                                char* host, size_t hostlen,
                                char* serv, size_t servlen,
                                nn::socket::NameInfoFlag flags) NN_NOEXCEPT
{
    return GetNameInfoPrivate(sa, salen, host, hostlen, serv, servlen, flags, nullptr, 0);
}

int GetNameInfo(const struct sockaddr* sa, socklen_t salen,
                char* host, size_t hostlen,
                char* serv, size_t servlen,
                int flags) NN_NOEXCEPT
{
    return GetNameInfoPrivate(sa, salen, host, hostlen, serv, servlen, flags, nullptr, 0);
};

nn::socket::AiErrno GetNameInfo(const nn::socket::SockAddr* sa, nn::socket::SockLenT salen,
                                char* host, size_t hostlen,
                                char* serv, size_t servlen,
                                nn::socket::NameInfoFlag flags,
                                const ResolverOption* options,
                                size_t optionsCount) NN_NOEXCEPT
{
    return GetNameInfoPrivate(sa, salen, host, hostlen, serv, servlen, flags, options, optionsCount);
}

int GetNameInfo(const struct sockaddr* sa, socklen_t salen,
                char* host, size_t hostlen,
                char* serv, size_t servlen,
                int flags,
                const ResolverOption* options,
                size_t optionsCount) NN_NOEXCEPT
{
    return GetNameInfoPrivate(sa, salen, host, hostlen, serv, servlen, flags, options, optionsCount);
};

} } } // namespace socket::resolver
