﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sf_HipcEmulatedNamedPipeSession.h"

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/hipc/detail/sf_HipcMessageBufferAccessor2.h>

#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>

#include "sf_HipcEmulatedNamedPipePort.h"
#include "sf_HipcEmulatedSessionRequest.h"
#include "detail/sf_HipcHandleRegistrationInternal.h"
#include "detail/sf_HipcWindowsNamedPipe.h"
#include "detail/sf_HipcWindowsThreadpoolWork.h"


#define NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(exp) \
    do { if (exp); else return false; } while (NN_STATIC_CONDITION(false))


namespace nn { namespace sf { namespace hipc {

namespace {

enum MapTransferType
{
    MapTransferType_Send,
    MapTransferType_Receive,
    MapTransferType_Exchange,
};

const int ReceiveListMax = (1 << detail::HipcFormat::ReceiveListCount::Width) - 1;
const int PointerTransferMax = (1 << detail::HipcFormat::PointerCount::Width) - 1;
const int MapTransferSendMax = (1 << detail::HipcFormat::SendCount::Width) - 1;
const int MapTransferReceiveMax = (1 << detail::HipcFormat::ReceiveCount::Width) - 1;
const int MapTransferExchangeMax = (1 << detail::HipcFormat::ExchangeCount::Width) - 1;
const int MapTransferMax = MapTransferSendMax + MapTransferReceiveMax + MapTransferExchangeMax;

struct MapTransferEmulationData
{
    Bit32 sendHandles[MapTransferSendMax];
    Bit32 receiveHandles[MapTransferReceiveMax];
    Bit32 exchangeHandles[MapTransferExchangeMax];
};


Bit32 ReinterpretAsTransferableValue(HANDLE handle) NN_NOEXCEPT
{
#ifdef NN_BUILD_CONFIG_ADDRESS_64
    return reinterpret_cast<uint64_t>(handle) & 0xffffffff;
#else
    return reinterpret_cast<uint32_t>(handle);
#endif
}

HANDLE ReinterpretAsNativeHandle(Bit32 value) NN_NOEXCEPT
{
#ifdef NN_BUILD_CONFIG_ADDRESS_64
    if (value & 0x80000000)
    {
        return reinterpret_cast<HANDLE>(static_cast<uint64_t>(value) | 0xffffffff00000000);
    }
    else
    {
        return reinterpret_cast<HANDLE>(static_cast<uint64_t>(value));
    }
#else
    return reinterpret_cast<HANDLE>(value);
#endif
}

detail::InternalHandleValue ReinterpretAsInternalHandleValue(os::NativeHandle handle) NN_NOEXCEPT
{
    detail::InternalHandleValue ret = { ReinterpretAsTransferableValue(handle) };
    return ret;
}

os::NativeHandle ReinterpretAsNativeHandle(detail::InternalHandleValue key) NN_NOEXCEPT
{
    return ReinterpretAsNativeHandle(key.value);
}

detail::InternalHandleValue MarkAsClientSessionHandle(detail::InternalHandleValue key) NN_NOEXCEPT
{
    // Windows ハンドル値の下位 2bit が使用されていないことを利用して ClientSession かどうか区別できるようにする
    NN_ABORT_UNLESS((key.value & 0x3) == 0);
    detail::InternalHandleValue ret = { key.value | 0x3 };
    return ret;
}

detail::InternalHandleValue UnmarkAsClientSessionHandle(detail::InternalHandleValue key) NN_NOEXCEPT
{
    NN_ABORT_UNLESS((key.value & 0x3) == 0x3);
    detail::InternalHandleValue ret = { key.value & ~0x3 };
    return ret;
}

bool IsMarkedAsClientSessionHandle(detail::InternalHandleValue key) NN_NOEXCEPT
{
    return (key.value & 0x3) == 0x3;
}

HANDLE DuplicateInternalHandle(HANDLE sourceHandle, DWORD targetProcessId, bool move) NN_NOEXCEPT
{
    HANDLE targetHandle;

    HANDLE targetProcessHandle = ::OpenProcess(PROCESS_DUP_HANDLE, FALSE, targetProcessId);
    NN_ABORT_UNLESS(
        targetProcessHandle,
        "OpenProcess failed with an unexpected error %lu", ::GetLastError());

    DWORD options = DUPLICATE_SAME_ACCESS | ((move) ? DUPLICATE_CLOSE_SOURCE : 0);
    NN_ABORT_UNLESS(
        ::DuplicateHandle(
            ::GetCurrentProcess(),
            sourceHandle,
            targetProcessHandle,
            &targetHandle,
            0,
            FALSE,
            options),
        "DuplicateHandle failed with an unexpected error %lu", ::GetLastError());

    NN_ABORT_UNLESS(
        ::CloseHandle(targetProcessHandle),
        "CloseHandle failed with an unexpected error %lu", ::GetLastError());

    return targetHandle;
}

HANDLE DuplicateExternalHandle(HANDLE sourceHandle, DWORD sourceProcessId, bool move) NN_NOEXCEPT
{
    HANDLE targetHandle;

    HANDLE sourceProcessHandle = ::OpenProcess(PROCESS_DUP_HANDLE, FALSE, sourceProcessId);
    NN_ABORT_UNLESS(
        sourceProcessHandle,
        "OpenProcess failed with an unexpected error %lu", ::GetLastError());

    DWORD options = DUPLICATE_SAME_ACCESS | ((move) ? DUPLICATE_CLOSE_SOURCE : 0);
    NN_ABORT_UNLESS(
        ::DuplicateHandle(
            sourceProcessHandle,
            sourceHandle,
            ::GetCurrentProcess(),
            &targetHandle,
            0,
            FALSE,
            options),
        "DuplicateHandle failed with an unexpected error %lu", ::GetLastError());

    NN_ABORT_UNLESS(
        ::CloseHandle(sourceProcessHandle),
        "CloseHandle failed with an unexpected error %lu", ::GetLastError());

    return targetHandle;
}

detail::InternalHandleValue ConvertInternalHandleValueToTransferableHandleValue(
    detail::InternalHandleValue internalHandleValue, bool duplicateHandle, DWORD clientProcessId, bool move) NN_NOEXCEPT
{
    if (internalHandleValue == detail::InvalidInternalHandleValue)
    {
        return detail::InvalidInternalHandleValue;
    }

    detail::InternalHandleType internalHandleType = detail::GetHandleType(internalHandleValue);

    os::NativeHandle transferableHandle;
    switch (internalHandleType)
    {
    case detail::InternalHandleType_OsHandle:
        transferableHandle = detail::UnregisterOsHandle(internalHandleValue);
        break;
    case detail::InternalHandleType_SessionHandle:
        if (move)
        {
            HipcEmulatedNamedPipeClientSession* pClientSession = static_cast<HipcEmulatedNamedPipeClientSession*>(detail::UnregisterSessionHandle(internalHandleValue));

            detail::HipcWindowsNamedPipe* pNamedPipe = pClientSession->DetachNamedPipe();
            pClientSession->Release();

            transferableHandle = pNamedPipe->DetachPipeHandle();
            pNamedPipe->Release();
        }
        else
        {
            transferableHandle = static_cast<HipcEmulatedNamedPipeClientSession*>(detail::UnregisterSessionHandle(internalHandleValue))->GetNamedPipe()->GetPipeHandle();
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    if (duplicateHandle)
    {
        transferableHandle = DuplicateInternalHandle(transferableHandle, clientProcessId, move);
    }

    switch (internalHandleType)
    {
    case detail::InternalHandleType_OsHandle:
        return ReinterpretAsInternalHandleValue(transferableHandle);
    case detail::InternalHandleType_SessionHandle:
        return MarkAsClientSessionHandle(ReinterpretAsInternalHandleValue(transferableHandle));
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void EmulateHipcSpecialDataTransferBeforeSend(
    const detail::HipcMessageReader& reader,
    detail::HipcMessageWriter& writer,
    bool duplicateHandle,
    DWORD clientProcessId) NN_NOEXCEPT
{
    const detail::HipcMessageHeaderInfo& headerInfo = reader.GetHeaderInfo().baseInfo;

    if (headerInfo.hasPid)
    {
        writer.SetProcessId(::GetCurrentProcessId());
    }

    for (int i = 0; i < headerInfo.copyHandleCount; i++)
    {
        writer.SetCopyHandle(i,
            ConvertInternalHandleValueToTransferableHandleValue(
                reader.GetCopyHandle(i), duplicateHandle, clientProcessId, false));
    }
    for (int i = 0; i < headerInfo.moveHandleCount; i++)
    {
        writer.SetMoveHandle(i,
            ConvertInternalHandleValueToTransferableHandleValue(
                reader.GetMoveHandle(i), duplicateHandle, clientProcessId, true));
    }
}

detail::InternalHandleValue ConvertTransferableHandleValueToInternalHandleValue(
    detail::InternalHandleValue internalHandleValue, bool duplicateHandle, DWORD clientProcessId, bool move) NN_NOEXCEPT
{
    if (internalHandleValue == detail::InvalidInternalHandleValue)
    {
        return detail::InvalidInternalHandleValue;
    }

    bool isSessionHandle = IsMarkedAsClientSessionHandle(internalHandleValue);

    os::NativeHandle transferableHandle;
    if (isSessionHandle)
    {
        transferableHandle = ReinterpretAsNativeHandle(UnmarkAsClientSessionHandle(internalHandleValue));
    }
    else
    {
        transferableHandle = ReinterpretAsNativeHandle(internalHandleValue);
    }

    if (duplicateHandle)
    {
        transferableHandle = DuplicateExternalHandle(transferableHandle, clientProcessId, move);
    }

    detail::InternalHandleValue key;
    if (isSessionHandle)
    {
        detail::HipcWindowsNamedPipe* pNamedPipe = detail::HipcWindowsNamedPipe::Create();
        pNamedPipe->Connect(transferableHandle);

        HipcEmulatedNamedPipeClientSession* pClientSession = HipcEmulatedNamedPipeClientSession::Create(pNamedPipe);

        key = detail::RegisterSessionHandle(pClientSession);
    }
    else
    {
        key = detail::RegisterOsHandle(transferableHandle);
    }
    return key;
}

void EmulateHipcSpecialDataTransferAfterReceive(
    const detail::HipcMessageReader& reader,
    detail::HipcMessageWriter& writer,
    bool duplicateHandle,
    DWORD clientProcessId) NN_NOEXCEPT
{
    const detail::HipcMessageHeaderInfo& headerInfo = reader.GetHeaderInfo().baseInfo;

    for (int i = 0; i < headerInfo.copyHandleCount; i++)
    {
        writer.SetCopyHandle(i,
            ConvertTransferableHandleValueToInternalHandleValue(
                reader.GetCopyHandle(i), duplicateHandle, clientProcessId, false));
    }
    for (int i = 0; i < headerInfo.moveHandleCount; i++)
    {
        writer.SetMoveHandle(i,
            ConvertTransferableHandleValueToInternalHandleValue(
                reader.GetMoveHandle(i), duplicateHandle, clientProcessId, true));
    }
}

void CopyReceiveList(
    sf::detail::PointerAndSize* pReceiveList,
    int receiveListMax,
    int* pOutReceiveListCount,
    const detail::HipcMessageReader& reader)
{
    const detail::HipcMessageHeaderInfo& headerInfo = reader.GetHeaderInfo().baseInfo;
    NN_SDK_ASSERT(receiveListMax >= headerInfo.receiveListCount);

    *pOutReceiveListCount = 0;
    for (int i = 0; i < headerInfo.receiveListCount && i < receiveListMax; i++)
    {
        detail::AddressValueAndSize avas = reader.GetReceiveList(i);
        pReceiveList[i].pointer = detail::UnregisterAddress(avas.pointer);
        pReceiveList[i].size = avas.size;
        (*pOutReceiveListCount)++;
    }
}

void CreateFileMappingViewFromClient(
    HANDLE* pOutHandle,
    void** ppOutView,
    uint64_t fileSize,
    MapTransferType mapTransferType) NN_NOEXCEPT
{
    NN_SDK_ASSERT(fileSize > 0);  // CreateFileMapping の仕様

    HANDLE fileMappingObjectHandle = ::CreateFileMapping(
        INVALID_HANDLE_VALUE,
        NULL,
        PAGE_READWRITE,
        fileSize >> 32,
        fileSize & 0xffffffff,
        NULL);

    NN_ABORT_UNLESS(
        fileMappingObjectHandle,
        "CreateFileMapping failed with an unexpected error %lu", ::GetLastError());

    DWORD desiredAccess;
    switch (mapTransferType)
    {
    case MapTransferType_Send:
        desiredAccess = FILE_MAP_WRITE;  // 実際には WRITE に READ 権限も含まれる。純粋な READ は指定不可
        break;
    case MapTransferType_Receive:
        desiredAccess = FILE_MAP_READ;
        break;
    case MapTransferType_Exchange:
        desiredAccess = FILE_MAP_READ | FILE_MAP_WRITE;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    LPVOID pView = ::MapViewOfFile(fileMappingObjectHandle, desiredAccess, 0, 0, 0);
    NN_ABORT_UNLESS(pView, "MapViewOfFile failed with an unexpected error %lu", ::GetLastError());

    *pOutHandle = fileMappingObjectHandle;
    *ppOutView = pView;
}

void OpenFileMappingViewFromServer(
    void** ppOutView,
    HANDLE fileMappingObjectHandle,
    MapTransferType mapTransferType) NN_NOEXCEPT
{
    DWORD desiredAccess;
    switch (mapTransferType)
    {
    case MapTransferType_Send:
        desiredAccess = FILE_MAP_READ;  // クライアントが送るので、サーバ側で必要な権限は READ
        break;
    case MapTransferType_Receive:
        desiredAccess = FILE_MAP_WRITE;  // クライアントが受けるので、サーバ側で必要な権限は WRITE
        break;
    case MapTransferType_Exchange:
        desiredAccess = FILE_MAP_READ | FILE_MAP_WRITE;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    LPVOID pView = ::MapViewOfFile(fileMappingObjectHandle, desiredAccess, 0, 0, 0);
    NN_ABORT_UNLESS(pView, "MapViewOfFile failed with an unexpected error %lu", ::GetLastError());

    *ppOutView = pView;
}

void CloseFileMappingView(HANDLE fileMappingObjectHandle, void* pView) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(
        ::UnmapViewOfFile(pView),
        "UnmapViewOfFile failed with an unexpected error %lu", ::GetLastError());
    NN_ABORT_UNLESS(
        ::CloseHandle(fileMappingObjectHandle),
        "CloseHandle failed with an unexpected error %lu", ::GetLastError());
}


bool ReceivePointerTransferData_ReceiveIntoConsecutiveBuffers(
    uintptr_t receiveStartAddress,
    size_t bufferSize,
    detail::HipcWindowsNamedPipe* pNamedPipe,
    const detail::HipcMessageReader& reader,
    detail::HipcMessageWriter& writer)
{
    const int pointerCount = reader.GetHeaderInfo().baseInfo.pointerCount;

    // 1 つのバッファにすべてのポインタ転送を受信する場合、データはインデックスの順番に格納していかないといけない
    // また、各転送の先頭アドレスは 16byte アラインメントされていないといけない
    // データがインデックスの順に届くとは限らないので、先に各データの先頭アドレスを決めることにする
    uintptr_t dataAddress[ReceiveListMax];
    std::memset(dataAddress, 0, sizeof(dataAddress));
    {
        uintptr_t p = receiveStartAddress;

        size_t dataSize[ReceiveListMax];
        std::memset(dataSize, 0, sizeof(dataSize));

        for (int i = 0; i < pointerCount; i++)
        {
            int index = reader.GetPointerReceiveIndex(i);
            dataSize[index] = reader.GetPointer(i).size;
        }
        for (int i = 0; i < ReceiveListMax; i++)
        {
            p = util::align_up(p, 16);
            NN_ABORT_UNLESS(p < receiveStartAddress + bufferSize);
            dataAddress[i] = p;
            p += dataSize[i];
        }
    }

    for (int i = 0; i < pointerCount; i++)
    {
        int index = reader.GetPointerReceiveIndex(i);
        size_t size = reader.GetPointer(i).size;

        NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(
            pNamedPipe->Read(reinterpret_cast<void*>(dataAddress[index]), size));

        writer.SetPointer(i, detail::RegisterAddress(dataAddress[index]), size);
    }

    return true;
}

bool ReceivePointerTransferData(
    void* pMessageBuffer,
    size_t bufferSize,
    sf::detail::PointerAndSize* receiveList,
    int receiveListCount,
    detail::HipcMessageReceiveBufferMode receiveMode,
    detail::HipcWindowsNamedPipe* pNamedPipe,
    const detail::HipcMessageReader& reader,
    detail::HipcMessageWriter& writer) NN_NOEXCEPT
{
    const detail::HipcMessageHeaderInfo& headerInfo = reader.GetHeaderInfo().baseInfo;

    NN_ABORT_UNLESS(headerInfo.pointerCount >= 0);
    if (headerInfo.pointerCount == 0)
    {
        return true;
    }

    switch (receiveMode)
    {
    case detail::HipcMessageReceiveBufferMode_None:
    {
        NN_ABORT("unexpected");
    }
    case detail::HipcMessageReceiveBufferMode_MessageBuffer:
    {
        uintptr_t receiveStartAddress = reinterpret_cast<uintptr_t>(pMessageBuffer) + reader.GetMessageByteSize();
        size_t remainingBufferSize = bufferSize - reader.GetMessageByteSize();
        NN_ABORT_UNLESS(remainingBufferSize < bufferSize);

        NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(
            ReceivePointerTransferData_ReceiveIntoConsecutiveBuffers(
                receiveStartAddress, remainingBufferSize, pNamedPipe, reader, writer));
        break;
    }
    case detail::HipcMessageReceiveBufferMode_Single:
    {
        NN_ABORT_UNLESS(receiveListCount == 1);
        uintptr_t receiveStartAddress = receiveList[0].pointer;
        size_t remainingBufferSize = receiveList[0].size;

        NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(
            ReceivePointerTransferData_ReceiveIntoConsecutiveBuffers(
                receiveStartAddress, remainingBufferSize, pNamedPipe, reader, writer));
        break;
    }
    case detail::HipcMessageReceiveBufferMode_Multi:
    {
        for (int i = 0; i < headerInfo.pointerCount; i++)
        {
            int receiveListIndex = reader.GetPointerReceiveIndex(i);
            NN_ABORT_UNLESS(receiveListIndex >= 0 && receiveListIndex < receiveListCount);

            uintptr_t pointer = receiveList[receiveListIndex].pointer;
            size_t size = receiveList[receiveListIndex].size;

            NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(
                pNamedPipe->Read(reinterpret_cast<void*>(pointer), size));

            writer.SetPointer(i, detail::RegisterAddress(pointer), size);
        }
        break;
    }
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return true;
}

}  // namespace unnamed


class HipcEmulatedNamedPipeSessionRequest
    : public HipcEmulatedObjectBase
    , public HipcEmulatedSessionRequest
{
private:
    struct FileMappingView
    {
        MapTransferType mapTransferType;
        HANDLE fileMappingObjectHandle;
        void* pBuffer;
        size_t bufferSize;
        void* pView;
    };

private:
    SharedPointer<detail::HipcWindowsNamedPipe> m_pNamedPipe;

    // 受信指定リスト
    detail::HipcMessageReceiveBufferMode m_ReceiveBufferMode;
    int m_ReceiveListCount;
    sf::detail::PointerAndSize m_ReceiveList[ReceiveListMax];

    // メッセージ転送のリクエスト
    detail::HipcWindowsNamedPipeAsyncRequest* m_pSendMessageRequest;

    // ポインタ転送のリクエスト
    int m_SendPointerRequestCount;
    detail::HipcWindowsNamedPipeAsyncRequest* m_pSendPointerRequest[PointerTransferMax];

    // マップ転送のリクエストと使用中のハンドルのリスト
    detail::HipcWindowsNamedPipeAsyncRequest* m_pSendMapRequest;
    int m_FileMappingViewCount;
    FileMappingView m_FileMappingView[MapTransferMax];
    MapTransferEmulationData m_SendMapData;

    // サーバの応答受信のリクエストと受信先バッファ
    detail::HipcWindowsNamedPipeAsyncRequest* m_pReceiveMessageRequest;
    void* m_pReceiveMessageBuffer;
    size_t m_ReceiveMessageBufferSize;

protected:
    explicit HipcEmulatedNamedPipeSessionRequest(SharedPointer<detail::HipcWindowsNamedPipe> pNamedPipe) NN_NOEXCEPT
        : m_pNamedPipe(std::move(pNamedPipe))
        , m_ReceiveBufferMode(detail::HipcMessageReceiveBufferMode_None)
        , m_ReceiveListCount(0)
        , m_pSendMessageRequest(nullptr)
        , m_SendPointerRequestCount(0)
        , m_pSendMapRequest(nullptr)
        , m_FileMappingViewCount(0)
        , m_pReceiveMessageRequest(nullptr)
        , m_pReceiveMessageBuffer(nullptr)
        , m_ReceiveMessageBufferSize(0)
    {
        for (int i = 0; i < PointerTransferMax; i++)
        {
            m_pSendPointerRequest[i] = nullptr;
        }
    }

    ~HipcEmulatedNamedPipeSessionRequest() NN_NOEXCEPT
    {
        CancelAllAsyncRequest();
    }

public:
    bool SendClientRequest(void* clientMessageBuffer, size_t clientMessageBufferSize) NN_NOEXCEPT
    {
        SendClientRequest_CheckNoOutstandingRequest();

        bool functionSuccess = false;
        NN_UTIL_SCOPE_EXIT
        {
            if (!functionSuccess)
            {
                CancelAllAsyncRequest();
            }
        };

        detail::HipcMessageReader hipcMessageReader;
        hipcMessageReader.Initialize(clientMessageBuffer);
        const detail::HipcMessageHeaderInfo& headerInfo = hipcMessageReader.GetHeaderInfo().baseInfo;
        const detail::HipcMessageDataInfo dataInfo = detail::MakeHipcMessageDataInfo(headerInfo);
        detail::HipcMessageWriter hipcMessageWriter(dataInfo, clientMessageBuffer, clientMessageBufferSize);

        // プロセス ID 値とハンドル値を本物の値に置き換える
        EmulateHipcSpecialDataTransferBeforeSend(hipcMessageReader, hipcMessageWriter, false, 0);

        // 受信指定リストを保存
        m_ReceiveBufferMode = headerInfo.receiveBufferMode;
        CopyReceiveList(m_ReceiveList, ReceiveListMax, &m_ReceiveListCount, hipcMessageReader);

        // HIPC メッセージ送信
        m_pSendMessageRequest = m_pNamedPipe->WriteAsync(clientMessageBuffer, hipcMessageReader.GetMessageByteSize());
        NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(m_pSendMessageRequest);

        // ポインタ転送を含むなら、ポインタごとに送信
        NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(
            SendClientRequest_SendPointerTransferData(hipcMessageReader));

        // マップ転送を含むなら、ファイルマッピングオブジェクトハンドルを送信
        NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(
            SendClientRequest_SendMapTransferData(hipcMessageReader));

        // 受信処理を非同期で開始
        // サーバはクライアントが送信するデータをすべて受信し終えてから応答を送信する前提で、クライアントのデータ送信完了は待たない
        m_pReceiveMessageRequest = m_pNamedPipe->ReadAsync(clientMessageBuffer, clientMessageBufferSize);
        NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(m_pReceiveMessageRequest);
        m_pReceiveMessageBuffer = clientMessageBuffer;
        m_ReceiveMessageBufferSize = clientMessageBufferSize;

        functionSuccess = true;
        return true;
    }

    virtual void AttachReplyEvent(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT_UNLESS(m_pReceiveMessageRequest);
        m_pNamedPipe->AttachAsyncRequestCompletionEvent(pHolder, m_pReceiveMessageRequest);
    }

    virtual bool WaitRequest() NN_NOEXCEPT NN_OVERRIDE
    {
        // すべての AsyncRequest の完了を待つ
        {
            NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(GetAsyncRequestResult(&m_pSendMessageRequest));

            for (int i = 0; i < m_SendPointerRequestCount; i++)
            {
                NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(GetAsyncRequestResult(&m_pSendPointerRequest[i]));
            }
            m_SendPointerRequestCount = 0;

            NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(GetAsyncRequestResult(&m_pSendMapRequest));

            NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(GetAsyncRequestResult(&m_pReceiveMessageRequest));
        }

        detail::HipcMessageReader hipcMessageReader;
        hipcMessageReader.Initialize(m_pReceiveMessageBuffer);
        const detail::HipcMessageHeaderInfo& headerInfo = hipcMessageReader.GetHeaderInfo().baseInfo;
        const detail::HipcMessageDataInfo dataInfo = detail::MakeHipcMessageDataInfo(headerInfo);
        detail::HipcMessageWriter hipcMessageWriter(dataInfo, m_pReceiveMessageBuffer, m_ReceiveMessageBufferSize);

        // ハンドル値を内部値に置き換える
        EmulateHipcSpecialDataTransferAfterReceive(hipcMessageReader, hipcMessageWriter, false, 0);

        // ポインタ転送を含むなら、保存しておいた受信指定リストに従ってデータを受信
        NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(
            ReceivePointerTransferData(
                m_pReceiveMessageBuffer,
                m_ReceiveMessageBufferSize,
                m_ReceiveList,
                m_ReceiveListCount,
                m_ReceiveBufferMode,
                m_pNamedPipe.Get(),
                hipcMessageReader,
                hipcMessageWriter));

        // マップ転送を含むなら、ファイルマッピングオブジェクトハンドルを破棄
        for (int i = 0; i < m_FileMappingViewCount; i++)
        {
            switch (m_FileMappingView[i].mapTransferType)
            {
            case MapTransferType_Send:
            {
                // memcpy は不要

                CloseFileMappingView(m_FileMappingView[i].fileMappingObjectHandle, m_FileMappingView[i].pView);

                break;
            }
            case MapTransferType_Receive:
            case MapTransferType_Exchange:
            {
                std::memcpy(m_FileMappingView[i].pBuffer, m_FileMappingView[i].pView, m_FileMappingView[i].bufferSize);

                CloseFileMappingView(m_FileMappingView[i].fileMappingObjectHandle, m_FileMappingView[i].pView);

                break;
            }
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
        m_FileMappingViewCount = 0;

        return true;
    }

    virtual void CloseRequest() NN_NOEXCEPT NN_OVERRIDE
    {
        this->Release();
    }

private:
    void SendClientRequest_CheckNoOutstandingRequest() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_pReceiveMessageRequest == nullptr);
        NN_ABORT_UNLESS(m_pSendMapRequest == nullptr);
        for (int i = 0; i < PointerTransferMax; i++)
        {
            NN_ABORT_UNLESS(m_pSendPointerRequest[i] == nullptr);
        }
        NN_ABORT_UNLESS(m_pSendMessageRequest == nullptr);
    }

    bool SendClientRequest_SendPointerTransferData(const detail::HipcMessageReader& reader) NN_NOEXCEPT
    {
        const detail::HipcMessageHeaderInfo& headerInfo = reader.GetHeaderInfo().baseInfo;

        m_SendPointerRequestCount = 0;
        for (int i = 0; i < headerInfo.pointerCount; i++)
        {
            detail::AddressValueAndSize avas = reader.GetPointer(i);
            uintptr_t pointer = detail::UnregisterAddress(avas.pointer);

            m_pSendPointerRequest[m_SendPointerRequestCount] = m_pNamedPipe->WriteAsync(reinterpret_cast<void*>(pointer), avas.size);
            NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(m_pSendPointerRequest[m_SendPointerRequestCount]);
            m_SendPointerRequestCount++;
        }

        return true;
    }

    bool SendClientRequest_SendMapTransferData(const detail::HipcMessageReader& reader) NN_NOEXCEPT
    {
        const detail::HipcMessageHeaderInfo& headerInfo = reader.GetHeaderInfo().baseInfo;

        m_FileMappingViewCount = 0;
        std::memset(&m_SendMapData, 0, sizeof(m_SendMapData));
        for (int i = 0; i < headerInfo.sendCount; i++)
        {
            detail::MapTransferBufferInfo bufferInfo = reader.GetSend(i);

            bool viewCreated = SendClientRequest_CreateFileMappingView(
                &m_FileMappingView[m_FileMappingViewCount],
                MapTransferType_Send,
                bufferInfo);

            if (viewCreated)
            {
                m_SendMapData.sendHandles[i]
                    = ReinterpretAsTransferableValue(m_FileMappingView[m_FileMappingViewCount].fileMappingObjectHandle);
                m_FileMappingViewCount++;
            }
        }
        for (int i = 0; i < headerInfo.receiveCount; i++)
        {
            detail::MapTransferBufferInfo bufferInfo = reader.GetReceive(i);

            bool viewCreated = SendClientRequest_CreateFileMappingView(
                &m_FileMappingView[m_FileMappingViewCount],
                MapTransferType_Receive,
                bufferInfo);

            if (viewCreated)
            {
                m_SendMapData.receiveHandles[i]
                    = ReinterpretAsTransferableValue(m_FileMappingView[m_FileMappingViewCount].fileMappingObjectHandle);
                m_FileMappingViewCount++;
            }
        }
        for (int i = 0; i < headerInfo.exchangeCount; i++)
        {
            detail::MapTransferBufferInfo bufferInfo = reader.GetExchange(i);

            bool viewCreated = SendClientRequest_CreateFileMappingView(
                &m_FileMappingView[m_FileMappingViewCount],
                MapTransferType_Exchange,
                bufferInfo);

            if (viewCreated)
            {
                m_SendMapData.exchangeHandles[i]
                    = ReinterpretAsTransferableValue(m_FileMappingView[m_FileMappingViewCount].fileMappingObjectHandle);
                m_FileMappingViewCount++;
            }
        }
        m_pSendMapRequest = m_pNamedPipe->WriteAsync(&m_SendMapData, sizeof(m_SendMapData));
        NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(m_pSendMapRequest);

        return true;
    }

    bool SendClientRequest_CreateFileMappingView(
        FileMappingView* pOutFileMappingView,
        MapTransferType mapTransferType,
        const detail::MapTransferBufferInfo& bufferInfo) NN_NOEXCEPT
    {
        void* pointer = reinterpret_cast<void*>(detail::UnregisterAddress(bufferInfo.pointer));
        size_t size = bufferInfo.size;

        if (size == 0)
        {
            // Windows の仕様で長さ 0 のファイルに対してマッピングすることはできない。長さ 0 なら何もしない
            return false;
        }

        HANDLE fileMappingObjectHandle;
        void* pView;
        CreateFileMappingViewFromClient(&fileMappingObjectHandle, &pView, size, mapTransferType);

        switch (mapTransferType)
        {
        case MapTransferType_Send:
        case MapTransferType_Exchange:
            std::memcpy(pView, pointer, size);
            break;
        case MapTransferType_Receive:
            // 受信なので memcpy は不要
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        pOutFileMappingView->bufferSize = size;
        pOutFileMappingView->fileMappingObjectHandle = fileMappingObjectHandle;
        pOutFileMappingView->mapTransferType = mapTransferType;
        pOutFileMappingView->pBuffer = pointer;
        pOutFileMappingView->pView = pView;

        return true;
    }

    bool GetAsyncRequestResult(detail::HipcWindowsNamedPipeAsyncRequest** ppRequest) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(*ppRequest);
        bool ret = m_pNamedPipe->GetAsyncRequestResult(*ppRequest);
        *ppRequest = nullptr;
        return ret;
    }

    void CancelAllAsyncRequest() NN_NOEXCEPT
    {
        if (m_pReceiveMessageRequest)
        {
            m_pNamedPipe->CancelAsyncRequest(m_pReceiveMessageRequest);
            m_pReceiveMessageRequest = nullptr;
        }
        if (m_pSendMapRequest)
        {
            m_pNamedPipe->CancelAsyncRequest(m_pSendMapRequest);
            m_pSendMapRequest = nullptr;
        }
        for (int i = 0; i < PointerTransferMax; i++)
        {
            if (m_pSendPointerRequest[i])
            {
                m_pNamedPipe->CancelAsyncRequest(m_pSendPointerRequest[i]);
                m_pSendPointerRequest[i] = nullptr;
            }
        }
        if (m_pSendMessageRequest)
        {
            m_pNamedPipe->CancelAsyncRequest(m_pSendMessageRequest);
            m_pSendMessageRequest = nullptr;
        }
    }
};


class HipcEmulatedServerSessionReceiveNotificationWorker
    : public detail::HipcWindowsThreadpoolWork
{
    // ServerSession の AttachReceiveEvent 対応のため、パイプからデータを Read 可能であることを通知するイベントが欲しいが
    // WindowsAPI では実現できないはず。なので実際に非同期で数バイト Read することで対応する。
    // ところが、AttachReceiveEvent は ServerSession の生存期間に渡って有効なイベントを提供しなくてはならないが、パイプから
    // 非同期 Read の完了を通知するイベントは Read するごとに変わってしまう。そのため、同一のイベントを提供する仕組みをここで作る
private:
    SharedPointer<detail::HipcWindowsNamedPipe> m_pNamedPipe;
    os::Event m_ReceivableEvent;
    os::Event m_ReceivedEvent;
    os::Event m_ExitEvent;
    Bit32 m_ReceiveBuffer;

protected:
    explicit HipcEmulatedServerSessionReceiveNotificationWorker(SharedPointer<detail::HipcWindowsNamedPipe> pNamedPipe) NN_NOEXCEPT
        : m_pNamedPipe(std::move(pNamedPipe))
        , m_ReceivableEvent(os::EventClearMode_ManualClear)
        , m_ReceivedEvent(os::EventClearMode_ManualClear)
        , m_ExitEvent(os::EventClearMode_ManualClear)
    {
        this->SubmitThreadpoolWork();
    }

    ~HipcEmulatedServerSessionReceiveNotificationWorker() NN_NOEXCEPT
    {
        m_ExitEvent.Signal();
        this->WaitForThreadpoolWork(true);
    }

public:
    void AttachReceivableEvent(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        os::InitializeMultiWaitHolder(pHolder, m_ReceivableEvent.GetBase());
    }

    void CopyUnwillinglyReceivedBuffer(void* pDestination, size_t destinationSize, size_t* pOutCopiedSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(destinationSize >= sizeof(m_ReceiveBuffer));

        m_ReceivableEvent.Wait();
        m_ReceivableEvent.Clear();
        std::memcpy(pDestination, &m_ReceiveBuffer, sizeof(m_ReceiveBuffer));
        *pOutCopiedSize = sizeof(m_ReceiveBuffer);
    }

    void ContinueToReceiveNext() NN_NOEXCEPT
    {
        m_ReceivedEvent.Signal();
    }

private:
    virtual void RunThreadpoolWork(PTP_CALLBACK_INSTANCE pInstance) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(pInstance);

        os::MultiWaitType multiWait;
        os::MultiWaitHolderType receivedEventHolder;
        os::MultiWaitHolderType exitEventHolder;

        os::InitializeMultiWait(&multiWait);
        os::InitializeMultiWaitHolder(&receivedEventHolder, m_ReceivedEvent.GetBase());
        os::InitializeMultiWaitHolder(&exitEventHolder, m_ExitEvent.GetBase());

        while (NN_STATIC_CONDITION(true))
        {
            detail::HipcWindowsNamedPipeAsyncRequest* pRequest = m_pNamedPipe->ReadAsync(&m_ReceiveBuffer, sizeof(m_ReceiveBuffer));
            if (pRequest == nullptr)
            {
                // パイプが閉じられたことを通知する必要がある ---> ReceivableEvent を直ちにシグナル状態にする
            }
            else
            {
                os::MultiWaitHolderType receivableEventHolder;
                m_pNamedPipe->AttachAsyncRequestCompletionEvent(&receivableEventHolder, pRequest);

                os::LinkMultiWaitHolder(&multiWait, &exitEventHolder);
                os::LinkMultiWaitHolder(&multiWait, &receivableEventHolder);

                auto signaled = os::WaitAny(&multiWait);
                os::UnlinkAllMultiWaitHolder(&multiWait);

                if (signaled == &exitEventHolder)
                {
                    m_pNamedPipe->CancelAsyncRequest(pRequest);
                    break;
                }
                else
                {
                    m_pNamedPipe->GetAsyncRequestResult(pRequest);
                }
            }

            m_ReceivableEvent.Signal();

            os::LinkMultiWaitHolder(&multiWait, &exitEventHolder);
            os::LinkMultiWaitHolder(&multiWait, &receivedEventHolder);

            auto signaled = os::WaitAny(&multiWait);
            os::UnlinkAllMultiWaitHolder(&multiWait);

            if (signaled == &exitEventHolder)
            {
                break;
            }
            else
            {
                m_ReceivedEvent.Clear();
            }
        }

        os::FinalizeMultiWaitHolder(&exitEventHolder);
        os::FinalizeMultiWaitHolder(&receivedEventHolder);
        os::FinalizeMultiWait(&multiWait);
    }
};


class HipcEmulatedServerSessionRequest
    : public HipcEmulatedObjectBase
{
private:
    struct FileMappingView
    {
        HANDLE fileMappingObjectHandle;
        void* pView;
    };

private:
    SharedPointer<detail::HipcWindowsNamedPipe> m_pNamedPipe;

    int m_FileMappingViewCount;
    FileMappingView m_FileMappingView[MapTransferMax];

    HipcEmulatedServerSessionReceiveNotificationWorker* m_pReceiveNotificationWorker;

protected:
    explicit HipcEmulatedServerSessionRequest(SharedPointer<detail::HipcWindowsNamedPipe> pNamedPipe) NN_NOEXCEPT
        : m_pNamedPipe(std::move(pNamedPipe))
        , m_FileMappingViewCount(0)
    {
        m_pReceiveNotificationWorker = Factory<HipcEmulatedServerSessionReceiveNotificationWorker>::Create(m_pNamedPipe);
    }

    ~HipcEmulatedServerSessionRequest() NN_NOEXCEPT
    {
        m_pReceiveNotificationWorker->Release();
    }

public:
    void AttachReceiveEvent(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        m_pReceiveNotificationWorker->AttachReceivableEvent(pHolder);
    }

    bool ReceiveClientMessage(void* pMessageBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        // 受信指定リストを保存する
        int receiveListCount = 0;
        sf::detail::PointerAndSize receiveList[ReceiveListMax];
        detail::HipcMessageReceiveBufferMode receiveBufferMode;
        {
            detail::HipcMessageReader hipcMessageReader;
            hipcMessageReader.Initialize(pMessageBuffer);

            receiveBufferMode = hipcMessageReader.GetHeaderInfo().baseInfo.receiveBufferMode;
            CopyReceiveList(receiveList, ReceiveListMax, &receiveListCount, hipcMessageReader);
        }
        // クライアントからの HIPC メッセージを待つ
        NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(
            ReceiveClientMessage_ReceiveHipcMessage(pMessageBuffer, bufferSize));

        detail::HipcMessageReader hipcMessageReader;
        hipcMessageReader.Initialize(pMessageBuffer);
        const detail::HipcMessageHeaderInfo& headerInfo = hipcMessageReader.GetHeaderInfo().baseInfo;
        const detail::HipcMessageDataInfo dataInfo = detail::MakeHipcMessageDataInfo(headerInfo);
        detail::HipcMessageWriter hipcMessageWriter(dataInfo, pMessageBuffer, bufferSize);

        // クライアントプロセスのハンドルを複製し、ハンドル値を内部値に置き換える
        EmulateHipcSpecialDataTransferAfterReceive(hipcMessageReader, hipcMessageWriter, true, m_pNamedPipe->GetClientProcessId());

        // ポインタ転送を含むなら、指定された受信指定リストに従ってデータを受信
        NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(
            ReceivePointerTransferData(
                pMessageBuffer,
                bufferSize,
                receiveList,
                receiveListCount,
                receiveBufferMode,
                m_pNamedPipe.Get(),
                hipcMessageReader,
                hipcMessageWriter));

        // マップ転送を含むなら、ビューを開いて HIPC メッセージ内のアドレスを書き換え
        NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(
            ReceiveClientMessage_ReceiveMapTransferData(hipcMessageReader, hipcMessageWriter));

        return true;
    }

    bool SendServerReply(void* pMessageBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        detail::HipcMessageReader hipcMessageReader;
        hipcMessageReader.Initialize(pMessageBuffer);
        const detail::HipcMessageHeaderInfo& headerInfo = hipcMessageReader.GetHeaderInfo().baseInfo;
        const detail::HipcMessageDataInfo dataInfo = detail::MakeHipcMessageDataInfo(headerInfo);
        detail::HipcMessageWriter hipcMessageWriter(dataInfo, pMessageBuffer, bufferSize);

        // プロセス ID 値とハンドル値を本物の値に置き換える
        EmulateHipcSpecialDataTransferBeforeSend(hipcMessageReader, hipcMessageWriter, true, m_pNamedPipe->GetClientProcessId());

        // HIPC メッセージ送信
        NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(m_pNamedPipe->Write(pMessageBuffer, hipcMessageReader.GetMessageByteSize()));

        // ポインタ転送を含むなら、ポインタごとに送信
        for (int i = 0; i < headerInfo.pointerCount; i++)
        {
            detail::AddressValueAndSize avas = hipcMessageReader.GetPointer(i);
            uintptr_t pointer = detail::UnregisterAddress(avas.pointer);

            NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(m_pNamedPipe->Write(reinterpret_cast<void*>(pointer), avas.size));
        }

        // マップ転送を含むなら、ファイルマッピングオブジェクトハンドルを破棄
        for (int i = 0; i < m_FileMappingViewCount; i++)
        {
            CloseFileMappingView(m_FileMappingView[i].fileMappingObjectHandle, m_FileMappingView[i].pView);
        }

        // 次のクライアントからの HIPC メッセージの受信通知を受けられるようにする
        m_pReceiveNotificationWorker->ContinueToReceiveNext();

        return true;
    }

private:
    bool ReceiveClientMessage_ReceiveHipcMessage(void* pMessageBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        size_t unwillinglyReceivedBufferSize;
        m_pReceiveNotificationWorker->CopyUnwillinglyReceivedBuffer(pMessageBuffer, bufferSize, &unwillinglyReceivedBufferSize);

        uintptr_t pRemainingDataHead = reinterpret_cast<uintptr_t>(pMessageBuffer) + unwillinglyReceivedBufferSize;

        NN_ABORT_UNLESS(bufferSize - unwillinglyReceivedBufferSize >= 0);

        NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(
            m_pNamedPipe->Read(reinterpret_cast<void*>(pRemainingDataHead), bufferSize - unwillinglyReceivedBufferSize));

        return true;
    }

    bool ReceiveClientMessage_ReceiveMapTransferData(
        const detail::HipcMessageReader& reader,
        detail::HipcMessageWriter& writer) NN_NOEXCEPT
    {
        const detail::HipcMessageHeaderInfo& headerInfo = reader.GetHeaderInfo().baseInfo;

        MapTransferEmulationData mapData;
        NN_SF_DETAIL_TRUE_EXP_OR_RETURN_FALSE(m_pNamedPipe->Read(&mapData, sizeof(mapData)));

        ULONG clientProcessId = m_pNamedPipe->GetClientProcessId();

        m_FileMappingViewCount = 0;
        for (int i = 0; i < headerInfo.sendCount; i++)
        {
            detail::MapTransferBufferInfo bufferInfo = reader.GetSend(i);

            if (bufferInfo.size == 0)
            {
                // Windows の仕様で長さ 0 のファイルに対してマッピングすることはできない。ハンドルも送られて来ないので nullptr に対してマッピングする
                writer.SetSend(i, detail::RegisterAddress(0), bufferInfo.size, bufferInfo.mapTransferAttribute);
                continue;
            }

            HANDLE fileMappingObjectHandle
                = DuplicateExternalHandle(ReinterpretAsNativeHandle(mapData.sendHandles[i]), clientProcessId, false);

            void* pView;
            OpenFileMappingViewFromServer(&pView, fileMappingObjectHandle, MapTransferType_Send);

            m_FileMappingView[m_FileMappingViewCount].fileMappingObjectHandle = fileMappingObjectHandle;
            m_FileMappingView[m_FileMappingViewCount].pView = pView;

            writer.SetSend(i,
                detail::RegisterAddress(reinterpret_cast<uintptr_t>(pView)), bufferInfo.size, bufferInfo.mapTransferAttribute);

            m_FileMappingViewCount++;
        }
        for (int i = 0; i < headerInfo.receiveCount; i++)
        {
            detail::MapTransferBufferInfo bufferInfo = reader.GetReceive(i);

            if (bufferInfo.size == 0)
            {
                writer.SetReceive(i, detail::RegisterAddress(0), bufferInfo.size, bufferInfo.mapTransferAttribute);
                continue;
            }

            HANDLE fileMappingObjectHandle
                = DuplicateExternalHandle(ReinterpretAsNativeHandle(mapData.receiveHandles[i]), clientProcessId, false);

            void* pView;
            OpenFileMappingViewFromServer(&pView, fileMappingObjectHandle, MapTransferType_Receive);

            m_FileMappingView[m_FileMappingViewCount].fileMappingObjectHandle = fileMappingObjectHandle;
            m_FileMappingView[m_FileMappingViewCount].pView = pView;

            writer.SetReceive(i,
                detail::RegisterAddress(reinterpret_cast<uintptr_t>(pView)), bufferInfo.size, bufferInfo.mapTransferAttribute);

            m_FileMappingViewCount++;
        }
        for (int i = 0; i < headerInfo.exchangeCount; i++)
        {
            detail::MapTransferBufferInfo bufferInfo = reader.GetExchange(i);

            if (bufferInfo.size == 0)
            {
                writer.SetExchange(i, detail::RegisterAddress(0), bufferInfo.size, bufferInfo.mapTransferAttribute);
                continue;
            }

            HANDLE fileMappingObjectHandle
                = DuplicateExternalHandle(ReinterpretAsNativeHandle(mapData.exchangeHandles[i]), clientProcessId, false);

            void* pView;
            OpenFileMappingViewFromServer(&pView, fileMappingObjectHandle, MapTransferType_Exchange);

            m_FileMappingView[m_FileMappingViewCount].fileMappingObjectHandle = fileMappingObjectHandle;
            m_FileMappingView[m_FileMappingViewCount].pView = pView;

            writer.SetExchange(i,
                detail::RegisterAddress(reinterpret_cast<uintptr_t>(pView)), bufferInfo.size, bufferInfo.mapTransferAttribute);

            m_FileMappingViewCount++;
        }

        return true;
    }
};


HipcEmulatedNamedPipeClientSession::HipcEmulatedNamedPipeClientSession(detail::HipcWindowsNamedPipe* pNamedPipe) NN_NOEXCEPT
    : m_pNamedPipe(pNamedPipe, false)
{
}

HipcEmulatedNamedPipeClientSession::~HipcEmulatedNamedPipeClientSession() NN_NOEXCEPT
{
}

HipcEmulatedNamedPipeClientSession* HipcEmulatedNamedPipeClientSession::Create(detail::HipcWindowsNamedPipe* pNamedPipe) NN_NOEXCEPT
{
    return Factory<HipcEmulatedNamedPipeClientSession>::Create(pNamedPipe);
}

HipcEmulatedSessionRequest* HipcEmulatedNamedPipeClientSession::CreateRequest(
    void* clientMessageBuffer, size_t clientMessageBufferSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_pNamedPipe);  // DetachNamedPipe() ---> CreateRequest() の防止

    HipcEmulatedNamedPipeSessionRequest* pRequest = Factory<HipcEmulatedNamedPipeSessionRequest>::Create(m_pNamedPipe);

    if (pRequest->SendClientRequest(clientMessageBuffer, clientMessageBufferSize))
    {
        return pRequest;
    }
    else
    {
        pRequest->Release();
        return nullptr;
    }
}


HipcEmulatedNamedPipeServerSession::HipcEmulatedNamedPipeServerSession(SharedPointer<detail::HipcWindowsNamedPipe> pNamedPipe) NN_NOEXCEPT
    : m_pNamedPipe(std::move(pNamedPipe))
    , m_pNotificationReceiver(nullptr)
{
    m_pSessionRequest = Factory<HipcEmulatedServerSessionRequest>::Create(m_pNamedPipe);
}

HipcEmulatedNamedPipeServerSession::~HipcEmulatedNamedPipeServerSession() NN_NOEXCEPT
{
    m_pSessionRequest->Release();

    if (m_pNotificationReceiver)
    {
        m_pNotificationReceiver->OnSessionClosing(this, m_pNamedPipe.Get());
    }
}

HipcEmulatedNamedPipeServerSession* HipcEmulatedNamedPipeServerSession::Create(
    SharedPointer<detail::HipcWindowsNamedPipe> pNamedPipe) NN_NOEXCEPT
{
    return Factory<HipcEmulatedNamedPipeServerSession>::Create(std::move(pNamedPipe));
}

void HipcEmulatedNamedPipeServerSession::SetNotificationReceiver(
    IHipcEmulatedNamedPipeServerSessionNotificationReceiver* pNotificationReceiver) NN_NOEXCEPT
{
    m_pNotificationReceiver = pNotificationReceiver;
}

void HipcEmulatedNamedPipeServerSession::AttachReceiveEvent(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    m_pSessionRequest->AttachReceiveEvent(pHolder);
}

Result HipcEmulatedNamedPipeServerSession::ReceiveRequest(bool* pClosed, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    *pClosed = !m_pSessionRequest->ReceiveClientMessage(messageBuffer, messageBufferSize);

    NN_RESULT_SUCCESS;
}

void HipcEmulatedNamedPipeServerSession::Reply(void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    m_pSessionRequest->SendServerReply(messageBuffer, messageBufferSize);
}


std::pair<
    HipcEmulatedNamedPipeServerSession*,
    HipcEmulatedNamedPipeClientSession*
> CreateHipcEmulatedNamedPipeSessionPair(nn::sf::ISharedObject* parentPort) NN_NOEXCEPT
{
    // TORIAEZU
    NN_UNUSED(parentPort);
    NN_SDK_ASSERT(parentPort == nullptr);

    auto ports = CreateHipcEmulatedNamedPipePortPair(1);
    HipcEmulatedNamedPipeServerPort* pServerPort = ports.first;
    HipcEmulatedNamedPipeClientPort* pClientPort = ports.second;

    HipcEmulatedNamedPipeClientSession* pClientSession = static_cast<HipcEmulatedNamedPipeClientSession*>(pClientPort->Connect(true));
    NN_SDK_ASSERT_NOT_NULL(pClientSession);

    HipcEmulatedNamedPipeServerSession* pServerSession = static_cast<HipcEmulatedNamedPipeServerSession*>(pServerPort->Accept());
    NN_SDK_ASSERT_NOT_NULL(pServerSession);

    pServerPort->Release();
    pClientPort->Release();

    return std::make_pair(pServerSession, pClientSession);
}

}}}  // namespace nn::sf::hipc
