﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstdio>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/sf/hipc/sf_HipcHandleTypes.h>
#include <nn/sf/hipc/sf_HipcServiceResolutionApi.h>

#include <nn/nn_Windows.h>
#include <tchar.h>

#include "sf_HipcEmulatedNamedPipePort.h"
#include "sf_HipcEmulatedNamedPipeServiceResolver.h"
#include "detail/sf_HipcWindowsNamedPipe.h"


namespace nn { namespace sf { namespace hipc {

namespace {

const TCHAR* HipcWindowsPipeNameGuid = _T("2ADE51CE-249F-4460-89B6-0F4D0D2F5529");

void CreateHipcPipeName(TCHAR* pName, size_t maxNameLength, const char* pServiceName) NN_NOEXCEPT
{
    TCHAR tServiceName[HipcServiceNameLengthMax + 1];
#ifdef UNICODE
    std::mbstowcs(tServiceName, pServiceName, HipcServiceNameLengthMax + 1);
#else
    std::strncpy(tServiceName, pServiceName, HipcServiceNameLengthMax + 1);
#endif

    int r = ::_sntprintf_s(
        pName,
        maxNameLength,
        _TRUNCATE,
        _T("\\\\.\\pipe\\NintendoSDK$nn_sf_%s_%s"),
        tServiceName,
        HipcWindowsPipeNameGuid);

    NN_UNUSED(r);
    NN_SDK_ASSERT(r >= 0);
}

}  // namespace unnamed


Result HipcEmulatedNamedPipeServiceResolver::RegisterService(HipcServerPortHandle* pOut, const char* name, int32_t maxSessions) NN_NOEXCEPT
{
    const int pipeNameLengthMax = detail::HipcWindowsNamedPipe::PipeNameLengthMax;

    TCHAR pipeName[pipeNameLengthMax];
    CreateHipcPipeName(pipeName, pipeNameLengthMax, name);

    auto serverPort = HipcEmulatedNamedPipeServerPort::Create(pipeName, maxSessions);
    serverPort->AddReference();
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        RegisterPortImpl(serverPort, name);
    }
    serverPort->StartAccepting();
    *pOut = serverPort;
    NN_RESULT_SUCCESS;
}

void HipcEmulatedNamedPipeServiceResolver::UnregisterService(const char* name) NN_NOEXCEPT
{
    HipcEmulatedNamedPipeServerPort* port;
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        port = static_cast<HipcEmulatedNamedPipeServerPort*>(UnregisterPortImpl(name));
    }
    port->StopAccepting();
    port->Release();
}

bool HipcEmulatedNamedPipeServiceResolver::IsServiceRegistered(const char* name) NN_NOEXCEPT
{
    const int pipeNameLengthMax = detail::HipcWindowsNamedPipe::PipeNameLengthMax;

    TCHAR pipeName[pipeNameLengthMax];
    CreateHipcPipeName(pipeName, pipeNameLengthMax, name);

    BOOL pipeExists = ::WaitNamedPipe(pipeName, 0);
    return pipeExists ? true : false;
}

HipcClientPortHandle HipcEmulatedNamedPipeServiceResolver::GetServicePortImpl(const char* name) NN_NOEXCEPT
{
    const int pipeNameLengthMax = detail::HipcWindowsNamedPipe::PipeNameLengthMax;

    TCHAR pipeName[pipeNameLengthMax];
    CreateHipcPipeName(pipeName, pipeNameLengthMax, name);

    return HipcEmulatedNamedPipeClientPort::Create(pipeName);
}

}}}
