﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/sf/sf_ISharedObject.h>

#include <nn/os/os_Mutex.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include <nn/nn_Windows.h>

#include "../sf_HipcEmulatedObjectBase.h"


namespace nn { namespace sf { namespace hipc { namespace detail {

class HipcWindowsNamedPipeAsyncRequest;
class HipcWindowsNamedPipeClientConnectWorker;
class HipcWindowsNamedPipeServerListenWorker;

class HipcWindowsNamedPipe
    : public HipcEmulatedObjectBase
{
public:
    static const int PipeNameLengthMax = 256;

private:
    enum State
    {
        State_None,
        State_Initialized,
        State_PipeConnecting,
        State_PipeConnected,
    };

    State m_State;
    HANDLE m_PipeHandle;
    bool m_IsServer;
    int m_IoRequestCounter;
    os::Mutex m_StateLock;

    // client
    HipcWindowsNamedPipeClientConnectWorker* m_pConnectWorker;

    // server
    HipcWindowsNamedPipeServerListenWorker* m_pListenWorker;

protected:
    HipcWindowsNamedPipe() NN_NOEXCEPT;
    ~HipcWindowsNamedPipe() NN_NOEXCEPT;

public:
    static HipcWindowsNamedPipe* Create() NN_NOEXCEPT;

    // client
    void InitializeAsClient(const TCHAR* pPipeName) NN_NOEXCEPT;

    // server
    void InitializeAsServer(const TCHAR* pPipeName, int maxSessions) NN_NOEXCEPT;

    // client
    bool Connect(bool blocking) NN_NOEXCEPT;
    void Connect(HANDLE pipeHandle) NN_NOEXCEPT;
    void AttachConnectEvent(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;

    // server
    void Listen() NN_NOEXCEPT;
    bool Accept(bool blocking) NN_NOEXCEPT;
    void AttachAcceptEvent(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;

    // Connect または Accept が true を返してから Disconnect を呼ぶまでの間 true を返す
    // 実際のパイプの状態には依らない
    bool IsConnected() const NN_NOEXCEPT { return m_State == State_PipeConnected; }

    bool Read(void* pBuffer, size_t bufferSize) NN_NOEXCEPT;
    bool Read(void* pBuffer, size_t bufferSize, size_t* pOutTransferredBytes) NN_NOEXCEPT;
    bool Write(void* pBuffer, size_t bufferSize) NN_NOEXCEPT;
    bool Write(void* pBuffer, size_t bufferSize, size_t* pOutTransferredBytes) NN_NOEXCEPT;

    HipcWindowsNamedPipeAsyncRequest* ReadAsync(void* pBuffer, size_t bufferSize) NN_NOEXCEPT;
    HipcWindowsNamedPipeAsyncRequest* WriteAsync(void* pBuffer, size_t bufferSize) NN_NOEXCEPT;
    bool GetAsyncRequestResult(HipcWindowsNamedPipeAsyncRequest* pRequest) NN_NOEXCEPT;
    bool GetAsyncRequestResult(HipcWindowsNamedPipeAsyncRequest* pRequest, size_t* pOutTransferredBytes) NN_NOEXCEPT;
    void AttachAsyncRequestCompletionEvent(os::MultiWaitHolderType* pHolder, HipcWindowsNamedPipeAsyncRequest* pRequest) NN_NOEXCEPT;
    void CancelAsyncRequest(HipcWindowsNamedPipeAsyncRequest* pRequest) NN_NOEXCEPT;

    void Disconnect() NN_NOEXCEPT;

    void Finalize() NN_NOEXCEPT;

    // client
    HANDLE GetPipeHandle() const NN_NOEXCEPT { return m_PipeHandle; }
    HANDLE DetachPipeHandle() NN_NOEXCEPT;

    // server
    ULONG GetClientProcessId() NN_NOEXCEPT;
    ULONG GetServerProcessId() NN_NOEXCEPT;
};

}}}}
