﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include <nn/sf/hipc/sf_HipcTlsBuffer.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SdkMultipleWaitApi.h>
#include <nn/nn_Windows.h>

#include <tchar.h>

#include "../sf_HipcEmulatedServiceResolver.h"
#include "sf_HipcWindowsThreadpoolWork.h"
#include "sf_HipcWindowsNamedPipe.h"


namespace {

void PrepareOverlapped(OVERLAPPED* pOverlapped) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(HasOverlappedIoCompleted(pOverlapped));
    NN_ABORT_UNLESS(
        ::ResetEvent(pOverlapped->hEvent),
        "ResetEvent failed with an unexpected error %lu", ::GetLastError());
    pOverlapped->Internal = 0;
    pOverlapped->InternalHigh = 0;
    pOverlapped->Offset = 0;
    pOverlapped->OffsetHigh = 0;
    pOverlapped->Pointer = NULL;
}

void CancelNamedPipeIo(HANDLE pipeHandle, OVERLAPPED* pOverlapped) NN_NOEXCEPT
{
    BOOL success = ::CancelIoEx(pipeHandle, pOverlapped);
    if (!success)
    {
        DWORD lastError = ::GetLastError();
        switch (lastError)
        {
        case ERROR_NOT_FOUND:
            // キャンセルリクエストする前に I/O 処理が完了した
            return;
        default:
            NN_ABORT("CancelIoEx failed with an unexpected error %lu", lastError);
        }
    }

    DWORD dummy;
    ::GetOverlappedResult(pipeHandle, pOverlapped, &dummy, TRUE);  // 成否は問わない
}

bool IsPipeDisconnectionError(DWORD error) NN_NOEXCEPT
{
    switch (error)
    {
    case ERROR_BROKEN_PIPE:
    case ERROR_NO_DATA:
    case ERROR_PIPE_NOT_CONNECTED:
    case ERROR_PIPE_LISTENING:
        return true;
    default:
        return false;
    }
}

}  // namespace unnamed


namespace nn { namespace sf { namespace hipc { namespace detail {

class HipcWindowsNamedPipeAsyncRequest
    : public nn::sf::ISharedObject
{
private:
    HANDLE m_Handle;
    OVERLAPPED m_Overlapped;
    bool m_IsRequestPending;  // HasOverlappedIoCompleted はエラー発生時に誤った結果を返すことがある。リクエスト進捗は自前で管理する

protected:
    HipcWindowsNamedPipeAsyncRequest() NN_NOEXCEPT
        : m_Handle(NULL)
        , m_IsRequestPending(false)
    {
        std::memset(&m_Overlapped, 0, sizeof(m_Overlapped));
        m_Overlapped.hEvent = ::CreateEvent(NULL, TRUE, FALSE, NULL);
        NN_ABORT_UNLESS(m_Overlapped.hEvent != NULL, "CreateEvent failed with an unexpected error %lu", ::GetLastError());
    }

    ~HipcWindowsNamedPipeAsyncRequest() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsRequestPending);
        ::CloseHandle(m_Overlapped.hEvent);
    }

public:
    bool StartReadIoRequest(HANDLE handle, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsRequestPending);

        PrepareOverlapped(&m_Overlapped);
        BOOL success = ::ReadFile(handle, buffer, static_cast<DWORD>(bufferSize), NULL, &m_Overlapped);
        if (!success)
        {
            DWORD lastError = ::GetLastError();
            if (lastError == ERROR_IO_PENDING)
            {
                // 正常。I/O 処理を実行中
            }
            else if (lastError == ERROR_MORE_DATA)
            {
                // HipcEmulatedServerSession でパイプにデータが到着しているかどうかを実際に
                // Read することで確認しているので、すべて読み切れていなくともエラーにしない
                // TODO: 本当にバッファが不足した場合と区別できるようにする必要がある
            }
            else if (IsPipeDisconnectionError(lastError))
            {
                NN_ABORT_UNLESS(
                    ::SetEvent(m_Overlapped.hEvent),
                    "SetEvent failed with an unexpected error %lu", ::GetLastError());
                return false;
            }
            else
            {
                NN_ABORT("ReadFile failed with an unexpected error %lu", lastError);
            }
        }
        m_Handle = handle;
        m_IsRequestPending = true;
        return true;
    }

    bool StartWriteIoRequest(HANDLE handle, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsRequestPending);

        PrepareOverlapped(&m_Overlapped);
        BOOL success = ::WriteFile(handle, buffer, static_cast<DWORD>(bufferSize), NULL, &m_Overlapped);
        if (!success)
        {
            DWORD lastError = ::GetLastError();
            if (lastError == ERROR_IO_PENDING)
            {
                // 正常。I/O 処理を実行中
            }
            else if (IsPipeDisconnectionError(lastError))
            {
                NN_ABORT_UNLESS(
                    ::SetEvent(m_Overlapped.hEvent),
                    "SetEvent failed with an unexpected error %lu", ::GetLastError());
                return false;
            }
            else
            {
                NN_ABORT("WriteFile failed with an unexpected error %lu", lastError);
            }
        }
        m_Handle = handle;
        m_IsRequestPending = true;
        return true;
    }

    void AttachIoRequestCompletionEvent(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        os::InitializeMultiWaitHolder(pHolder, m_Overlapped.hEvent);
    }

    bool GetResult() NN_NOEXCEPT
    {
        size_t dummy;
        return GetResult(&dummy);
    }

    bool GetResult(size_t* pOutTransferredBytes) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsRequestPending);

        DWORD bytesTransferred;
        BOOL success = ::GetOverlappedResult(m_Handle, &m_Overlapped, &bytesTransferred, TRUE);
        m_IsRequestPending = false;
        if (!success)
        {
            DWORD lastError = ::GetLastError();
            if (lastError == ERROR_MORE_DATA)
            {
                // HipcEmulatedServerSession でパイプにデータが到着しているかどうかを実際に
                // Read することで確認しているので、すべて読み切れていなくともエラーにしない
                // TODO: 本当にバッファが不足した場合と区別できるようにする必要がある
            }
            else if (IsPipeDisconnectionError(lastError))
            {
                return false;
            }
            else
            {
                NN_ABORT("GetOverlappedResult failed with an unexpected error %lu", lastError);
            }
        }

        *pOutTransferredBytes = bytesTransferred;
        return true;
    }

    void CancelIoRequest() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsRequestPending);

        CancelNamedPipeIo(m_Handle, &m_Overlapped);
        m_IsRequestPending = false;
    }
};


class HipcWindowsNamedPipeClientConnectWorker
    : public HipcWindowsThreadpoolWork
{
    // 指定された名前のパイプが利用可能になったかどうかを待機可能なハンドルで確認する手段はない。WaitNamedPipe でポーリングしないといけない
    // AttachConnectEvent による同時待ちを可能とするため、ポーリングを別スレッドで行う
private:
    static const uint64_t WaitNamedPipeIntervalMilliSeconds = 50;

    TCHAR m_PipeName[HipcWindowsNamedPipe::PipeNameLengthMax];
    os::Event m_ConnectableEvent;
    os::Event m_ExitEvent;
    bool m_IsConnectWorkActive;

protected:
    explicit HipcWindowsNamedPipeClientConnectWorker(const TCHAR* pPipeName) NN_NOEXCEPT
        : m_ConnectableEvent(os::EventClearMode_ManualClear)
        , m_ExitEvent(os::EventClearMode_ManualClear)
        , m_IsConnectWorkActive(false)
    {
        ::_tcsncpy_s(m_PipeName, pPipeName, _TRUNCATE);
    }

    ~HipcWindowsNamedPipeClientConnectWorker() NN_NOEXCEPT
    {
        m_ExitEvent.Signal();
        this->WaitForThreadpoolWork(true);
    }

public:
    HANDLE Connect(bool blocking) NN_NOEXCEPT
    {
        HANDLE pipeHandle = NULL;

        while (pipeHandle == NULL || pipeHandle == INVALID_HANDLE_VALUE)
        {
            if (!m_IsConnectWorkActive)
            {
                StartConnectWork();
            }

            if (blocking)
            {
                m_ConnectableEvent.Wait();
            }
            else
            {
                // FIXME: パイプに接続可能な状態であっても StartConnectWork してからここに来るまでの間にスレッドプールのスレッドがイベントをシグナル状態にしている保証がない
                bool signaled = m_ConnectableEvent.TryWait();
                if (!signaled)
                {
                    return NULL;
                }
            }

            pipeHandle = ::CreateFile(
                m_PipeName,
                GENERIC_READ | GENERIC_WRITE,
                0,
                NULL,
                OPEN_EXISTING,
                FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED,
                NULL);

            if (pipeHandle == INVALID_HANDLE_VALUE)
            {
                DWORD lastError = ::GetLastError();
                switch (lastError)
                {
                case ERROR_FILE_NOT_FOUND:  // パイプが破棄された
                case ERROR_PIPE_BUSY:       // パイプは存在するが、すべてのインスタンスが使用されている
                    if (blocking)
                    {
                        // ループして再試行する
                    }
                    else
                    {
                        return NULL;
                    }
                    break;
                default:
                    NN_ABORT("CreateFile failed with an unexpected error %lu", lastError);
                }
            }
        }

        DWORD mode = PIPE_READMODE_MESSAGE;
        if (!::SetNamedPipeHandleState(pipeHandle, &mode, NULL, NULL))
        {
            DWORD lastError = ::GetLastError();
            if (IsPipeDisconnectionError(lastError))
            {
                // パイプへの接続は成功したが、その後すぐにサーバ側から切断されてしまった場合はここに来る
                // 一度サーバへの接続には成功しているので、再接続試行せずにこのままパイプハンドルを返して成功扱いにする
                // 実際にパイプを使用する段階で切断を検出する
            }
            else
            {
                NN_ABORT("SetNamedPipeHandleState failed with an unexpected error %lu", lastError);
            }
        }

        return pipeHandle;
    }

    void AttachConnectableEvent(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        os::InitializeMultiWaitHolder(pHolder, m_ConnectableEvent.GetBase());
    }

private:
    virtual void RunThreadpoolWork(PTP_CALLBACK_INSTANCE pInstance) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(pInstance);

        NN_UTIL_SCOPE_EXIT
        {
            m_IsConnectWorkActive = false;
        };

        while (NN_STATIC_CONDITION(true))
        {
            BOOL pipeConnectable = ::WaitNamedPipe(m_PipeName, 0);
            if (pipeConnectable)
            {
                m_ConnectableEvent.Signal();
                break;
            }
            else
            {
                DWORD lastError = ::GetLastError();
                switch (lastError)
                {
                case ERROR_FILE_NOT_FOUND:  // パイプがまだ作られていない。一定時間を置いて再試行
                case ERROR_SEM_TIMEOUT:     // パイプは存在するが、すべてのインスタンスが使用されている。一定時間を置いて再試行
                {
                    bool shouldExit = m_ExitEvent.TimedWait(TimeSpan::FromMilliSeconds(WaitNamedPipeIntervalMilliSeconds));
                    if (shouldExit)
                    {
                        return;
                    }
                    break;
                }
                default:
                    NN_ABORT("WaitNamedPipe failed with an unexpected error %lu", lastError);
                }
            }
        }
    }

    void StartConnectWork() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsConnectWorkActive);

        m_ConnectableEvent.Clear();
        m_ExitEvent.Clear();
        m_IsConnectWorkActive = true;
        this->SubmitThreadpoolWork();
    }
};


class HipcWindowsNamedPipeServerListenWorker
    : public HipcWindowsThreadpoolWork
{
    // ConnectNamedPipe は Overlapped I/O に対応しているが、イベントがシグナルされるのはエラーが発生した時も含まれるので
    // AttachAcceptEvent に使用するイベントとしては不適切。仕方ないのでイベントは自前で準備し、別スレッドで ConnectNamedPipe の結果を監視する
private:
    HANDLE m_PipeHandle;
    os::Event m_ListenableEvent;
    os::Event m_AcceptableEvent;
    os::Event m_RefreshEvent;  // 「Listen の停止」を通知する手段と「スレッドの終了」を通知する手段を兼ねる。どちらの通知かは m_ShouldExit で判断
    bool m_ShouldExit;

protected:
    explicit HipcWindowsNamedPipeServerListenWorker(HANDLE pipeHandle) NN_NOEXCEPT
        : m_PipeHandle(pipeHandle)
        , m_ListenableEvent(os::EventClearMode_ManualClear)
        , m_AcceptableEvent(os::EventClearMode_ManualClear)
        , m_RefreshEvent(os::EventClearMode_ManualClear)
        , m_ShouldExit(false)
    {
        this->SubmitThreadpoolWork();
    }

    ~HipcWindowsNamedPipeServerListenWorker() NN_NOEXCEPT
    {
        m_ShouldExit = true;
        m_RefreshEvent.Signal();
        this->WaitForThreadpoolWork(true);
    }

public:
    void StartListen() NN_NOEXCEPT
    {
        m_ListenableEvent.Signal();
    }

    bool Accept(bool blocking) NN_NOEXCEPT
    {
        bool accepted;
        if (blocking)
        {
            m_AcceptableEvent.Wait();
            accepted = true;
        }
        else
        {
            accepted = m_AcceptableEvent.TryWait();
        }

        if (accepted)
        {
            m_AcceptableEvent.Clear();
        }
        return accepted;
    }

    void AttachAcceptableEvent(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        os::InitializeMultiWaitHolder(pHolder, m_AcceptableEvent.GetBase());
    }

    void StopListen() NN_NOEXCEPT
    {
        m_ListenableEvent.Clear();
        m_RefreshEvent.Signal();
    }

private:
    virtual void RunThreadpoolWork(PTP_CALLBACK_INSTANCE pInstance) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(pInstance);

        os::MultiWaitType multiWait;
        os::MultiWaitHolderType listenableEventHolder;
        os::MultiWaitHolderType refreshEventHolder;

        os::InitializeMultiWait(&multiWait);
        os::InitializeMultiWaitHolder(&listenableEventHolder, m_ListenableEvent.GetBase());
        os::InitializeMultiWaitHolder(&refreshEventHolder, m_RefreshEvent.GetBase());

        while (!m_ShouldExit)
        {
            // Listen を要求されるまで待機する
            {
                os::LinkMultiWaitHolder(&multiWait, &listenableEventHolder);
                os::LinkMultiWaitHolder(&multiWait, &refreshEventHolder);

                auto pSignaled = os::WaitAny(&multiWait);
                os::UnlinkAllMultiWaitHolder(&multiWait);

                if (pSignaled == &listenableEventHolder)
                {
                    m_ListenableEvent.Clear();
                }
                else
                {
                    m_RefreshEvent.Clear();
                    continue;
                }
            }
            // Listen してクライアントの接続を待つ
            {
                OVERLAPPED overlapped;
                std::memset(&overlapped, 0, sizeof(overlapped));

                if (!::ConnectNamedPipe(m_PipeHandle, &overlapped))
                {
                    DWORD lastError = ::GetLastError();
                    switch (lastError)
                    {
                    case ERROR_IO_PENDING:      // 正常。接続待機中
                        break;
                    case ERROR_PIPE_CONNECTED:  // 既にクライアントが接続している
                    case ERROR_NO_DATA:         // 一度クライアントが接続してきたが、すぐに切断された (この場合も接続成功と見なす。実際にパイプを使用する段階で切断を検出する)
                        m_AcceptableEvent.Signal();
                        break;
                    default:
                        NN_ABORT("ConnectNamedPipe failed with an unexpected error %lu", lastError);
                    }
                }
                if (m_AcceptableEvent.TryWait())
                {
                    continue;
                }

                os::MultiWaitHolderType overlappedEventHolder;
                os::InitializeMultiWaitHolder(&overlappedEventHolder, m_PipeHandle);

                os::LinkMultiWaitHolder(&multiWait, &overlappedEventHolder);
                os::LinkMultiWaitHolder(&multiWait, &refreshEventHolder);

                auto pSignaled = os::WaitAny(&multiWait);
                os::UnlinkAllMultiWaitHolder(&multiWait);
                os::FinalizeMultiWaitHolder(&overlappedEventHolder);

                if (pSignaled == &refreshEventHolder)
                {
                    m_RefreshEvent.Clear();

                    // 非同期 I/O を野放しにするわけにはいかないので、キャンセルを要求して I/O 処理の停止を待つ
                    if (!::CancelIoEx(m_PipeHandle, &overlapped))
                    {
                        DWORD lastError = ::GetLastError();
                        switch (lastError)
                        {
                        case ERROR_NOT_FOUND:
                            // キャンセル要求する前に I/O 処理が完了した ---> このまま GetOverlappedResult で結果を見る
                            break;
                        default:
                            NN_ABORT("CancelIoEx failed with an unexpected error %lu", lastError);
                        }
                    }
                }

                DWORD dummy;
                if (::GetOverlappedResult(m_PipeHandle, &overlapped, &dummy, TRUE))
                {
                    m_AcceptableEvent.Signal();
                }
                else
                {
                    DWORD lastError = ::GetLastError();
                    switch (lastError)
                    {
                    case ERROR_OPERATION_ABORTED:  // 非同期 I/O はキャンセルされた (CancelIoEx が正常に処理された)
                        break;
                    default:
                        NN_ABORT("ConnectNamedPipe failed with an unexpected error %lu", lastError);
                    }
                }
            }
        }

        os::FinalizeMultiWaitHolder(&refreshEventHolder);
        os::FinalizeMultiWaitHolder(&listenableEventHolder);
        os::FinalizeMultiWait(&multiWait);
    }
};


HipcWindowsNamedPipe::HipcWindowsNamedPipe() NN_NOEXCEPT
    : m_State(State_None)
    , m_PipeHandle(NULL)
    , m_IsServer(false)
    , m_IoRequestCounter(0)
    , m_StateLock(true)
    , m_pConnectWorker(nullptr)
    , m_pListenWorker(nullptr)
{
}

HipcWindowsNamedPipe::~HipcWindowsNamedPipe() NN_NOEXCEPT
{
    Finalize();
}

HipcWindowsNamedPipe* HipcWindowsNamedPipe::Create() NN_NOEXCEPT
{
    return Factory<HipcWindowsNamedPipe>::Create();
}

void HipcWindowsNamedPipe::InitializeAsServer(const TCHAR* pPipeName, int maxSessions) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(m_State == State_None);

    // TORIAEZU: サイズは適当。パフォーマンスを求めるならベンチマークする
    const DWORD outBufferSize = 1024 * 4;
    const DWORD inBufferSize = 1024 * 4;

    HANDLE pipeHandle = ::CreateNamedPipe(
        pPipeName,
        PIPE_ACCESS_DUPLEX | FILE_FLAG_OVERLAPPED,
        PIPE_TYPE_MESSAGE | PIPE_READMODE_MESSAGE | PIPE_WAIT | PIPE_REJECT_REMOTE_CLIENTS,
        maxSessions,
        outBufferSize,
        inBufferSize,
        0,
        NULL);

    if (pipeHandle == INVALID_HANDLE_VALUE)
    {
        DWORD lastError = ::GetLastError();
        switch (lastError)
        {
        case ERROR_PIPE_BUSY:
            // 既にパイプインスタンスが最大まで作られている
            // これは sf の実装に誤りがあるか、または複数プロセスが同じサービスを提供しようとしている可能性がある
            // TORIAEZU: アボートする
        default:
            NN_ABORT("CreateNamedPipe failed with an unexpected error %lu", lastError);
        }
    }

    m_pListenWorker = Factory<HipcWindowsNamedPipeServerListenWorker>::Create(pipeHandle);

    m_PipeHandle = pipeHandle;
    m_State = State_Initialized;
    m_IsServer = true;
}

void HipcWindowsNamedPipe::InitializeAsClient(const TCHAR* pPipeName) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(m_State == State_None);

    m_pConnectWorker = Factory<HipcWindowsNamedPipeClientConnectWorker>::Create(pPipeName);

    m_PipeHandle = NULL;
    m_State = State_Initialized;
    m_IsServer = false;
}

bool HipcWindowsNamedPipe::Connect(bool blocking) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(!m_IsServer);
    NN_SDK_ASSERT(m_State == State_Initialized || m_State == State_PipeConnecting);
    NN_SDK_ASSERT_NOT_NULL(m_pConnectWorker);  // Connect(pipeHandle) ---> Disconnect() ---> Connect(blocking) の防止

    HANDLE pipeHandle = m_pConnectWorker->Connect(blocking);
    if (pipeHandle == NULL)
    {
        return false;
    }

    m_PipeHandle = pipeHandle;
    m_State = State_PipeConnected;
    return true;
}

void HipcWindowsNamedPipe::Connect(HANDLE pipeHandle) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(m_State == State_None);
    NN_SDK_ASSERT(pipeHandle != NULL && pipeHandle != INVALID_HANDLE_VALUE);
    NN_SDK_ASSERT(m_pConnectWorker == nullptr);  // InitializeAsClient(pPipeName) ---> Connect(pipeHandle) の防止

    DWORD flags;
    NN_ABORT_UNLESS(
        ::GetNamedPipeInfo(pipeHandle, &flags, NULL, NULL, NULL),
        "GetNamedPipeInfo failed with an unexpected error %lu", ::GetLastError());
    NN_ABORT_UNLESS((flags & PIPE_SERVER_END) == 0);  // PIPE_CLIENT_END == 0 なので (flags & PIPE_CLIENT_END) == PIPE_CLIENT_END ではダメ
    NN_ABORT_UNLESS((flags & PIPE_TYPE_MESSAGE) == PIPE_TYPE_MESSAGE);

    DWORD mode = PIPE_READMODE_MESSAGE;
    NN_ABORT_UNLESS(
        ::SetNamedPipeHandleState(pipeHandle, &mode, NULL, NULL),
        "SetNamedPipeHandleState failed with an unexpected error %lu", ::GetLastError());

    m_PipeHandle = pipeHandle;
    m_State = State_PipeConnected;
    m_IsServer = false;
}

void HipcWindowsNamedPipe::AttachConnectEvent(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(!m_IsServer);
    NN_SDK_ASSERT(m_State == State_Initialized || m_State == State_PipeConnecting);
    NN_SDK_ASSERT_NOT_NULL(m_pConnectWorker);

    m_pConnectWorker->AttachConnectableEvent(pHolder);
}

void HipcWindowsNamedPipe::Listen() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(m_IsServer);
    NN_SDK_ASSERT(m_State == State_Initialized);

    m_pListenWorker->StartListen();
    m_State = State_PipeConnecting;
}

bool HipcWindowsNamedPipe::Accept(bool blocking) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(m_IsServer);
    NN_SDK_ASSERT(m_State == State_PipeConnecting);

    bool accepted = m_pListenWorker->Accept(blocking);
    if (accepted)
    {
        m_State = State_PipeConnected;
    }
    return accepted;
}

void HipcWindowsNamedPipe::AttachAcceptEvent(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(m_IsServer);
    NN_SDK_ASSERT(m_State == State_Initialized || m_State == State_PipeConnecting);

    m_pListenWorker->AttachAcceptableEvent(pHolder);
}

bool HipcWindowsNamedPipe::Read(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    size_t dummy;
    return Read(pBuffer, bufferSize, &dummy);
}

bool HipcWindowsNamedPipe::Read(void* pBuffer, size_t bufferSize, size_t* pOutTransferredBytes) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(m_State == State_PipeConnected);

    auto request = ReadAsync(pBuffer, bufferSize);
    if (!request)
    {
        return false;
    }
    return GetAsyncRequestResult(request, pOutTransferredBytes);
}

bool HipcWindowsNamedPipe::Write(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    size_t dummy;
    return Write(pBuffer, bufferSize, &dummy);
}

bool HipcWindowsNamedPipe::Write(void* pBuffer, size_t bufferSize, size_t* pOutTransferredBytes) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(m_State == State_PipeConnected);

    auto request = WriteAsync(pBuffer, bufferSize);
    if (!request)
    {
        return false;
    }
    return GetAsyncRequestResult(request, pOutTransferredBytes);
}

HipcWindowsNamedPipeAsyncRequest* HipcWindowsNamedPipe::ReadAsync(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(m_State == State_PipeConnected);

    auto request = Factory<HipcWindowsNamedPipeAsyncRequest>::Create();
    bool requestStarted = request->StartReadIoRequest(m_PipeHandle, pBuffer, bufferSize);

    if (!requestStarted)
    {
        request->Release();
        return nullptr;
    }

    m_IoRequestCounter++;
    return request;
}

HipcWindowsNamedPipeAsyncRequest* HipcWindowsNamedPipe::WriteAsync(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(m_State == State_PipeConnected);

    auto request = Factory<HipcWindowsNamedPipeAsyncRequest>::Create();
    bool requestStarted = request->StartWriteIoRequest(m_PipeHandle, pBuffer, bufferSize);

    if (!requestStarted)
    {
        request->Release();
        return nullptr;
    }

    m_IoRequestCounter++;
    return request;
}

bool HipcWindowsNamedPipe::GetAsyncRequestResult(HipcWindowsNamedPipeAsyncRequest* pRequest) NN_NOEXCEPT
{
    size_t dummy;
    return GetAsyncRequestResult(pRequest, &dummy);
}

bool HipcWindowsNamedPipe::GetAsyncRequestResult(
    HipcWindowsNamedPipeAsyncRequest* pRequest, size_t* pOutTransferredBytes) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pRequest);

    bool success = pRequest->GetResult(pOutTransferredBytes);
    pRequest->Release();

    {
        std::lock_guard<os::Mutex> lk(m_StateLock);
        m_IoRequestCounter--;
        NN_SDK_ASSERT(m_IoRequestCounter >= 0);
    }

    return success;
}

void HipcWindowsNamedPipe::AttachAsyncRequestCompletionEvent(
    os::MultiWaitHolderType* pHolder, HipcWindowsNamedPipeAsyncRequest* pRequest) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pRequest);

    pRequest->AttachIoRequestCompletionEvent(pHolder);
}

void HipcWindowsNamedPipe::CancelAsyncRequest(HipcWindowsNamedPipeAsyncRequest* pRequest) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pRequest);

    pRequest->CancelIoRequest();
    pRequest->Release();

    {
        std::lock_guard<os::Mutex> lk(m_StateLock);
        m_IoRequestCounter--;
        NN_SDK_ASSERT(m_IoRequestCounter >= 0);
    }
}

void HipcWindowsNamedPipe::Disconnect() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(m_State == State_PipeConnecting || m_State == State_PipeConnected);
    NN_SDK_ASSERT(m_IoRequestCounter == 0);

    switch (m_State)
    {
    case State_PipeConnecting:
        if (m_IsServer)
        {
            m_pListenWorker->StopListen();
        }
        else
        {
            // nop
        }
        break;
    case State_PipeConnected:
        if (m_IsServer)
        {
            // これらの関数は既にパイプが切断されていても成功する
            NN_ABORT_UNLESS(
                ::FlushFileBuffers(m_PipeHandle),
                "FlushFileBuffers failed with an unexpected error %lu", ::GetLastError());
            NN_ABORT_UNLESS(
                ::DisconnectNamedPipe(m_PipeHandle),
                "DisconnectNamedPipe failed with an unexpected error %lu", ::GetLastError());
        }
        else
        {
            // 既にパイプが切断されていても成功する
            NN_ABORT_UNLESS(
                ::CloseHandle(m_PipeHandle),
                "CloseHandle failed with an unexpected error %lu", ::GetLastError());
            m_PipeHandle = NULL;
        }
        break;
    case State_None:
    case State_Initialized:
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    m_State = State_Initialized;
}

void HipcWindowsNamedPipe::Finalize() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(m_IoRequestCounter == 0);

    switch (m_State)
    {
    case State_None:
        // 既に Finalize されているので、何もしない
        return;
    case State_Initialized:
        break;
    case State_PipeConnecting:
    case State_PipeConnected:
        Disconnect();
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // この時点で m_State == State_Initialized に相当する状態

    if (m_IsServer)
    {
        m_pListenWorker->Release();
        m_pListenWorker = nullptr;
        NN_ABORT_UNLESS(
            ::CloseHandle(m_PipeHandle),
            "CloseHandle failed with an unexpected error %lu", ::GetLastError());
        m_PipeHandle = NULL;
    }
    else
    {
        if (m_pConnectWorker)
        {
            m_pConnectWorker->Release();
            m_pConnectWorker = nullptr;
        }
        // クライアントの場合、パイプハンドルは Disconnect() の中で破棄されている
    }

    m_State = State_None;
}

HANDLE HipcWindowsNamedPipe::DetachPipeHandle() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(!m_IsServer);
    NN_SDK_ASSERT(m_State == State_PipeConnected);
    NN_SDK_ASSERT(m_IoRequestCounter == 0);

    if (m_pConnectWorker)
    {
        m_pConnectWorker->Release();
        m_pConnectWorker = nullptr;
    }

    HANDLE ret = m_PipeHandle;
    m_PipeHandle = NULL;
    m_State = State_None;
    return ret;
}

ULONG HipcWindowsNamedPipe::GetClientProcessId() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(m_IsServer);
    NN_SDK_ASSERT(m_State == State_PipeConnected);

    ULONG ret;
    NN_ABORT_UNLESS(
        ::GetNamedPipeClientProcessId(m_PipeHandle, &ret),
        "GetNamedPipeClientProcessId failed with an unexpected error %lu", ::GetLastError());

    return ret;
}

ULONG HipcWindowsNamedPipe::GetServerProcessId() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);
    NN_SDK_ASSERT(m_IsServer);
    NN_SDK_ASSERT(m_State == State_PipeConnected);

    ULONG ret;
    NN_ABORT_UNLESS(
        ::GetNamedPipeServerProcessId(m_PipeHandle, &ret),
        "GetNamedPipeServerProcessId failed with an unexpected error %lu", ::GetLastError());

    return ret;
}

}}}}  // namespace nn::sf::hipc::detail
