﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "sdmmc_SdHostStandardController.h"
#include "sdmmc_ClockResetController.tegra.h"
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace sdmmc {
namespace detail {

const size_t SdmmcRegistersSize = 0x200;
#if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
    const int PeripheralInterruptOffset = 32;
#endif

class SdmmcController : public SdHostStandardController
{
private:
    struct SdmmcRegisters
    {
        volatile SdHostStandardRegisters sdHostStandardRegisters;
        volatile uint32_t   vendorClockCntrl0;              // 0x100
        volatile uint32_t   vendorSysSwCntrl0;              // 0x104
        volatile uint8_t    reserved0[4];
        volatile uint32_t   vendorCapOverrides0;            // 0x10c
        volatile uint32_t   vendorBootCntrl0;               // 0x110
        volatile uint32_t   vendorBootActTimeout0;          // 0x014
        volatile uint32_t   bootDatTimeout0;                // 0x118
        volatile uint32_t   vendorDebounceCount0;           // 0x11c
        volatile uint32_t   vendorMiscCntrl0;               // 0x120
        volatile uint32_t   maxCurrentOverride0;            // 0x124
        volatile uint32_t   maxCurrentOverrideHi0;          // 0x128
        volatile uint8_t    reserved1[0xa4];
        volatile uint32_t   vendorClkGateHysteresisCount0;  // 0x1d0
        volatile uint32_t   vendorPresetVal00;              // 0x1d4
        volatile uint32_t   vendorPresetVal10;              // 0x1d8
        volatile uint32_t   vendorPresetVal20;              // 0x1dc
        volatile uint32_t   sdmemcomppadctrl0;              // 0x1e0
        volatile uint32_t   autoCalConfig0;                 // 0x1e4
        volatile uint32_t   autoCalInterval0;               // 0x1e8
        volatile uint32_t   autoCalStatus0;                 // 0x1ec
        volatile uint8_t    reserved2[4];
        volatile uint32_t   mmcifFifoctrl0;                 // 0x1f4
        volatile uint32_t   timeoutWcalSdmmc0;              // 0x1f8
    };

    SdmmcRegisters* m_pSdmmcRegisters;
    bool m_IsShutdowned;

    void ReleaseReset(SpeedMode speedMode) NN_NOEXCEPT;
    void AssertReset() NN_NOEXCEPT;

protected:
    virtual detail::ClockResetController::Module GetClockResetModule() const NN_NOEXCEPT = 0;

    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        virtual int GetInterruptNumber() const NN_NOEXCEPT = 0;
        virtual nn::os::InterruptEventType* GetInterruptEvent() const NN_NOEXCEPT = 0;
    #endif

public:
    explicit SdmmcController(nn::dd::PhysicalAddress registersPhysicalAddress) NN_NOEXCEPT
        : SdHostStandardController(registersPhysicalAddress, SdmmcRegistersSize)
    {
        uintptr_t registersAddress = nn::dd::QueryIoMappingAddress(registersPhysicalAddress, SdmmcRegistersSize);
        NN_ABORT_UNLESS(registersAddress != 0);
        m_pSdmmcRegisters = reinterpret_cast<SdmmcRegisters*>(registersAddress);
        m_IsShutdowned = true;
    }

    void LogRegisters() NN_NOEXCEPT;

    virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE
    {
        detail::ClockResetController::Initialize(GetClockResetModule());

        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            nn::os::InterruptEventType* pInterruptEvent = GetInterruptEvent();
            // レベル割り込みのため、ManualClear を指定する
            nn::os::InitializeInterruptEvent(pInterruptEvent, GetInterruptNumber(), nn::os::EventClearMode_ManualClear);
            SdHostStandardController::PreSetInterruptEvent(pInterruptEvent);
        #endif

        SdHostStandardController::Initialize();
    }

    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        SdHostStandardController::Finalize();

        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            nn::os::FinalizeInterruptEvent(GetInterruptEvent());
        #endif

        detail::ClockResetController::Finalize(GetClockResetModule());
    }

    virtual Result Startup(BusPower busPower, BusWidth busWidth, SpeedMode speedMode, bool isPowerSavingEnable) NN_NOEXCEPT NN_OVERRIDE;
    virtual void Shutdown() NN_NOEXCEPT NN_OVERRIDE;

    virtual void PutToSleep() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT("PutToSleep isn't implemented.\n");
    }

    virtual Result Awaken() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT("Awaken isn't implemented.\n");
        return ResultNotImplemented();
    }

    virtual Result SetSpeedMode(SpeedMode speedMode) NN_NOEXCEPT NN_OVERRIDE;

    virtual SpeedMode GetSpeedMode() const NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT("GetSpeedMode isn't implemented.\n");
        return SpeedMode_MmcIdentification; // 適当な値
    }

    virtual Result GetInternalStatus() const NN_NOEXCEPT NN_OVERRIDE
    {
        // 特に保持している内部状態はないため、問題ない旨を返す
        NN_RESULT_SUCCESS;
    }
};

// SDMMC1, SDMMC2 は使用予定がないため、未実装

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    const nn::dd::PhysicalAddress Sdmmc3RegistersPhysicalAddress = 0x700B0400ull;

    class Sdmmc3Controller : public SdmmcController
    {
    private:
        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            static nn::os::InterruptEventType m_InterruptEvent;
        #endif

    protected:
        virtual detail::ClockResetController::Module GetClockResetModule() const NN_NOEXCEPT NN_OVERRIDE
        {
            return detail::ClockResetController::Module_Sdmmc3;
        }

        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            virtual int GetInterruptNumber() const NN_NOEXCEPT NN_OVERRIDE
            {
                return PeripheralInterruptOffset + 19;
            }
        #endif

        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            virtual nn::os::InterruptEventType* GetInterruptEvent() const NN_NOEXCEPT NN_OVERRIDE
            {
                return &m_InterruptEvent;
            }
        #endif

    public:
        Sdmmc3Controller() NN_NOEXCEPT
            : SdmmcController(Sdmmc3RegistersPhysicalAddress)
        {
        }

        virtual bool IsSupportedBusPower(BusPower busPower) const NN_NOEXCEPT NN_OVERRIDE
        {
            switch (busPower)
            {
            case BusPower_Off:
                return true;
            case BusPower_1_8V:
                return false;   // TODO: 未実装（HW は対応している）
            case BusPower_3_3V:
                return true;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }

        virtual bool IsSupportedBusWidth(BusWidth busWidth) const NN_NOEXCEPT NN_OVERRIDE
        {
            switch (busWidth)
            {
            case BusWidth_1Bit:
                return true;
            case BusWidth_4Bit:
                return true;
            case BusWidth_8Bit:
                return false;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    };
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_MMC_0_ENABLE))
    const nn::dd::PhysicalAddress Sdmmc4RegistersPhysicalAddress = 0x700B0600ull;

    class Sdmmc4Controller : public SdmmcController
    {
    private:
        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            static nn::os::InterruptEventType m_InterruptEvent;
        #endif

    protected:
        virtual detail::ClockResetController::Module GetClockResetModule() const NN_NOEXCEPT NN_OVERRIDE
        {
            return detail::ClockResetController::Module_Sdmmc4;
        }

        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            virtual int GetInterruptNumber() const NN_NOEXCEPT NN_OVERRIDE
            {
                return PeripheralInterruptOffset + 31;
            }
        #endif

        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            virtual nn::os::InterruptEventType* GetInterruptEvent() const NN_NOEXCEPT NN_OVERRIDE
            {
                return &m_InterruptEvent;
            }
        #endif

    public:
        Sdmmc4Controller() NN_NOEXCEPT
            : SdmmcController(Sdmmc4RegistersPhysicalAddress)
        {
        }

        virtual bool IsSupportedBusPower(BusPower busPower) const NN_NOEXCEPT NN_OVERRIDE
        {
            switch (busPower)
            {
            case BusPower_Off:
                return true;
            case BusPower_1_8V:
                return true;
            case BusPower_3_3V:
                return false;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }

        virtual bool IsSupportedBusWidth(BusWidth busWidth) const NN_NOEXCEPT NN_OVERRIDE
        {
            switch (busWidth)
            {
            case BusWidth_1Bit:
                return true;
            case BusWidth_4Bit:
                return true;
            case BusWidth_8Bit:
                return true;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    };
#endif

} // namespace detail {
}} // namespace nn { namespace sdmmc {
